//-< TTree.java >----------------------------------------------------*--------*
// JSQL                       Version 1.04       (c) 1999  GARRET    *     ?  *
// (Java SQL)                                                        *   /\|  *
//                                                                   *  /  \  *
//                          Created:      2-Dec-2002  K.A. Knizhnik  * / [] \ *
//                          Last update:  2-Dec-2002  K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// T-Tree container
//-------------------------------------------------------------------*--------*

package org.garret.jsql;

import java.lang.reflect.Field;

class PageReference { 
    TTreePage pg;
    
    PageReference(TTreePage p) { pg = p; }
}


class TTreePage implements java.io.Serializable { 
    TTreePage left;
    TTreePage right;
    int       balance;
    int       nItems;
    Object    item[];
    
    static final int pageSize = 125;
    static final int minItems = pageSize - 2; // minimal number of items in internal node

    TTreePage(Object rec) 
    { 
        nItems = 1;
        item = new Object[pageSize];
        item[0] = rec;
    }

    boolean find(Field field, Comparable minValue, Comparable maxValue, int inclusive, Query q) 
      throws IllegalAccessException 
    {
        int l, r, m, n = nItems;

        if (minValue != null) { 
            if (minValue.compareTo(field.get(item[0])) >= inclusive) {	    
                if (minValue.compareTo(field.get(item[n-1])) >= inclusive) {
                    if (right != null) { 
                        return right.find(field, minValue, maxValue, inclusive, q); 
                    } 
                    return true;
                }
                for (l = 0, r = n; l < r;) { 
                    m = (l + r) >> 1;
                    if (minValue.compareTo(field.get(item[m])) >= inclusive) {
                        l = m+1;
                    } else { 
                        r = m;
                    }
                }
                while (r < n) { 
                    if (maxValue != null
                        && -maxValue.compareTo(field.get(item[r])) >= inclusive)
                    { 
                        return false;
                    }
                    if (!q.add(item[r++])) { 
                        return false;
                    }
                }
                if (right != null) { 
                    return right.find(field, minValue, maxValue, inclusive, q); 
                } 
                return true;	
            }
        }	
        if (left != null) { 
            if (!left.find(field, minValue, maxValue, inclusive, q)) { 
                return false;
            }
        }
        for (l = 0; l < n; l++) { 
            if (maxValue != null && -maxValue.compareTo(field.get(item[l])) >= inclusive) 
            {
                return false;
            }
            if (!q.add(item[l])) { 
                return false;
            }
        }
        if (right != null) { 
            return right.find(field, minValue, maxValue, inclusive, q);
        }         
        return true;
    }
    
    boolean insert(Field field, Object obj, Comparable key, PageReference ref) 
      throws IllegalAccessException 
    { 
        int n = nItems;
        int diff = key.compareTo(field.get(item[0]));
        TTreePage pg;
        if (diff <= 0) { 
            if ((left == null || diff == 0) && nItems != pageSize) { 
                for (int i = n; i > 0; i--) item[i] = item[i-1];
                item[0] = obj;
                nItems += 1;
                return false;
            } 
            if (left == null) { 
                left = new TTreePage(obj);
            } else {
                pg = ref.pg;
                ref.pg = left;
                boolean grow = left.insert(field, obj, key, ref);
                left = ref.pg;
                ref.pg = pg;
                if (!grow) return false;
            }
            if (balance > 0) { 
                balance = 0;
                return false;
            } else if (balance == 0) { 
                balance = -1;
                return true;
            } else { 
                TTreePage left = this.left;
                if (left.balance < 0) { // single LL turn
                    this.left = left.right;
                    left.right = this;
                    balance = 0;
                    left.balance = 0;
                    ref.pg = left;
                } else { // double LR turn
                    TTreePage right = left.right;
                    left.right = right.left;
                    right.left = left;
                    this.left = right.right;
                    right.right = this;
                    balance = (right.balance < 0) ? 1 : 0;
                    left.balance = (right.balance > 0) ? -1 : 0;
                    right.balance = 0;
                    ref.pg = right;
                }
                return false;
            }
        } 
        diff = key.compareTo(field.get(item[n-1]));
        if (diff >= 0) { 
            if ((right == null || diff == 0) && nItems != pageSize) { 
                item[n] = obj;
                nItems += 1;
                return false;
            }
            if (right == null) { 
                right = new TTreePage(obj);
            } else { 
                pg = ref.pg;
                ref.pg = right;
                boolean grow = right.insert(field, obj, key, ref);
                right = ref.pg;
                ref.pg = pg;
                if (!grow) return false;
            }
            if (balance < 0) { 
                balance = 0;
                return false;
            } else if (balance == 0) { 
                balance = 1;
                return true;
            } else { 
                TTreePage right = this.right;
                if (right.balance > 0) { // single RR turn
                    this.right = right.left;
                    right.left = this;
                    balance = 0;
                    right.balance = 0;
                    ref.pg = right;
                } else { // double RL turn
                    TTreePage left = right.left;
                    right.left = left.right;
                    left.right = right;
                    this.right = left.left;
                    left.left = this;
                    balance = (left.balance > 0) ? -1 : 0;
                    right.balance = (left.balance < 0) ? 1 : 0;
                    left.balance = 0;
                    ref.pg = left;
                }
                return false;
            }
        }
        int l = 1, r = n-1;
        while (l < r)  {
            int i = (l+r) >> 1;
            diff = key.compareTo(field.get(item[i]));
            if (diff > 0) { 
                l = i + 1;
            } else { 
                r = i;
                if (diff == 0) { 
                    break;
                }
            }
        }
        // Insert before item[r]
        if (n != pageSize) {
            for (int i = n; i > r; i--) item[i] = item[i-1]; 
            item[r] = obj;
            nItems += 1;
            return false;
        } else { 
            Object reinsertItem;
            if (balance >= 0) { 
                reinsertItem = item[0];
                for (int i = 1; i < r; i++) item[i-1] = item[i]; 
                item[r-1] = obj;
            } else { 
                reinsertItem = item[n-1];
                for (int i = n-1; i > r; i--) item[i] = item[i-1]; 
                item[r] = obj;
            }
            return insert(field, reinsertItem, (Comparable)field.get(reinsertItem), ref); 
        }
    }
       
    int balanceLeftBranch(PageReference ref) throws IllegalAccessException 
    {
        if (balance < 0) { 
            balance = 0;
            return 1;
        } else if (balance == 0) { 
            balance = 1;
            return 0;
        } else { 
            TTreePage right = this.right;
            if (right.balance >= 0) { // single RR turn
                this.right = right.left;
                right.left = this;
                if (right.balance == 0) { 
                    this.balance = 1;
                    right.balance = -1;
                    ref.pg = right;
                    return 0;
                } else { 
                    balance = 0;
                    right.balance = 0;
                    ref.pg = right;
                    return 1;
                }
            } else { // double RL turn
                TTreePage left= right.left;
                right.left = left.right;
                left.right = right;
                this.right = left.left;
                left.left = this;
                balance = left.balance > 0 ? -1 : 0;
                right.balance = left.balance < 0 ? 1 : 0;
                left.balance = 0;
                ref.pg = left;
                return 1;
            }
        }
    }

    int balanceRightBranch(PageReference ref) throws IllegalAccessException 
    {
        if (balance > 0) { 
            balance = 0;
            return 1;
        } else if (balance == 0) { 
            balance = -1;
            return 0;
        } else { 
            TTreePage left = this.left;
            if (left.balance <= 0) { // single LL turn
                this.left = left.right;
                left.right = this;
                if (left.balance == 0) { 
                    balance = -1;
                    left.balance = 1;
                    ref.pg = left;
                    return 0;
                } else { 
                    balance = 0;
                    left.balance = 0;
                    ref.pg = left;
                    return 1;
                }
            } else { // double LR turn
                TTreePage right = left.right;
                left.right = right.left;
                right.left = left;
                this.left = right.right;
                right.right = this;
                balance = right.balance < 0 ? 1 : 0;
                left.balance = right.balance > 0 ? -1 : 0;
                right.balance = 0;
                ref.pg = right;
                return 1;
            }
        }
    }
    
    int remove(Field field, Object obj, Comparable key, PageReference ref)
      throws IllegalAccessException 
    {
        TTreePage pg;
        int n = nItems;
        int diff = key.compareTo(field.get(item[0]));
        if (diff <= 0) { 
            if (left != null) { 
                pg = ref.pg;
                ref.pg = left;
                int h = left.remove(field, obj, key, ref);
                left = ref.pg;
                ref.pg = pg;
                if (h > 0) { 
                    return balanceLeftBranch(ref);
                } else if (h == 0) { 
                    return 0;
                }
            }
            //assert (diff == 0);
        }
        diff = key.compareTo(field.get(item[n-1]));
        if (diff <= 0) {	    
            for (int i = 0; i < n; i++) { 
                if (item[i] == obj) { 
                    if (n == 1) { 
                        if (right == null) { 
                            // delete this
                            ref.pg = left;
                            return 1;
                        } else if (left == null) { 
                            // delete this
                            ref.pg = right;
                            return 1;
                        } 
                    } 
                    if (n <= minItems) { 
                        if (left != null && balance <= 0) {  
                            TTreePage prev = left;
                            while (prev.right != null) { 
                                prev = prev.right;
                            }
                            while (--i >= 0) { 
                                item[i+1] = item[i];
                            }
                            item[0] = prev.item[prev.nItems-1];
                            pg = ref.pg;
                            ref.pg = left;
                            int h = left.remove(field, item[0], (Comparable)field.get(item[0]), ref);
                            left = ref.pg;
                            ref.pg = pg;
                            if (h > 0) {
                                h = balanceLeftBranch(ref);
                            }
                            return h;
                        } else if (right != null) { 
                            TTreePage next = right;
                            while (next.left != null) { 
                                next = next.left;
                            }
                            while (++i < n) { 
                                item[i-1] = item[i];
                            }
                            item[n-1] = next.item[0];
                            pg = ref.pg;
                            ref.pg = right;
                            int h = right.remove(field, item[n-1], (Comparable)field.get(item[n-1]), ref);
                            right = ref.pg;
                            ref.pg = pg;
                            if (h > 0) {
                                h = balanceRightBranch(ref);
                            }
                            return h;
                        }
                    }
                    while (++i < n) { 
                        item[i-1] = item[i];
                    }
                    nItems -= 1;
                    return 0;
                }
            }
        }
        if (right != null) { 
            pg = ref.pg;
            ref.pg = right;
            int h = right.remove(field, obj, key, ref);
            right = ref.pg;
            ref.pg = pg;
            if (h > 0) { 
                return balanceRightBranch(ref);
            } else { 
                return h;
            }
        }
        return -1;
    }


    void traverseForward(TTree.Processor proc) { 
        if (left != null) { 
            left.traverseForward(proc);
        }
        for (int i = 0, n = nItems; i < n; i++) { 
            proc.process(item[i]);
        }
        if (right != null) { 
            right.traverseForward(proc);
        }
    }

        
    void traverseBackward(TTree.Processor proc) { 
        if (right != null) { 
            right.traverseForward(proc);
        }
        for (int i = nItems; --i >= 0;) { 
            proc.process(item[i]);
        }
        if (left != null) { 
            left.traverseForward(proc);
        }
    }
        
    TTreePage(int n) {
        nItems = n;
        item = new Object[pageSize];
    }
            
};



/**
 * T-Tree contaniner for objects. 
 * T-Tree is most efficient data structure for exact and range data queries in the assumption
 * that all data is present in memory.
 * This class can be used in implementation of query iterators to provide fast indexed access to the records
 */
public class TTree implements java.io.Serializable { 
    /**
     * Constructor of T-Tree
     * @param f indexed field in the record
     */
    public TTree(Field f) { 
        field = f;
    }

    /**
     * Locate record within sepcified key range
     * @param minValue low bound for key value (if <code>null</code>, then there
     * is no low bound
     * @param maxValue high bound for key value (if <code>null</code>, then there
     * is no high bound
     * @param inclusive whether bounds are inclusive or exclusive
     * @param query query to which records belongin to the specified range should be added using 
     * <code>add</code> method
     */
    public void select(Object minValue, Object maxValue, boolean inclusive, Query query)
    {
        if (root != null) { 
            try { 
                root.find(field, (Comparable)minValue, (Comparable)maxValue, inclusive ? 1 : 0, query);
            } catch (IllegalAccessException x) { 
                throw new IllegalAccessError();
            }
        }
    }

    /**
     * Add new object to T-Tree
     * @param obj object to be inserted in T-Tree
     */
    public void add(Object obj) { 
        if (root == null) { 
            root = new TTreePage(obj);
        } else {
            try { 
                PageReference ref = new PageReference(root);
                root.insert(field, obj, (Comparable)field.get(obj), ref);
                root = ref.pg;
            } catch (IllegalAccessException x) { 
                x.printStackTrace();
                throw new IllegalAccessError();
            }
        }
        size += 1;
    }

    /**
     * Remove object from T-Tree
     * @param obj object to be removed from T-Tree
     */
    public void remove(Object obj) { 
        try { 
            PageReference ref = new PageReference(root);
            root.remove(field, obj, (Comparable)field.get(obj), ref);
            root = ref.pg;
            size -= 1;
        } catch (IllegalAccessException x) { 
            throw new IllegalAccessError();
        }
    }

    /**
     * Remove all elements from T-Tree
     */
    public void clear() { 
        root = null;
        size = 0;
    }

    /**
     * Interface for class processing elements of T-Tree during tree traversal
     */
    public static interface Processor { 
        public void process(Object o);
    }
    
    /**
     * Traverse T-Tree elements in ascent order
     */
    public void traverseForward(Processor proc) { 
        if (root != null) {
            root.traverseForward(proc);
        }
    }

    /**
     * Traverse T-Tree elements in descent order
     */
    public void traverseBackward(Processor proc) {
        if (root != null) {
            root.traverseBackward(proc);
        }
    }

    static class TreeSerializer implements Processor { 
        java.io.ObjectOutputStream out;

        TreeSerializer(java.io.ObjectOutputStream  s) { 
            out = s;
        }

        public void process(Object o) {
            try { 
                out.writeObject(o);
            } catch (java.io.IOException x) { 
                throw new Error("Failed to serialize T-Tree: " + x);
            }
        }
    }

    static class TreeConstructor { 
        int                       height;
        java.io.ObjectInputStream in;

        TreeConstructor(java.io.ObjectInputStream s) { in = s; }

        void readPage(TTreePage pg) throws java.io.IOException, ClassNotFoundException 
        { 
            for (int i = 0, n = pg.nItems; i < n; i++) { 
                pg.item[i] = in.readObject();
            }
        }

        TTreePage reconstructTree(int n) throws java.io.IOException, ClassNotFoundException 
        {
            TTreePage root;

            if (n <= TTreePage.pageSize) { 
                height += 1;
                root = new TTreePage(n);
                readPage(root);
            } else if (n <= TTreePage.pageSize*2) { 
                height += 2;
                root = new TTreePage(TTreePage.pageSize);
                root.left = new TTreePage(n - TTreePage.pageSize);
                readPage(root.left);
                readPage(root);
                root.balance = -1;
            } else { 
                root = new TTreePage(TTreePage.pageSize);
                int less = (n - TTreePage.pageSize) >>> 1;
                int greater = n - TTreePage.pageSize - less;;
                int rootHeight = ++height;
                root.left = reconstructTree(less);
                int leftHeight = height;
                readPage(root);
                height = rootHeight;
                root.right = reconstructTree(greater);
                int rightHeight = height;
                if (leftHeight > rightHeight) { 
                    root.balance = -1;
                    height = leftHeight;
                } else if (leftHeight < rightHeight) { 
                    root.balance = 1;
                    height = rightHeight;
                } else { 
                    height = leftHeight;
                }
            }
            return root;
        }
    }
    
        
    /**
     * Reconstitute the <tt>T-Tree</tt> instance from a stream (i.e.,
     * deserialize it).
     */
    private void readObject(final java.io.ObjectInputStream s)
        throws java.io.IOException, ClassNotFoundException 
    {
        size = s.readInt();
        Class cls = (Class)s.readObject();
        String fieldName = (String)s.readObject();
        try { 
            field = cls.getDeclaredField(fieldName);
        } catch (NoSuchFieldException x) { 
            throw new NoSuchFieldError(fieldName);
        }
        field.setAccessible(true);
        if (size > 0) { 
            root = new TreeConstructor(s).reconstructTree(size);
        }
    }


    /**
     * Save the state of the <tt>T-Tree</tt> instance to a stream (i.e.,
     * serialize it).
     */
    private void writeObject(java.io.ObjectOutputStream s)
        throws java.io.IOException 
    {
        s.writeInt(size);
        s.writeObject(field.getDeclaringClass());
        s.writeObject(field.getName());
        traverseForward(new TreeSerializer(s));
    }

    TTreePage root;
    int       size;
    Field     field;
}
        



