//-< ThreadPool.java >-----------------------------------------------*--------*
// JSQL                       Version 1.04       (c) 1999  GARRET    *     ?  *
// (Java SQL)                                                        *   /\|  *
//                                                                   *  /  \  *
//                          Created:      4-Dec-2002  K.A. Knizhnik  * / [] \ *
//                          Last update:  4-Dec-2002  K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Pool of threads used to minimize overhead of creation new threads each time
// it is needed
//-------------------------------------------------------------------*--------*

package org.garret.jsql;

/**
 * Pool of threads used to minimize overhead of creation new threads each time
 * it is needed
 */
public class ThreadPool { 
    /**
     * Get instance of thread pool
     */
    public static ThreadPool getInstance() { 
        return theInstance;
    }

    static class PooledThread extends Thread { 
        PooledThread next;
        Object       ready;
        Object       done;
        boolean      busy;
        boolean      doneNotificationNeeded;
        Runnable     task;
        ThreadPool   pool;


        public void run() { 
            try { 
                synchronized(ready) { 
                    while (!pool.closed) { 
                        synchronized(done) { 
                            busy = false;
                            if (doneNotificationNeeded) { 
                                done.notify();
                            }
                        }
                        ready.wait();
                        if (task != null) { 
                            task.run();
                        } else { 
                            break;
                        }
                    }
                }
            } catch (InterruptedException x) {}
        }
              
        final void wakeUp(Runnable t) { 
            synchronized(ready) { 
                busy = true;
                doneNotificationNeeded = false;
                task = t;
                ready.notify();
            }
        }

        final void waitCompletion() throws InterruptedException {
            synchronized(done) { 
                if (busy) { 
                    doneNotificationNeeded = true;
                    done.wait();
                } 
            }
        }

        PooledThread(ThreadPool pool) { 
            this.pool = pool;
            ready = new Object();
            done = new Object();
            busy = true;
            setDaemon(true);
            this.start();
        }
    }

    /**
     * Get thread from thread pool and start specified task in this thread
     * @param task class impkemeting Runnable interface which run method will be executed by pooled thread
     * @return pooled thread (it should be used only in <code>ThreadPool.join</code> method)
     */
    public Thread start(Runnable task) 
    { 
        PooledThread thread;
        synchronized (this) { 
            while (availableThreadList == null) { 
                try { 
                    if (nActiveThreads == maxThreads) { 
                        deficit += 1;
                        wait();
                    } else {
                        availableThreadList = new PooledThread(this);
                        availableThreadList.waitCompletion();
                    }
                } catch (InterruptedException x) { 
                    return null;
                }
            }
            thread = availableThreadList;
            availableThreadList = thread.next;
            nActiveThreads += 1;
        }
        thread.wakeUp(task);
        return thread;
    }

    /**
     * Wait completion of task
     * @param thread pooled thread assigned to the task by <code>ThreadPool.start</code> method
     */
    public void join(Thread thread) throws InterruptedException 
    {
        PooledThread t = (PooledThread)thread;
        t.waitCompletion();
        synchronized (this) {
            t.next = availableThreadList;
            availableThreadList = t;
            nActiveThreads -= 1;
            if (deficit > 0) { 
                notify();
                deficit -= 1;
            }
        }
    }

    /**
     * Wait termination of all active tasks and stop all threads in thread pool
     */
    public synchronized void close()
    {
        closed = true;
        while (nActiveThreads > 0) { 
            try { 
                deficit += 1;
                wait();
            } catch (InterruptedException x) {}
        }
        while (availableThreadList != null) { 
            availableThreadList.wakeUp(null);
            availableThreadList = availableThreadList.next;
        }        
    }

    /**
     * Constructor of thread pool with limited number of threads
     * @param maximal number of threads
     */
    public ThreadPool(int maxThreads) { 
        this.maxThreads = maxThreads;
    }

    /**
     * Constructor of thread pool with unlimited number of threads
     */
    public ThreadPool() {
        this(Integer.MAX_VALUE);
    }


    PooledThread availableThreadList;
    int          nActiveThreads;
    int          deficit; // number of threads which could not be created because
                          // maxThreads limit is reached

    int          maxThreads;
    boolean      closed;

    static ThreadPool theInstance = new ThreadPool();
}














