//-< BLOB.H >-------------------------------------------------------*--------*
// OODBC                     Version 1.0         (c) 1999  GARRET    *     ?  *
// (Object adapter for OODBC)                                        *   /\|  *
//                                                                   *  /  \  *
//                          Created:     09-Sep-99    K.A. Knizhnik  * / [] \ *
//                          Last update: 09-Sep-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Binary large objet interface
//-------------------------------------------------------------------*--------*

#ifndef __OODBC_BLOB_H__
#define __OODBC_BLOB_H__

#include "config.h"

BEGIN_OODBC_NAMESPACE

class dbFieldDescriptor;

//
// Class dbBlob makes it possible to retieve and store long data part by part. 
// There are two possible ways of retriving data:
// 1. By explicit invocations of dbBlob::fetchData method.
// 2. Deriving class from dbBlob and redefinition of virtual
//    startDataProcessing and processData methods.
// 
// Update and insert statements will store Blob object to the database 
// using redefined getData virtual method, which should provide Blob data.
//
class dbBlob { 
    friend class dbAnyCursor;
    friend class dbConnection;
  protected:
    char*        buf;
    size_t       bufSize;
    int          columnNo;
    SQLINTEGER   len;
    SQLINTEGER   totalLength;
    dbAnyCursor* cursor;
    dbBlob*      next;

  public:
    bool   isNull() const { 
	return len == SQL_NULL_DATA; 
    }

    char*  getFetchedDataPtr() { return buf; }

    operator char*() { return getFetchedDataPtr(); }

    size_t getFetchedDataSize() { 
	return size_t(len == SQL_NULL_DATA  ? 0 : len); 
    }

    //
    // Returns pointer to the record returned by the current fetch operation.
    //
    void*  getCurrentRecordPtr();

    //
    // Programmer can redefined this function to obtain length of
    // stored Blob. Some ODBC driver requires length of object to be specified
    // before sending data to the driver. If this function returns false
    // (as default implementation does), then it means that information 
    // about length is not available. In last case update operation will fail
    // with drivers requiring length of stored long value.
    //
    virtual bool   getDataLen(SQLINTEGER& length);

    //
    // getData method is called by update or insert statement to obtain
    // part of Blob object data. It should return actual number of bytes
    // avilable. If there are no more data, 0 should be returned.
    //
    virtual size_t getData(void* buf, size_t buf_size);


    //
    // This method is called by dbAnyCursopr::fetch method.
    // If this method returns true, then OODBC will automaticall extract all 
    // BLOB parts and invoke processData method for each of them.
    // If this method returns false, then it is considered that programmer
    // will retrieve BLOB data himself by cyclic invokations of fetchData 
    // method
    //
    virtual bool   startDataProcessing();

    //
    // Method is called by dbAnyCursor::fetch method for all BLOB
    // parts if startDataProcessing method returns true.
    // Data will be retrieved until reaching one of two conditions:
    // 1. End of BLOB object is reached. In this case endDataProcessing 
    //    virtual method will be invoked. 
    // 2. processData method returns false.
    //
    virtual bool   processData(void* buf, size_t bufSize);

    //
    // Method is invoked after retreiving all BLOB data.
    //
    virtual void   endDataProcessing();

    //
    // Fetch part of BLOB data. Size of retrieved data can be get by
    // getFetchedDataSize method and pointer to the buffer - 
    // by getFetchedDataPtr method.
    // 
    bool           fetchData();

    dbFieldDescriptor* dbDescribeComponents(dbFieldDescriptor*) {
	return NULL; 
    }

    dbBlob(size_t blobPartSize = 4096);

    virtual~dbBlob();
};

class dbLongVar : public dbBlob { 
  protected:
    size_t         bufUsed;

    virtual bool   startDataProcessing();

    virtual bool   processData(void* buf, size_t bufSize);

    virtual void   endDataProcessing();

    virtual bool   getDataLen(SQLINTEGER& length);

    virtual size_t getData(void* buf, size_t bufSize);

  public:
    void           setSize(size_t newSize);
    void           append(size_t tailSize);

    dbLongVar(size_t size = 4096) : dbBlob(size) {}
};

END_OODBC_NAMESPACE

#endif
