//-< CONNECT.CPP >---------------------------------------------------*--------*
// OODBC                     Version 1.0         (c) 1999  GARRET    *     ?  *
// (Object adapter for OODBC)                                        *   /\|  *
//                                                                   *  /  \  *
//                          Created:     10-Jul-99    K.A. Knizhnik  * / [] \ *
//                          Last update: 14-Jul-99    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Encapsulation of ODBC connection
//-------------------------------------------------------------------*--------*

#include "oodbc.h"

USE_OODBC_NAMESPACE

bool dbConnection::remove(dbTableDescriptor& table, void* object)
{
    char stmt[1024];
    int i, pos;
    int result;
    assert(opened);
    HSTMT hDeleteStmt;
    assert(table.keys != NULL);
    dbFieldDescriptor* fd = table.keys;
 
#if (ODBCVER >= 0x0300)
    result = SQLAllocHandle(SQL_HANDLE_STMT, hdbc, &hDeleteStmt); 
#else
    result = SQLAllocStmt(hdbc, &hDeleteStmt); 
#endif    
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLAllocHandle, result, NULL);
	return false;
    }
    
    pos = sprintf(stmt, "delete from %s where ", table.name);
    for (i = 1, fd = table.keys; fd != NULL; i++, fd = fd->nextKey) { 
	if (i != 1) { 
	    pos = sprintf(stmt + pos, " and ");
	}
	pos += sprintf(stmt + pos, "%s=?", fd->longName);
	result = 
	    SQLBindParameter(hDeleteStmt, i, SQL_PARAM_INPUT,
			     dbFieldDescriptor::CtypeMapping[fd->type],
			     dbFieldDescriptor::SQLtypeMapping[fd->type],
			     0, 0, (char*)object + fd->offs, 0, NULL);
	if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	    OODBC_FAILURE(SQLBindParameter, result, hDeleteStmt);
	    SQLFreeStmt(hDeleteStmt, SQL_DROP);
	    return false;
	}
    }
    result = SQLExecDirect(hDeleteStmt, (SQLCHAR*)stmt, SQL_NTS);
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLExecDirect, result, hDeleteStmt);
	SQLFreeStmt(hDeleteStmt, SQL_DROP);
	return false;
    }
    SQLFreeStmt(hDeleteStmt, SQL_DROP);
    return true;
}


bool dbConnection::update(dbTableDescriptor& table, void* object)
{
    int i, pos;
    int result;
    assert(opened);
    HSTMT hUpdateStmt;
    assert(table.keys != NULL);
    dbFieldDescriptor* fd;

#if (ODBCVER >= 0x0300)
    result = SQLAllocHandle(SQL_HANDLE_STMT, hdbc, &hUpdateStmt); 
#else
    result = SQLAllocStmt(hdbc, &hUpdateStmt);
#endif 
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLAllocHandle, result, NULL);
	return false;
    }
    dbLocalBuf stmt(table.totalNamesLength + table.nFields*3 + 256);

    pos = sprintf(stmt, "update %s set ", table.name);
    for (fd = table.fields, i = 1; fd != NULL; fd = fd->nextField, i++) { 
	pos += sprintf(stmt + pos, i == 1 ? "%s=?" : ",%s=?", fd->longName);
	if (fd->type == dbFieldDescriptor::tpBlob) { 
	    dbBlob* blob = (dbBlob*)((char*)object + fd->offs);
	    SQLINTEGER blobLength;
	    if (blob->getDataLen(blobLength)) {
		blob->len = SQL_LEN_DATA_AT_EXEC(blobLength);
	    } else { 
		blob->len = SQL_DATA_AT_EXEC;
	    }
	    result = SQLBindParameter(hUpdateStmt, i, SQL_PARAM_INPUT,
				      SQL_C_BINARY, SQL_LONGVARBINARY,
				      0, 0, (char*)blob, 0, 
				      &blob->len);
	} else { 
	    result = 
		SQLBindParameter(hUpdateStmt, i, SQL_PARAM_INPUT,
				 dbFieldDescriptor::CtypeMapping[fd->type],
				 dbFieldDescriptor::SQLtypeMapping[fd->type],
				 0, 0, (char*)object + fd->offs, 0, NULL);
	}
	if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	    OODBC_FAILURE(SQLBindParameter, result, hUpdateStmt);
	    SQLFreeStmt(hUpdateStmt, SQL_DROP);
	    return false;
	}
    }
    pos += sprintf(stmt + pos, " where ");
    bool first = true;
    for (fd = table.keys; fd != NULL; fd = fd->nextKey) { 
	if (!first) { 
	    pos = sprintf(stmt + pos, " and ");
	}
	first = false;
	pos += sprintf(stmt + pos, "%s=?", fd->longName);
	result = SQLBindParameter(hUpdateStmt, i++, SQL_PARAM_INPUT,
				  dbFieldDescriptor::CtypeMapping[fd->type],
				  dbFieldDescriptor::SQLtypeMapping[fd->type],
				  0, 0, (char*)object + fd->offs, 0, NULL);
	if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	    OODBC_FAILURE(SQLBindParameter, result, hUpdateStmt);
	    SQLFreeStmt(hUpdateStmt, SQL_DROP);
	    return false;
	}
    }
    result = SQLExecDirect(hUpdateStmt, stmt, SQL_NTS);
    while (result == SQL_NEED_DATA) {
	SQLPOINTER pToken;
	result = SQLParamData(hUpdateStmt, &pToken);
	if (result == SQL_NEED_DATA) {
	    dbBlob* blob = (dbBlob*)pToken;
	    size_t len;
	    while ((len = blob->getData(blob->buf, blob->bufSize)) != 0) {
		result = SQLPutData(hUpdateStmt, blob->buf, len);
		if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) 
		{
		    OODBC_FAILURE(SQLPutData, result, hUpdateStmt);
		    SQLFreeStmt(hUpdateStmt, SQL_DROP);
		    return false;
		}
	    }
	    result = SQL_NEED_DATA;
	}
    }
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLExecDirect, result, hUpdateStmt);
	SQLFreeStmt(hUpdateStmt, SQL_DROP);
	return false;
    }
    SQLFreeStmt(hUpdateStmt, SQL_DROP);
    return true;
}


bool dbConnection::insert(dbTableDescriptor& table, void* object)
{
    int i, pos;
    int result;
    assert(opened);
    HSTMT hInsertStmt;
#if (ODBCVER >= 0x0300)
    result = SQLAllocHandle(SQL_HANDLE_STMT, hdbc, &hInsertStmt); 
#else
    result = SQLAllocStmt(hdbc, &hInsertStmt); 
#endif
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLAllocHandle, result, NULL);
	return false;
    }
    dbLocalBuf stmt(table.totalNamesLength + table.nFields*2 + 256);

    dbFieldDescriptor* fd = table.fields;
    pos = sprintf(stmt, "insert into %s (%s", table.name, fd->longName);
    while ((fd = fd->nextField) != NULL) { 
	pos += sprintf(stmt + pos, ",%s", fd->longName);
    }
    pos += sprintf(stmt + pos, ") values (");
    for (i = 1, fd = table.fields; fd != NULL; fd = fd->nextField, i++) { 
	pos += sprintf(stmt + pos, i != 1 ? ",?" : "?");
	if (fd->type == dbFieldDescriptor::tpBlob) { 
	    dbBlob* blob = (dbBlob*)((char*)object + fd->offs);
	    SQLINTEGER blobLength;
	    if (blob->getDataLen(blobLength)) {
		blob->len = SQL_LEN_DATA_AT_EXEC(blobLength);
	    } else { 
		blob->len = SQL_DATA_AT_EXEC;
	    }
	    result = SQLBindParameter(hInsertStmt, i, SQL_PARAM_INPUT,
				      SQL_C_BINARY, SQL_LONGVARBINARY,
				      0, 0, (char*)blob, 0, &blob->len);
	} else { 
	    result = 
		SQLBindParameter(hInsertStmt, i, SQL_PARAM_INPUT,
				 dbFieldDescriptor::CtypeMapping[fd->type],
				 dbFieldDescriptor::SQLtypeMapping[fd->type],
				 0, 0, (char*)object + fd->offs, 0, NULL);
	}
	if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	    OODBC_FAILURE(SQLBindParameter, result, hInsertStmt);
	    SQLFreeStmt(hInsertStmt, SQL_DROP);
	    return false;
	}
    }
    pos += sprintf(stmt + pos, ")");
    result = SQLExecDirect(hInsertStmt, stmt, SQL_NTS);
    while (result == SQL_NEED_DATA) {
	SQLPOINTER pToken;
	result = SQLParamData(hInsertStmt, &pToken);
	if (result == SQL_NEED_DATA) {
	    dbBlob* blob = (dbBlob*)pToken;
	    size_t len;
	    while ((len = blob->getData(blob->buf, blob->bufSize)) != 0)
	    {
		result = SQLPutData(hInsertStmt, blob->buf, len);
		if (result != SQL_SUCCESS && 
		    result != SQL_SUCCESS_WITH_INFO) 
		{
		    OODBC_FAILURE(SQLPutData, result, hInsertStmt);
		    SQLFreeStmt(hInsertStmt, SQL_DROP);
		    return false;
		}
	    }
	    result = SQL_NEED_DATA;
	}
    }
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLExecDirect, result, hInsertStmt);
	SQLFreeStmt(hInsertStmt, SQL_DROP);
	return false;
    }
    SQLFreeStmt(hInsertStmt, SQL_DROP);
    return true;
}

bool dbConnection::execute(char const* sql)
{
    int result;
    assert(opened);
    HSTMT hstmt;
#if (ODBCVER >= 0x0300)
    result = SQLAllocHandle(SQL_HANDLE_STMT, hdbc, &hstmt); 
#else
    result = SQLAllocStmt(hdbc, &hstmt); 
#endif
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLAllocHandle, result, NULL);
	return false;
    }
    result = SQLExecDirect(hstmt, (SQLCHAR*)sql, SQL_NTS);
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLExecDirect, result, hstmt);
	SQLFreeStmt(hstmt, SQL_DROP);
	return false;
    }
    SQLFreeStmt(hstmt, SQL_DROP);
    return true;
}


bool dbConnection::execute(dbQuery& query)
{
    int result;
    assert(opened);
    HSTMT hstmt;
#if (ODBCVER >= 0x0300)
    result = SQLAllocHandle(SQL_HANDLE_STMT, hdbc, &hstmt); 
#else
    result = SQLAllocStmt(hdbc, &hstmt);
#endif
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLAllochandle, result, NULL);
	return false;
    }
    dbLocalBuf stmt(query.stmtLen + 1);
    dbQueryElement* elem = query.elements;
    int i = 1;
    int pos = 0;
    while (elem != NULL) { 
	if (elem->type == dbFieldDescriptor::tpQueryText) { 
	    pos += sprintf(stmt + pos, (char*)elem->ptr);
	} else { 
	    pos += sprintf(stmt + pos, "?");
	    result = 
	      SQLBindParameter(hstmt, i++, SQL_PARAM_INPUT,
			       dbFieldDescriptor::CtypeMapping[elem->type],
			       dbFieldDescriptor::SQLtypeMapping[elem->type],
			       0, 0, elem->ptr, 0, NULL);
	    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
		OODBC_FAILURE(SQLBindParameter, result, hstmt);
		SQLFreeStmt(hstmt, SQL_DROP);
		return false;
	    }
	}
	elem = elem->next;
    }
    result = SQLExecDirect(hstmt, stmt, SQL_NTS);
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLExecDirect, result, hstmt);
	SQLFreeStmt(hstmt, SQL_DROP);
	return false;
    }
    SQLFreeStmt(hstmt, SQL_DROP);
    return true;
}


bool dbConnection::open(char const* dataSource, char const* userName, 
			char const* password)
{
    int result;
#if (ODBCVER >= 0x0300)
    result = SQLAllocHandle(SQL_HANDLE_ENV, SQL_NULL_HANDLE, &henv);  
#else
    result = SQLAllocEnv(&henv);  
#endif
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLAllocHandle, result, NULL);
	return false;
    }
    /* Set the ODBC version environment attribute */
#ifdef SQL_ATTR_ODBC_VERSION
    result = SQLSetEnvAttr(henv, SQL_ATTR_ODBC_VERSION, (void*)SQL_OV_ODBC2,0);
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLSetEnvAttr, result, NULL);
	return false;
    }
#endif
#if (ODBCVER >= 0x0300)
    result = SQLAllocHandle(SQL_HANDLE_DBC, henv, &hdbc); 
#else
    result = SQLAllocConnect(henv, &hdbc); 
#endif
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLAllocHandle, result, NULL);
	return false;
    }
    /* Set login timeout to 5 seconds. */
#if (ODBCVER >= 0x0300)
    SQLSetConnectAttr(hdbc, SQL_LOGIN_TIMEOUT, (void*)5, 0);
#else
    SQLSetConnectOption(hdbc, SQL_LOGIN_TIMEOUT, 5);
#endif
    /* Connect to data source */
    result = SQLConnect(hdbc, (SQLCHAR*)dataSource, SQL_NTS,
			(SQLCHAR*)userName, SQL_NTS,
			(SQLCHAR*)password, SQL_NTS);
    
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) {
	OODBC_FAILURE(SQLConnect, result, NULL);
	return false;
    }
    opened = true;
    return true;
}


bool dbConnection::close()
{
    int result = SQL_SUCCESS;
    if (opened) { 
	result = SQLDisconnect(hdbc);
	if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) { 
	    OODBC_FAILURE(SQLDisconnect, result, NULL);
	}
	opened = false;
    }
    if (hdbc) { 
#if (ODBCVER >= 0x0300)
	SQLFreeHandle(SQL_HANDLE_DBC, hdbc);
#else
	SQLFreeConnect(hdbc);
#endif
	hdbc = 0;
    }
    if (henv) { 
#if (ODBCVER >= 0x0300)
	SQLFreeHandle(SQL_HANDLE_ENV, henv);
#else
	SQLFreeEnv(henv);
#endif
	henv = 0;
    }
    return result == SQL_SUCCESS || result == SQL_SUCCESS_WITH_INFO;
}


bool dbConnection::commit()
{
    int result;
#if (ODBCVER >= 0x0300)
    result = SQLEndTran(SQL_HANDLE_DBC, hdbc, SQL_COMMIT);
#else
    result = SQLTransact(henv, hdbc, SQL_COMMIT);
#endif
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) { 
	OODBC_FAILURE(SQLEndTran, result, NULL);
	return false;
    }
    return true;
}


bool dbConnection::rollback()
{
    int result;
#if (ODBCVER >= 0x0300)
    result = SQLEndTran(SQL_HANDLE_DBC, hdbc, SQL_ROLLBACK);
#else
    result = SQLTransact(henv, hdbc, SQL_ROLLBACK);
#endif
    if (result != SQL_SUCCESS && result != SQL_SUCCESS_WITH_INFO) { 
	OODBC_FAILURE(SQLEndTrans, result, NULL);
	return false;
    }
    return true;
}

void dbConnection::onError(int result, char const* function, char const* file,
			   int line, HSTMT hstmt)
{
    SDWORD nativeErrorCode;
    SWORD len;
    char buf[SQL_MAX_MESSAGE_LENGTH - 1];
    char sqlStatus[16];
    char location[256];
    int rc = SQLError(henv, hdbc, hstmt, (SQLCHAR*)sqlStatus, &nativeErrorCode,
		      (SQLCHAR*)buf, sizeof(buf), &len);
    sprintf(location, "Error in function %s at %s:%d", function, file, line);
    if (rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO) { 
	handleError(result, location, "Unknown error");
    } else { 
	handleError(result, location, buf);
    }
}

void dbConnection::handleError(int result, char const* place, char const* msg)
{
    fprintf(stderr, "%s error code %d: %s\n", place, result, msg);
}



