"""
Tools for working with AQuA event features

Designed to accompany the paper "Network-level encoding of local
neurotransmitters in cortical astrocytes" (DOI: TODO)
"""


## ------------------------------------------------------------------------ ##
## Imports
## ------------------------------------------------------------------------ ##

## ** Typing

from typing import (
    Optional,
    Union,
)
from numpy import (
    ndarray,
)
from numpy.typing import (
    ArrayLike,
)
from pandas import (
    DataFrame,
)

# Type shortcuts
Window = Union[
    ArrayLike,
    list,
    'tuple[float, float]'
]

DecoratorSpec = Union[
    str,
    tuple
]

import numpy as np
import pandas as pd

from tqdm import tqdm

## ** Local imports

from . import utils


## ------------------------------------------------------------------------ ##
## Helper functions
## ------------------------------------------------------------------------ ##

## Resampling

def _perm_events_shift_group(
        df: DataFrame,
        window: Window,
        t_key: str = 'start_time_rel',
        level: str = 'cell_global_all'
    ) -> DataFrame:
    """
    Produce a surrogate dataset by shifting events, grouped at `level`

    Parameters
    ----------
    df : pandas.DataFrame
        the original, unperturbed events
    window : Union[ArrayLike, list, tuple[float, float]]
        the time window in which to shift events
    t_key : str, optional
        the column in `df` to use as time (default: "start_time_rel")
    level : str, optional
        the grouping at which to hold events together (default:
        "cell_global_all")
    
    Returns
    -------
    DataFrame
        the perturbed events (as a copy)
    """
    
    ret = df.copy()
    window_size = window[1] - window[0]
    
    for group in ret[level].unique():
        ts_all = np.array( ret[t_key] )
        
        filter_group = ret[level] == group
        filter_window = is_within( ts_all, window )
        filter_shift_cur = filter_group & filter_window
        
        ts_cur = ts_all[filter_shift_cur]
        ts_rel = ts_cur - window[0]
        
        t_shift = np.random.uniform( window_size )
        ts_rel_shift = ts_rel + t_shift
        ts_rel_shift[ts_rel_shift >= window_size] -= window_size
        
        ts_all[filter_shift_cur] = ts_rel_shift + window[0]
        ret[t_key] = ts_all
    
    return ret


## Reporting

def summarize_n(
        events: DataFrame,
        grid_number: str = None,
    ):
    """
    Print a summary of the n hierarchy for the given dataset of `events`

    Parameters
    ----------
    events : pandas.DataFrame
        events dataset, as generated by a `HiveManager` in `aqua`
    grid_number : str
        TODO Should this be changed over to the number itself, leaving the
        string formatting to be done internally here?
    """
    
    print( f"Mice:            n = {len( events['mouse'].unique() )}" )
    if 'slice_coreg' in events.keys():
        print( f"Slices:          n = {len( events['slice_coreg'].unique() )}" )
    print( f"Datasets:        n = {len( events['dataset_id_all'].unique() )}" )
    if 'cell_global_all' in events.keys():
        print( f"Cell-recordings: n = {len( events['cell_global_all'].unique() )}" )
    if 'cell_global_coreg' in events.keys():
        print( f"Cells:           n = {len( events['cell_global_coreg'].unique() )}" )
    if grid_number is not None:
        print( f"ROI-recordings:  n = {len( events[f'grid_global_{grid_number}'].unique() )}" )
        print( f"ROIs:            n = {len( events[f'grid_global_{grid_number}_coreg'].unique() )}" )
    print( f"Events:          n = {events.shape[0]}" )

    if 'slice_coreg' in events.keys():
        n_per_mouse = { mouse: len( df_mouse['slice_coreg'].unique() )
                        for mouse, df_mouse in events.groupby( 'mouse' ) }
        print( "Slices per mouse:" )
        print( f"    Low:  {min( n_per_mouse.values() )}" )
        print( f"    High: {max( n_per_mouse.values() )}" )

    n_per_mouse = { mouse: len( df_mouse['dataset_id_all'].unique() )
                    for mouse, df_mouse in events.groupby( 'mouse' ) }
    print( "Datasets per mouse:" )
    print( f"    Low:  {min( n_per_mouse.values() )}" )
    print( f"    High: {max( n_per_mouse.values() )}" )

    if 'slice_coreg' in events.keys():
        n_per_slice = { mouse: len( df_mouse['dataset_id_all'].unique() )
                        for mouse, df_mouse in events.groupby( 'slice_coreg' ) }
        print( "Datasets per slice:" )
        print( f"    Low:  {min( n_per_slice.values() )}" )
        print( f"    High: {max( n_per_slice.values() )}" )

    if 'cell_global_all' in events.keys():
        n_per_dataset = { dataset: len( df_dataset['cell_global_all'].unique() )
                        for dataset, df_dataset in events.groupby( 'dataset_id_all' ) }
        print( "Cells per dataset:" )
        print( f"    Low:  {min( n_per_dataset.values() )}" )
        print( f"    High: {max( n_per_dataset.values() )}" )

    if grid_number is not None:
        n_per_dataset_roi = { dataset: len( df_dataset[f'grid_global_{grid_number}'].unique() )
                            for dataset, df_dataset in events.groupby( 'dataset_id_all' ) }
        print( "ROIs per dataset:" )
        print( f"    Low:  {min( n_per_dataset_roi.values() )}" )
        print( f"    High: {max( n_per_dataset_roi.values() )}" )

    if 'cell_global_all' in events.keys():
        n_per_cell = { cell: df_cell.shape[0]
                    for cell, df_cell in events.groupby( 'cell_global_all' ) }
        print( "Events per cell-recording:" )
        print( f"    Low:  {min( n_per_cell.values() )}" )
        print( f"    High: {max( n_per_cell.values() )}" )

    if grid_number is not None:
        n_per_roi = { roi: df_roi.shape[0]
                    for roi, df_roi in events.groupby( f'grid_global_{grid_number}' ) }
        print( "Events per ROI-recording:" )
        print( f"    Low:  {min( n_per_roi.values() )}" )
        print( f"    High: {max( n_per_roi.values() )}" )

    if 'GABA' in events['condition'].unique():

        print()

        filter_gaba = events['condition'] == 'GABA'
        events_gaba = events[filter_gaba]

        print( '** GABA:' )
        print( f"Datasets:        n = {len( events_gaba['dataset_id_all'].unique() )}" )
        if 'cell_global_all' in events_gaba.keys():
            print( f"Cell-recordings: n = {len( events_gaba['cell_global_all'].unique() )}" )
        if grid_number is not None:
            print( f"ROI-recordings:  n = {len( events_gaba[f'grid_global_{grid_number}'].unique() )}" )
        print( f"Events:          n = {events_gaba.shape[0]}" )

        if 'cell_global_all' in events_gaba.keys():
            n_per_dataset_gaba = { dataset: len( df_dataset['cell_global_all'].unique() )
                                for dataset, df_dataset in events_gaba.groupby( 'dataset_id_all' ) }
            print( "Cells per dataset:" )
            print( f"    Low:  {min( n_per_dataset_gaba.values() )}" )
            print( f"    High: {max( n_per_dataset_gaba.values() )}" )

        if grid_number is not None:
            n_per_dataset_roi_gaba = { dataset: len( df_dataset[f'grid_global_{grid_number}'].unique() )
                                    for dataset, df_dataset in events_gaba.groupby( 'dataset_id_all' ) }
            print( "ROIs per dataset:" )
            print( f"    Low:  {min( n_per_dataset_roi_gaba.values() )}" )
            print( f"    High: {max( n_per_dataset_roi_gaba.values() )}" )

    if 'Glu' in events['condition'].unique():

        print()

        filter_glu = events['condition'] == 'Glu'
        events_glu = events[filter_glu]

        print( '** Glu:' )
        print( f"Datasets:        n = {len( events_glu['dataset_id_all'].unique() )}" )
        if 'cell_global_all' in events_glu.keys():
            print( f"Cell-recordings: n = {len( events_glu['cell_global_all'].unique() )}" )
        if grid_number is not None:
            print( f"ROI-recordings:  n = {len( events_glu[f'grid_global_{grid_number}'].unique() )}" )
        print( f"Events:          n = {events_glu.shape[0]}" )

        if 'cell_global_all' in events_glu.keys():
            n_per_dataset_glu = { dataset: len( df_dataset['cell_global_all'].unique() )
                                for dataset, df_dataset in events_glu.groupby( 'dataset_id_all' ) }
            print( "Cells per dataset:" )
            print( f"    Low:  {min( n_per_dataset_glu.values() )}" )
            print( f"    High: {max( n_per_dataset_glu.values() )}" )

        if grid_number is not None:
            n_per_dataset_roi_glu = { dataset: len( df_dataset[f'grid_global_{grid_number}'].unique() )
                                    for dataset, df_dataset in events_glu.groupby( 'dataset_id_all' ) }
            print( "ROIs per dataset:" )
            print( f"    Low:  {min( n_per_dataset_roi_glu.values() )}" )
            print( f"    High: {max( n_per_dataset_roi_glu.values() )}" )


## General-purpose event analysis

def is_within(
        xs: ArrayLike,
        window: Window,
    ) -> ndarray:
    """
    Return a binary vector of whether each of `xs` lies within `window`

    Uses half-open convention; i.e., for `window = (a, b)`, uses
    * `xs >= a`, and
    * `xs < b`

    Parameters
    ----------
    xs : ArrayLike
        data points to check
    window : Window
        interval of form `(a, b)` to check against
    
    Returns
    -------
    ArrayLike
        binary vector, True wherever `xs` is within `window`
    """
    return (xs >= window[0]) & (xs < window[1])

def rate_within(
        ts: ArrayLike,
        window: Window
    ) -> float:
    """
    Rate of events with times `ts` inside of `window`

    Parameters
    ----------
    ts : ArrayLike
        time points to check
    window : Window
        interval of form `(a, b)` to check against
    
    Returns
    -------
    float
        the rate of `ts` within `window`, in units of 1 / X for X the unit of
        `ts` and `window`
    """
    filter_within = is_within( ts, window )
    width = window[1] - window[0]
    return np.sum( filter_within ) / width

def split_events_prop(
        events: DataFrame,
        keys: 'Optional[list[str]]' = None,
        new_key: str = 'mark_propTotal',
        threshold: float = 1.,
    ) -> 'tuple[DataFrame, DataFrame, DataFrame]':
    """
    Deterimne propagative and non-propagative subsets of `events

    Parameters
    ----------
    events : DataFrame
        table of event characteristics
    keys : list[str], optional
        list of mark keys (columns of `events`) to sum to determine total
        propagativeness; default sums over all the AQuA-generated cardinal
        direction propagation marks
    new_key : str, optional
        name of the new column that holds the summed propagation feature
        (default: 'mark_propTotal')
    threshold : float, optional
        events with the sum of the entries in `keys` > `threshold` are
        considered propagative, while all other events are considered non-
        propagative (default: 1., as in the paper). Note that the
        default AQuA units for propagation features are in µm.

    Returns
    -------
    events_new : DataFrame
        a copy of `events` with an added column for total propagation under
        '{new_key}', as well as thresholded total propagation under
        '{new_key}_bin'
    events_prop : DataFrame
        a copy of the propagative events from `events_new`
    events_no_prop : DataFrame
        a copy of the non-propagative events from `events_new`
    """

    events_new = events.copy()

    if keys is None:
        keys = [
            'mark_propGrowAnterior',
            'mark_propGrowLeft',
            'mark_propGrowPosterior',
            'mark_propGrowRight'
        ]

    events_new[new_key] = 0.
    for k in keys:
        events_new[new_key] += events_new[k]
        
    new_key_bin = f'{new_key}_bin'
    events_new[new_key_bin] = (events_new[new_key] > threshold).astype( float )

    filter_prop = events_new[new_key] > threshold
    events_prop = events_new[filter_prop].copy()
    events_no_prop = events_new[~filter_prop].copy()

    return events_new, events_prop, events_no_prop

def responders(
        df: DataFrame,
        group_key: str,
        t_key: str = 'start_time_rel',
        window_pre: Window = (-60., 0.),
        window_post: Window = (0., 120.),
        min_pre: int = 1,
        ratio_threshold: Optional[float] = 1.5,
        diff_threshold: Optional[float] = None,
        keep_keys: 'list[str]' = ['condition', 'dataset_id']
    ) -> DataFrame:
    """
    Determine whether individual groupings modulate their event rates between windows

    Parameters
    ----------
    df : pandas.DataFrame
        table of event characteristics
    group_key : str
        the key within `df` to aggregate event responses over (e.g., 'cell')
    t_key : str, optional
        the key within `df` that corresponds to time (default: 'start_time_rel')
    window_pre : Window, optional
        the time window corresponding to the baseline period (default:
        `(-60., 0.)`)
    window_post : Window, optional
        the time window in which responses are compared to the baseline
        (default: `(0., 120.)`)
    min_pre : int, optional
        groups with < `min_pre` events in the baseline period are excluded
        (default: 1, i.e., cells with any baseline events are included)
    ratio_threshold : float, optional
        the 'ratio_active' key in a group's result is set to 1 if that group's
        event rate in `window_post` is >= `ratio_threshold` times its rate in
        `window_pre` (default: 1.5). If set to `None`, then thresholded ratios
        are not computed.
    diff_threshold : float, optional
        the 'diff_active' key in a group's result is set to 1 if that group's
        event rate in `window_post` is >= `diff_threshold` above (in absolute
        units) its rate in `window_pre`; if set to `None`, then thresholded
        ratios are not computed (default: `None`)
    keep_keys : list[str], optional
        any keys that should be cached from each group in the result (default:
        `['condition', 'dataset_id']`)
    
    Returns
    -------
    pandas.DataFrame
        a table containing the thresholded results and any cached keys
    """
    
    ret = pd.DataFrame()
    
    for group, df_group in df.groupby( group_key ):
        
        n_pre = np.sum( is_within( df_group[t_key], window_pre ) )
        if min_pre is not None:
            if n_pre < min_pre:
                continue
        
        cur_row = dict()
        cur_row[group_key] = group
        for k in keep_keys:
            cur_row[k] = df_group.iloc[0][k]
        
        rate_pre = rate_within( df_group[t_key], window_pre )
        rate_post = rate_within( df_group[t_key], window_post )
        
        cur_row['rate_pre'] = rate_pre
        cur_row['rate_post'] = rate_post
        
        cur_row['ratio_post_pre'] = rate_post / rate_pre
        cur_row['diff_post_pre'] = rate_post - rate_pre
    
        if ratio_threshold is not None:
            cur_row['ratio_active'] = 1 if cur_row['ratio_post_pre'] >= ratio_threshold else 0
        if diff_threshold is not None:
            cur_row['diff_active'] = 1 if cur_row['diff_post_pre'] >= diff_threshold else 0

        cur_row_df = pd.DataFrame( cur_row,
                                   index = [group] )
        ret = pd.concat( [ret, cur_row_df],
                         ignore_index = True )
    
    # Re-introduce proper type consistency
    if ratio_threshold is not None:
        ret['ratio_active'] = ret['ratio_active'].astype( int )
    if diff_threshold is not None:
        ret['diff_active'] = ret['diff_active'].astype( int )
    ret[group_key] = ret[group_key].astype( df[group_key].dtype )
    for k in keep_keys:
        ret[k] = ret[k].astype( df[k].dtype )
    
    return ret

def features(
        df: DataFrame,
        group_key: str,
        t_key: str = 'start_time_rel',
        window: Window = (-60, 0),
        feature_keys: 'Optional[list[str]]' = None,
        keep_keys: 'list[str]' = ['condition', 'dataset_id'],
    ) -> DataFrame:
    """
    Compute the per-group average feature value of events within a window

    Parameters
    ----------
    df : pandas.DataFrame
        table of event characteristics
    group_key : str
        the key within `df` to aggregate event responses over (e.g., 'cell')
    t_key : str, optional
        the key within `df` that corresponds to time (default: 'start_time_rel')
    window : Window, optional
        the time window in which to do the averaging (default: `(-60., 0.)`)
    feature_keys : list[str], optional
        list of keys in `df` to produce averages for in each group (default:
        use all keys that start with )
    keep_keys : list[str], optional
        any keys that should be cached from each group in the result (default:
        `['condition', 'dataset_id']`)
        
    Returns
    -------
    pandas.DataFrame
        the average feature values for each member of `group_key`
    """
    
    ret_data = []
    
    if feature_keys is None:
        feature_keys = [ k
                         for k in df.keys()
                         if k.startswith( 'mark_' ) ]
    
    for group, df_group in df.groupby( group_key ):
        
        cur_row = dict()
        cur_row[group_key] = group
        for k in keep_keys:
            cur_row[k] = df_group.iloc[0][k]
        
        filter_window = is_within( df_group[t_key], window )
        df_group_window = df_group[filter_window]
        
        for k in feature_keys:
            val = np.nanmean( df_group_window[k] )
            cur_row[k] = val
        
        ret_data.append( cur_row )
    
    ret = pd.DataFrame( ret_data )
    
    # Re-introduce proper type consistency
    ret[group_key] = ret[group_key].astype( df[group_key].dtype )
    for k in keep_keys:
        ret[k] = ret[k].astype( df[k].dtype )
    
    return ret

def rates(
        event_kinds: 'dict[str, DataFrame]',
        group_key: str,
        t_key: str = 'start_time_rel',
        window: Window = (-60, 0),
        keep_keys: 'list[str]' = ['condition', 'dataset_id'],
    ) -> DataFrame:
    """
    Summarize the overall per-group rate of events of different specified kinds

    TODO Terrible nomenclature; `group_rates` is a very plausible confusion

    Parameters
    ----------
    event_kinds : dict[str, pandas.DataFrame]
        maps `k` to a dataset (as from a `HiveManager`) of events of kind `k`
    gorup_key : str
        the key within each event table to aggregate event responses over (e.g.,
        'cell')
    t_key : str, optional
        the key within `df` that corresponds to time (default: 'start_time_rel')
    window : Window
        the time window in which to determine the average rate per-group
        (default: `(-60., 0.)`)
    keep_keys : list[str], optional
        any keys that should be cached from each group in the result (default:
        `['condition', 'dataset_id']`)
    
    Returns
    -------
    pandas.DataFrame
        the rate of events for each specified kind within each group
    """
    
    ret_data = []
    
    kind_groups = { kind: events[group_key].unique()
                    for kind, events in event_kinds.items() }
    
    groups = set()
    for kind, gs in kind_groups.items():
        gs_set = set( list( gs.astype( int ) ) )
        groups = groups | set( gs )
    
    for group in groups:
        
        cur_ret = dict()
        cur_ret[group_key] = group
        
        keep_set = { k: False for k in keep_keys }
    
        for kind, events in event_kinds.items():
            
            filter_group = events[group_key] == group
            df_group = events[filter_group]
            
            for k in keep_keys:
                if keep_set[k]:
                    continue
                
                if df_group.shape[0] < 1:
                    continue
                
                cur_ret[k] = df_group.iloc[0][k]
            
            cur_ret[f'rate_{kind}'] = rate_within( df_group[t_key], window )
        
        ret_data.append( cur_ret )
        
    ret = pd.DataFrame( ret_data )
    
    # Re-introduce proper type consistency
    ret[group_key] = ret[group_key].astype( events[group_key].dtype )
    for k in keep_keys:
        ret[k] = ret[k].astype( events[k].dtype )
    
    return ret

def group_rates(
        df: DataFrame,
        group_key: str,
        windows: 'list[Window]',
        keep_keys: 'list[str]' = []
    ) -> DataFrame:
    """
    Summarize the per-group rate of events in `df` in each of the specified `windows`

    TODO Terrible nomenclature; `rates` is a very possible confusion

    Parameters
    ----------
    df : pandas.DataFrame
        table of event characteristics (as from a `HiveManager`)
    group_key : str
        the key within each event table to aggregate event responses over (e.g.,
        'cell')
    windows : list[Window]
        a list of time windows in which to compute the event rates for each
        group
    keep_keys : list[str], optional
        any keys that should be cached from each group in the result (default:
        `['condition', 'dataset_id']`)
    
    Returns
    -------
    pandas.DataFrame
        the rate of events within each window for each group
    """
    
    ret_data = dict()
    ret_data[group_key] = []
    for k in keep_keys:
        ret_data[k] = []
    for i_window, _ in enumerate( windows ):
        ret_data[f'rate_{i_window + 1}'] = []
    
    for group, df_group in df.groupby( group_key ):
        ret_data[group_key].append( group )
        for k in keep_keys:
            ret_data[k].append( df_group.iloc[0][k] )
        for i_window, window in enumerate( windows ):
            ret_data[f'rate_{i_window + 1}'].append( rate_within( df_group['start_time_rel'], window ) )
    
    return pd.DataFrame( ret_data )


## Postprocessing

def _add_event_decorators(
        headers, # TODO Type annotation
        events, # TODO Type annotation
        decorators: 'list[DecoratorSpec]' = None,
        verbose: bool = True
    ) -> DataFrame:
    """
    Apply the postprocessing decorators specified in `decorators` to data
    
    Parameters
    ----------
    headers
        header data for each dataset, as generated by a `HiveManager`
        TODO describe structure
    events
        TODO describe structure
    decorators : list[DecoratorSpec], optional
        a list of decorator specifications describing which additional
        postprocessing steps should be performed (default: produce decorators
        that compute 'start_time_rel' and 'is_uncage')

        Suppported decorators include:
        * 'start_time_rel': compute the time relative to the uncaging start
            using the 'uncageFrameStart' header entry
        * 'is_uncage': compute a boolean value of whether the current cell is
            the cell being uncaged in (`True`) or a neighbor (`False`) using
            the 'uncageCell' header entry
        * `('coreg', name, keys)`: create a new column called 'name' that
            produces a unique ID for each unique combination of values in `keys`
            using `pandas.DataFrame.groupby`
        * `('grid', size, max)`: create new columns grouping events by where
            the 'centroidRel_UM' values indicate the centroid of the event was
            relative to the uncaging site; `size` specifies the height and width
            of each grid cell in um, and `max` specifies how far in each
            cardinal direction from the uncaging site to build the grid
        * `('grid_abs', size, max)`: same as 'grid', but in absolute coordinates
            (only extending in the positive direction from (0, 0))
    verbose : bool, optional
        if `True`, produces progress bars for generating grid decorators, which
        can be somewhat slow (default: `True`)
    
    Returns
    -------
    pandas.DataFrame
        the events in `events`, but with additional columns holding the newly
        computed postprocessing data
    """
    
    if decorators is None:
        decorators = ['start_time_rel', 'is_uncage']
        
    ret = events.copy()
        
    for decorator in decorators:
        
        if type( decorator ) == str:
            decorator_name = decorator
            decorator_params = None
        if type( decorator ) == tuple:
            decorator_name = decorator[0]
            decorator_params = decorator[1:]
        
        if decorator_name == 'start_time_rel':
            it = utils._zip_by_dataset( headers, events )
            for dataset_header, (dataset_id, dataset_events) in it:
                dataset_t_uncage = dataset_header['Ts'] * dataset_header['uncageFrameStart'][0,0]
                ret.loc[dataset_events.index, 'start_time_rel'] = dataset_events['start_time'] - dataset_t_uncage
            
        elif decorator_name == 'is_uncage':
            it = utils._zip_by_dataset( headers, events )
            for dataset_header, (dataset_id, dataset_events) in it:
                ret.loc[dataset_events.index, 'is_uncage'] = dataset_events['cell'] == dataset_header['uncageCell'][0,0]
                
        elif decorator_name == 'coreg':
            coreg_name = decorator_params[0]
            coreg_keys = decorator_params[1]
            for i_group, (group_idx, group_slice) in enumerate( ret.groupby( coreg_keys ) ):
                ret.loc[group_slice.index, coreg_name] = i_group + 1
            ret[coreg_name] = ret[coreg_name].astype( int )
        
        elif decorator_name == 'grid':
            grid_size = decorator_params[0]
            grid_max = decorator_params[1]
            
            grid_edges_x = np.arange( -grid_max, grid_max + 0.01, grid_size )
            grid_edges_y = np.arange( -grid_max, grid_max + 0.01, grid_size )

            grid_dataset_key = f'grid_dataset_{grid_size:0d}'
            grid_global_key = f'grid_global_{grid_size:0d}'
            grid_x_key = f'grid_x_center_um_{grid_size:0d}'
            grid_y_key = f'grid_y_center_um_{grid_size:0d}'
            
            ret[grid_dataset_key] = -1
            ret[grid_global_key] = -1
            ret[grid_x_key] = np.nan
            ret[grid_y_key] = np.nan
            
            it = utils._zip_by_dataset( headers, events )
            if verbose:
                it = tqdm( it, desc = 'Grid decorators' )
            grid_global = 1
            for dataset_header, (dataset_id, dataset_events) in it:
                
                grid_dataset = 1
                for i_grid_x in range( grid_edges_x.shape[0] - 1 ):
                    
                    grid_left = grid_edges_x[i_grid_x]
                    grid_right = grid_edges_x[i_grid_x + 1]
                    for i_grid_y in range( grid_edges_y.shape[0] - 1 ):

                        grid_low = grid_edges_y[i_grid_y]
                        grid_high = grid_edges_y[i_grid_y + 1]
                        filter_grid = ((dataset_events['mark_centroidRelXUM'] >= grid_left)
                                       & (dataset_events['mark_centroidRelXUM'] < grid_right)
                                       & (dataset_events['mark_centroidRelYUM'] >= grid_low)
                                       & (dataset_events['mark_centroidRelYUM'] < grid_high))
                        grid_events = dataset_events[filter_grid]

                        ret.loc[grid_events.index, grid_dataset_key] = grid_dataset
                        ret.loc[grid_events.index, grid_global_key] = grid_global

                        ret.loc[grid_events.index, grid_x_key] = 0.5 * (grid_left + grid_right)
                        ret.loc[grid_events.index, grid_y_key] = 0.5 * (grid_low + grid_high)

                        grid_dataset += 1
                        grid_global += 1
                                            
                        # i_grid_y
                    # i_grid_x
                # dataset_id
            # elif
            
            ret[grid_dataset_key] = ret[grid_dataset_key].astype( int )
            ret[grid_global_key] = ret[grid_global_key].astype( int )
        
        # TODO Duplication
        elif decorator_name == 'grid_abs':
            grid_size = decorator_params[0]
            grid_max = decorator_params[1]
            
            grid_edges_x = np.arange( 0, grid_max + 0.01, grid_size )
            grid_edges_y = np.arange( 0, grid_max + 0.01, grid_size )

            grid_dataset_key = f'grid_dataset_abs_{grid_size:0d}'
            grid_global_key = f'grid_global_abs_{grid_size:0d}'
            grid_x_key = f'grid_x_center_um_abs_{grid_size:0d}'
            grid_y_key = f'grid_y_center_um_abs_{grid_size:0d}'
            
            ret[grid_dataset_key] = -1
            ret[grid_global_key] = -1
            ret[grid_x_key] = np.nan
            ret[grid_y_key] = np.nan
            
            it = utils._zip_by_dataset( headers, events )
            if verbose:
                it = tqdm( it, desc = 'Grid decorators' )
            grid_global = 1
            for dataset_header, (dataset_id, dataset_events) in it:

                # print( f'{dataset_id}: {dataset_events.shape[0]}' )

                n_selected = 0

                grid_dataset = 1
                for i_grid_x in range( grid_edges_x.shape[0] - 1 ):
                    
                    grid_left = grid_edges_x[i_grid_x]
                    grid_right = grid_edges_x[i_grid_x + 1]
                    for i_grid_y in range( grid_edges_y.shape[0] - 1 ):

                        grid_low = grid_edges_y[i_grid_y]
                        grid_high = grid_edges_y[i_grid_y + 1]

                        filter_grid = ((dataset_events['mark_centroidXUM'] >= grid_left)
                                       & (dataset_events['mark_centroidXUM'] < grid_right)
                                       & (dataset_events['mark_centroidYUM'] >= grid_low)
                                       & (dataset_events['mark_centroidYUM'] < grid_high))
                        grid_events = dataset_events[filter_grid]
                        # n_selected += grid_events.shape[0]

                        ret.loc[grid_events.index, grid_dataset_key] = grid_dataset
                        ret.loc[grid_events.index, grid_global_key] = grid_global

                        ret.loc[grid_events.index, grid_x_key] = 0.5 * (grid_left + grid_right)
                        ret.loc[grid_events.index, grid_y_key] = 0.5 * (grid_low + grid_high)

                        grid_dataset += 1
                        grid_global += 1
                                            
                        # i_grid_y
                    # i_grid_x

                # print( n_selected )

                # dataset_id
            # elif
            
            ret[grid_dataset_key] = ret[grid_dataset_key].astype( int )
            ret[grid_global_key] = ret[grid_global_key].astype( int )

        elif decorator_name == 'centroid_dist':
            raise NotImplemented()
        
        else:
            # Unrecognized decorator
            raise Exception( f'Unrecognized decorator: {decorator}' )
    
    return ret