//
//  Copyright (C) 2001-2017 Greg Landrum and Rational Discovery LLC
//
//   @@ All Rights Reserved @@
//  This file is part of the RDKit.
//  The contents are covered by the terms of the BSD license
//  which is included in the file license.txt, found at the root
//  of the RDKit source tree.
//

//  Automatically generated from elements.txt
#include "atomic_data.h"
#include <boost/tokenizer.hpp>
#include <sstream>
#include <locale>

typedef boost::tokenizer<boost::char_separator<char>> tokenizer;

namespace RDKit {

namespace constants {
const double electronMass =
    0.00054857991;  // value from http://physics.nist.gov/cgi-bin/cuu/Value?me
}

// these data came from a variety of sources. If it's not explicitly listed
// here, then it has likely been lost in antiquity. Columns:
//  atomic number
//  symbol
//  rCov
//  rB0
//  rVdw (taken from BODR v10.1)
//  atomic mass
//  # outershell electrons
//  most common isotope
//  most common isotopic mass
//  list of allowed valences, -1 for anything
const std::string periodicTableAtomData =
    "0	*	0	0	0	0	0	0	0	-1 \n \
1	H 	0.23	0.33	1.2	1.008	1	1	1.007825032	1 \n \
2	He	0.93	0.7	1.4	4.003	2	4	4.002603254	0  \n \
3	Li	0.68	1.23	2.2	6.941	1	7	7.01600455	1  \n \
4	Be	0.35	0.9	1.9	9.012	2	9	9.0121822	2  \n \
5	B	0.830	0.82	1.8	10.812	3	11	11.0093054	3  \n \
6	C	0.68	0.77	1.7	12.011	4	12	12	4 \n \
7	N	0.68	0.7	1.6	14.007	5	14	14.003074	3 \n \
8	O	0.68	0.66	1.55	15.999	6	16	15.99491462	2 \n \
9	F	0.64	0.611	1.5	18.998	7	19	18.99840322	1 \n \
10	Ne	1.12	0.7	1.54	20.18	8	20	19.99244018	0 \n \
11	Na	0.97	1.54	2.4	22.99	1	23	22.98976928	1 \n \
12	Mg	1.1	1.36	2.2	24.305	2	24	23.9850417	2 \n \
13	Al	1.35	1.18	2.1	26.982	3	27	26.98153863	3	6 \n \
14	Si	1.2	0.937	2.1	28.086	4	28	27.97692653	4	6 \n \
15	P	0.75	0.89	1.95	30.974	5	31	30.97376163	3	5	7 \n \
16	S	1.02	1.04	1.8	32.067	6	32	31.972071	2	4	6 \n \
17	Cl	0.99	0.997	1.8	35.453	7	35	34.96885268	1 \n \
18	Ar	1.57	1.74	1.88	39.948	8	40	39.96238312	0 \n \
19	K	1.33	2.03	2.8	39.098	1	39	38.96370668	1 \n \
20	Ca	0.99	1.74	2.4	40.078	2	40	39.96259098	2 \n"
    "21	Sc	1.44	1.44	2.3	44.956	3	45	44.9559119	-1 \n \
22	Ti	1.47	1.32	2.15	47.867	4	48	47.9479463	-1 \n \
23	V	1.33	1.22	2.05	50.942	5	50	49.9471585	-1 \n \
24	Cr	1.35	1.18	2.05	51.996	6	52	51.9405075	-1 \n \
25	Mn	1.35	1.17	2.05	54.938	7	55	54.9380451	-1 \n \
26	Fe	1.34	1.17	2.05	55.845	8	56	55.9349375	-1 \n \
27	Co	1.33	1.16	2.0	58.933	9	59	58.933195	-1 \n \
28	Ni	1.5	1.15	2.0	58.693	10	58	57.9353429	-1 \n \
29	Cu	1.52	1.17	2.0	63.546	11	63	62.9295975	-1 \n \
30	Zn	1.45	1.25	2.1	65.39	2	64	63.9291422	-1 \n \
31	Ga	1.22	1.26	2.1	69.723	3	69	68.9255736	3	   \n \
32	Ge	1.17	1.188	2.1	72.61	4	74	73.9211778	4	   \n \
33	As	1.21	1.2	2.05	74.922	5	75	74.9215965	3	5	7 \n \
34	Se	1.22	1.17	1.9	78.96	6	80	79.9165213	2	4	6 \n \
35	Br	1.21	1.167	1.9	79.904	7	79	78.9183371	1 \n \
36	Kr	1.91	1.91	2.02	83.8	8	84	83.911507	0 \n \
37	Rb	1.47	2.16	2.9	85.468	1	85	84.91178974	1 \n \
38	Sr	1.12	1.91	2.55	87.62	2	88	87.9056121	2 \n \
39	Y	1.78	1.62	2.4	88.906	3	89	88.9058483	-1 \n \
40	Zr	1.56	1.45	2.3	91.224	4	90	89.9047044	-1 \n"
    "41	Nb	1.48	1.34	2.15	92.906	5	93	92.9063781	-1 \n \
42	Mo	1.47	1.3	2.1	95.94	6	98	97.9054082	-1 \n \
43	Tc	1.35	1.27	2.05	98	7	97	96.906365	-1 \n \
44	Ru	1.4	1.25	2.05	101.07	8	102	101.9043493	-1 \n \
45	Rh	1.45	1.25	2.0	102.906	9	103	102.905504	-1 \n \
46	Pd	1.5	1.28	2.05	106.42	10	106	105.903486	-1 \n \
47	Ag	1.59	1.34	2.1	107.868	11	107	106.905097	-1 \n \
48	Cd	1.69	1.48	2.2	112.412	2	114	113.9033585	-1 \n \
49	In	1.63	1.44	2.2	114.818	3	115	114.903878	3 \n \
50	Sn	1.46	1.385	2.25	118.711	4	120	119.9021947	2	4 \n \
51	Sb	1.46	1.4	2.2	121.76	5	121	120.9038157	3	5	7 \n \
52	Te	1.47	1.378	2.1	127.6	6	130	129.9062244	2 4 6\n \
53	I	1.4	1.387	2.1	126.904	7	127	126.904473	1	2	5 \n \
54	Xe	1.98	1.98	2.16	131.29	8	132	131.9041535	0 	2 \n \
55	Cs	1.67	2.35	3.0	132.905	1	133	132.9054519	1 \n \
56	Ba	1.34	1.98	2.7	137.328	2	138	137.9052472	2 \n \
57	La	1.87	1.69	2.5	138.906	3	139	138.9063533	-1 \n \
58	Ce	1.83	1.83	2.48	140.116	4	140	139.9054387	-1 \n \
59	Pr	1.82	1.82	2.47	140.908	3	141	140.9076528	-1 \n \
60	Nd	1.81	1.81	2.45	144.24	4	142	141.9077233	-1 \n"
    "61	Pm	1.8	1.8	2.43	145	5	145	144.912749	-1 \n \
62	Sm	1.8	1.8	2.42	150.36	6	152	151.9197324	-1 \n \
63	Eu	1.99	1.99	2.4	151.964	7	153	152.9212303	-1 \n \
64	Gd	1.79	1.79	2.38	157.25	8	158	157.9241039	-1 \n \
65	Tb	1.76	1.76	2.37	158.925	9	159	158.9253468	-1 \n \
66	Dy	1.75	1.75	2.35	162.5	10	164	163.9291748	-1 \n \
67	Ho	1.74	1.74	2.33	164.93	11	165	164.9303221	-1 \n \
68	Er	1.73	1.73	2.32	167.26	12	166	165.9302931	-1 \n \
69	Tm	1.72	1.72	2.3	168.934	13	169	168.9342133	-1 \n \
70	Yb	1.94	1.94	2.28	173.04	14	174	173.9388621	-1 \n \
71	Lu	1.72	1.72	2.27	174.967	15	175	174.9407718	-1 \n \
72	Hf	1.57	1.44	2.25	178.49	4	180	179.94655	-1 \n \
73	Ta	1.43	1.34	2.2	180.948	5	181	180.9479958	-1 \n \
74	W	1.37	1.3	2.1	183.84	6	184	183.9509312	-1 \n \
75	Re	1.35	1.28	2.05	186.207	7	187	186.9557531	-1 \n \
76	Os	1.37	1.26	2.0	190.23	8	192	191.9614807	-1 \n \
77	Ir	1.32	1.27	2.0	192.217	9	193	192.9629264	-1 \n \
78	Pt	1.5	1.3	2.05	195.078	10	195	194.9647911	-1 \n \
79	Au	1.5	1.34	2.1	196.967	11	197	196.9665687	-1 \n \
80	Hg	1.7	1.49	2.05	200.59	2	202	201.970643	-1 \n"
    "81	Tl	1.55	1.48	2.2	204.383	3	205	204.9744275	3 \n \
82	Pb	1.54	1.48	2.3	207.2	4	208	207.9766521	2	4 \n \
83	Bi	1.54	1.45	2.3	208.98	5	209	208.9803987	3	5	7 \n \
84	Po	1.68	1.46	2.0	209	6	209	208.9824304	2 \n \
85	At	1.7	1.45	2.0	210	7	210	209.987148	1 \n \
86	Rn	2.4	2.4	2.0	222	8	211	210.990601	0 \n \
87	Fr	2	2	2.0	223	1	223	223.0197359	1 \n \
88	Ra	1.9	1.9	2.0	226	2	223	223.0185022	2 \n \
89	Ac	1.88	1.88	2.0	227	3	227	227.0277521	-1 \n \
90	Th	1.79	1.79	2.4	232.038	4	232	232.0380553	-1 \n \
91	Pa	1.61	1.61	2.0	231.036	3	231	231.035884	-1 \n \
92	U	1.58	1.58	2.3	238.029	4	238	238.0507882	-1 \n \
93	Np	1.55	1.55	2.0	237	5	236	236.04657	-1 \n \
94	Pu	1.53	1.53	2.0	244	6	238	238.0495599	-1 \n \
95	Am	1.51	1.07	2.0	243	7	241	241.0568291	-1 \n \
96	Cm	1.5	0	2.0	247	8	243	243.0613891	-1 \n \
97	Bk	1.5	0	2.0	247	9	247	247.070307	-1 \n \
98	Cf	1.5	0	2.0	251	10	249	249.0748535	-1 \n \
99	Es	1.5	0	2.0	252	11	252	252.08298	-1 \n \
100	Fm	1.5	0	2.0	257	12	257	257.095105	-1 \n"
    "101	Md	1.5	0	2.0	258	13	258	258.098431	-1 \n \
102	No	1.5	0	2.0	259	14	259	259.10103	-1 \n \
103	Lr	1.5	0	2.0	262	15	262	262.10963	-1 \n \
104	Rf	1.57	0	2.0	267	2	267	267.12153	-1 \n \
105	Db	1.49	0	2.0	268	2 	268	268.12545	-1 \n \
106	Sg	1.43	0	2.0	269	2	271	271.13347	-1 \n \
107	Bh	1.41	0	2.0	270	2	270	270.13362	-1 \n \
108	Hs	1.34	0	2.0	269	2	269	269.13406	-1 \n \
109	Mt	1.29	0	2.0	278	2	278	278.15481	-1 \n \
110	Ds	1.28	0	2.0	281	2	281	281.16206	-1 \n \
111	Rg	1.21	0	2.0	281	2	281	281.16537	-1 \n \
112	Cn	1.22	0	2.0	285	2	285	285.17411	-1 \n"
    // The Blue Obelisk Data Repository has no VdW radii for those:
    // Ds Rg Cn Uut Fl Mc Uup Lv Ts Og; we use 2.0 for all of them
    // ---
    // added from BODR 30.10.2016
    // atomic mass data from NIST
    // we leave Uut and Uup in here for backwards
    // compatibility. Nh and Mc (the entries appearing first
    // for a particular atomic number) will be the values returned
    // when looking an atomic symbol up using atomic number.
    // In the event that symbols need to be corrected in the future,
    // follow this same pattern to ensure backwards compatibility.
    "113	Nh	1.36	0	2.0	284	2	284	284.17873	-1 \n \
113	Uut	1.36	0	2.0	284	2	284	284.17873	-1 \n \
114	Fl	1.43	0	2.0	289	2	289	289.19042	-1 \n \
115	Mc	1.62	0	2.0	288	2	288	288.19274	-1 \n \
115	Uup	1.62	0	2.0	288	2	288	288.19274	-1 \n \
116	Lv	1.75	0	2.0	293	2	293	293.20449	-1 \n \
117	Ts	1.65	0	2.0	292	2	292	292.20746	-1 \n \
118	Og	1.57	0	2.0	294	2	294	294.21392	-1 \n \
";

atomicData::atomicData(const std::string &dataLine) {
  boost::char_separator<char> spaceSep(" \t");
  tokenizer tokens(dataLine, spaceSep);
  tokenizer::iterator token = tokens.begin();
  std::istringstream istr;
  istr.imbue(std::locale("C"));

  // atomic number first
  istr.clear();
  istr.str(*token);
  istr >> anum;
  ++token;

  // element symbol
  symb = *token;
  ++token;

  // covalent radius
  istr.clear();
  istr.str(*token);
  istr >> rCov;
  ++token;

  // rB0
  istr.clear();
  istr.str(*token);
  istr >> rB0;
  ++token;

  //  Van derWaal radius
  istr.clear();
  istr.str(*token);
  istr >> rVdw;
  ++token;

  // atomic mass
  istr.clear();
  istr.str(*token);
  istr >> mass;
  ++token;

  // number of outshell electrons
  istr.clear();
  istr.str(*token);
  istr >> nVal;
  ++token;

  // most common isotope
  istr.clear();
  istr.str(*token);
  istr >> commonIsotope;
  ++token;

  // most common isotopic mass
  istr.clear();
  istr.str(*token);
  istr >> commonIsotopeMass;
  ++token;

  // now the valences
  valence.clear();
  while (token != tokens.end()) {
    istr.clear();
    istr.str(*token);
    int tval;
    istr >> tval;
    valence.push_back(tval);
    ++token;
  }
}

// data taken from the BODR
// http://bodr.svn.sourceforge.net/viewvc/bodr/trunk/bodr/isotopes/isotopes.xml?revision=30

// another joy of VC++ "compiler limit: string exceeds 65535 bytes in length"
// compels us to
// break this into pieces
const std::string isotopesAtomData[] = {
    "1	H	1	1.007825032	99.9885 \n \
1	H	2	2.014101778	0.0115 \n \
1	H	3	3.016049278	0 \n \
1	H	4	4.02781	0 \n \
1	H	5	5.03531	0 \n \
1	H	6	6.04494	0 \n \
1	H	7	7.05275	0 \n \
2	He	3	3.016029319	0.000137 \n \
2	He	4	4.002603254	99.999863 \n \
2	He	5	5.01222	0 \n \
2	He	6	6.0188891	0 \n \
2	He	7	7.028021	0 \n \
2	He	8	8.033922	0 \n \
2	He	9	9.04395	0 \n \
2	He	10	10.0524	0 \n \
3	Li	3	3.03078	0 \n \
3	Li	4	4.02719	0 \n \
3	Li	5	5.01254	0 \n \
3	Li	6	6.015122795	7.59 \n \
3	Li	7	7.01600455	92.41 \n \
3	Li	8	8.02248736	0 \n \
3	Li	9	9.0267895	0 \n \
3	Li	10	10.035481	0 \n \
3	Li	11	11.043798	0 \n \
3	Li	12	12.05378	0 \n \
4	Be	5	5.04079	0 \n \
4	Be	6	6.019726	0 \n \
4	Be	7	7.01692983	0 \n \
4	Be	8	8.0053051	0 \n \
4	Be	9	9.0121822	100 \n \
4	Be	10	10.0135338	0 \n \
4	Be	11	11.021658	0 \n \
4	Be	12	12.026921	0 \n \
4	Be	13	13.03569	0 \n \
4	Be	14	14.04289	0 \n \
4	Be	15	15.05346	0 \n \
4	Be	16	16.06192	0 \n \
5	B	6	6.04681	0 \n \
5	B	7	7.02992	0 \n \
5	B	8	8.0246072	0 \n \
5	B	9	9.0133288	0 \n \
5	B	10	10.012937	19.9 \n \
5	B	11	11.0093054	80.1 \n \
5	B	12	12.0143521	0 \n \
5	B	13	13.0177802	0 \n \
5	B	14	14.025404	0 \n \
5	B	15	15.031103	0 \n \
5	B	16	16.03981	0 \n \
5	B	17	17.04699	0 \n \
5	B	18	18.05617	0 \n \
5	B	19	19.06373	0 \n",
    "6	C	8	8.037675	0 \n \
6	C	9	9.0310367	0 \n \
6	C	10	10.0168532	0 \n \
6	C	11	11.0114336	0 \n \
6	C	12	12	98.93 \n \
6	C	13	13.00335484	1.07 \n \
6	C	14	14.00324199	0 \n \
6	C	15	15.0105993	0 \n \
6	C	16	16.014701	0 \n \
6	C	17	17.022586	0 \n \
6	C	18	18.02676	0 \n \
6	C	19	19.03481	0 \n \
6	C	20	20.04032	0 \n \
6	C	21	21.04934	0 \n \
6	C	22	22.0572	0 \n \
7	N	10	10.04165	0 \n \
7	N	11	11.02609	0 \n \
7	N	12	12.0186132	0 \n \
7	N	13	13.00573861	0 \n \
7	N	14	14.003074	99.632 \n \
7	N	15	15.0001089	0.368 \n \
7	N	16	16.0061017	0 \n \
7	N	17	17.00845	0 \n \
7	N	18	18.014079	0 \n \
7	N	19	19.017029	0 \n \
7	N	20	20.02337	0 \n \
7	N	21	21.02711	0 \n \
7	N	22	22.03439	0 \n \
7	N	23	23.04122	0 \n \
7	N	24	24.05104	0 \n \
7	N	25	25.06066	0 \n \
8	O	12	12.034405	0 \n \
8	O	13	13.024812	0 \n \
8	O	14	14.00859625	0 \n \
8	O	15	15.0030656	0 \n \
8	O	16	15.99491462	99.757 \n \
8	O	17	16.9991317	0.038 \n \
8	O	18	17.999161	0.205 \n \
8	O	19	19.00358	0 \n \
8	O	20	20.0040767	0 \n \
8	O	21	21.008656	0 \n \
8	O	22	22.00997	0 \n \
8	O	23	23.01569	0 \n \
8	O	24	24.02047	0 \n \
8	O	25	25.02946	0 \n \
8	O	26	26.03834	0 \n \
8	O	27	27.04826	0 \n \
8	O	28	28.05781	0 \n \
9	F	14	14.03506	0 \n",
    "9	F	15	15.01801	0 \n \
9	F	16	16.011466	0 \n \
9	F	17	17.00209524	0 \n \
9	F	18	18.000938	0 \n \
9	F	19	18.99840322	100 \n \
9	F	20	19.99998132	0 \n \
9	F	21	20.999949	0 \n \
9	F	22	22.002999	0 \n \
9	F	23	23.00357	0 \n \
9	F	24	24.00812	0 \n \
9	F	25	25.0121	0 \n \
9	F	26	26.01962	0 \n \
9	F	27	27.02676	0 \n \
9	F	28	28.03567	0 \n \
9	F	29	29.04326	0 \n \
9	F	30	30.0525	0 \n \
9	F	31	31.06043	0 \n \
10	Ne	16	16.025761	0 \n \
10	Ne	17	17.017672	0 \n \
10	Ne	18	18.0057082	0 \n \
10	Ne	19	19.0018802	0 \n \
10	Ne	20	19.99244018	90.48 \n \
10	Ne	21	20.99384668	0.27 \n \
10	Ne	22	21.99138511	9.25 \n \
10	Ne	23	22.9944669	0 \n \
10	Ne	24	23.9936108	0 \n \
10	Ne	25	24.997737	0 \n \
10	Ne	26	26.000461	0 \n \
10	Ne	27	27.00759	0 \n \
10	Ne	28	28.01207	0 \n \
10	Ne	29	29.01939	0 \n \
10	Ne	30	30.0248	0 \n \
10	Ne	31	31.03311	0 \n \
10	Ne	32	32.04002	0 \n \
10	Ne	33	33.04938	0 \n \
10	Ne	34	34.05703	0 \n \
11	Na	18	18.02597	0 \n \
11	Na	19	19.013877	0 \n \
11	Na	20	20.007351	0 \n \
11	Na	21	20.9976552	0 \n \
11	Na	22	21.9944364	0 \n \
11	Na	23	22.98976928	100 \n \
11	Na	24	23.99096278	0 \n \
11	Na	25	24.989954	0 \n \
11	Na	26	25.992633	0 \n \
11	Na	27	26.994077	0 \n \
11	Na	28	27.998938	0 \n \
11	Na	29	29.002861	0 \n \
11	Na	30	30.008976	0 \n \
11	Na	31	31.01359	0 \n \
11	Na	32	32.02047	0 \n",
    "11	Na	33	33.02672	0 \n \
11	Na	34	34.03517	0 \n \
11	Na	35	35.04249	0 \n \
11	Na	36	36.05148	0 \n \
11	Na	37	37.05934	0 \n \
12	Mg	19	19.03547	0 \n \
12	Mg	20	20.018863	0 \n \
12	Mg	21	21.011713	0 \n \
12	Mg	22	21.9995738	0 \n \
12	Mg	23	22.9941237	0 \n \
12	Mg	24	23.9850417	78.99 \n \
12	Mg	25	24.98583692	10.00 \n \
12	Mg	26	25.98259293	11.01 \n \
12	Mg	27	26.98434059	0 \n \
12	Mg	28	27.9838768	0 \n \
12	Mg	29	28.9886	0 \n \
12	Mg	30	29.990434	0 \n \
12	Mg	31	30.996546	0 \n \
12	Mg	32	31.998975	0 \n \
12	Mg	33	33.005254	0 \n \
12	Mg	34	34.00946	0 \n \
12	Mg	35	35.01734	0 \n \
12	Mg	36	36.023	0 \n \
12	Mg	37	37.0314	0 \n \
12	Mg	38	38.03757	0 \n \
12	Mg	39	39.04677	0 \n \
12	Mg	40	40.05393	0 \n \
13	Al	21	21.02804	0 \n \
13	Al	22	22.01952	0 \n \
13	Al	23	23.007267	0 \n \
13	Al	24	23.9999389	0 \n \
13	Al	25	24.9904281	0 \n \
13	Al	26	25.98689169	0 \n \
13	Al	27	26.98153863	100 \n \
13	Al	28	27.98191031	0 \n \
13	Al	29	28.980445	0 \n \
13	Al	30	29.98296	0 \n \
13	Al	31	30.983947	0 \n \
13	Al	32	31.98812	0 \n \
13	Al	33	32.99084	0 \n \
13	Al	34	33.99685	0 \n \
13	Al	35	34.99986	0 \n \
13	Al	36	36.00621	0 \n \
13	Al	37	37.01068	0 \n \
13	Al	38	38.01723	0 \n \
13	Al	39	39.02297	0 \n \
13	Al	40	40.03145	0 \n \
13	Al	41	41.03833	0 \n \
13	Al	42	42.04689	0 \n \
14	Si	22	22.03453	0 \n \
14	Si	23	23.02552	0 \n",
    "14	Si	24	24.011546	0 \n \
14	Si	25	25.004106	0 \n \
14	Si	26	25.99233	0 \n \
14	Si	27	26.98670491	0 \n \
14	Si	28	27.97692653	92.2297 \n \
14	Si	29	28.9764947	4.6832 \n \
14	Si	30	29.97377017	3.0872 \n \
14	Si	31	30.97536323	0 \n \
14	Si	32	31.97414808	0 \n \
14	Si	33	32.978	0 \n \
14	Si	34	33.978576	0 \n \
14	Si	35	34.98458	0 \n \
14	Si	36	35.9866	0 \n \
14	Si	37	36.99294	0 \n \
14	Si	38	37.99563	0 \n \
14	Si	39	39.00207	0 \n \
14	Si	40	40.00587	0 \n \
14	Si	41	41.01456	0 \n \
14	Si	42	42.01979	0 \n \
14	Si	43	43.02866	0 \n \
14	Si	44	44.03526	0 \n \
15	P	24	24.03435	0 \n \
15	P	25	25.02026	0 \n \
15	P	26	26.01178	0 \n \
15	P	27	26.99923	0 \n \
15	P	28	27.992315	0 \n \
15	P	29	28.9818006	0 \n \
15	P	30	29.9783138	0 \n \
15	P	31	30.97376163	100 \n \
15	P	32	31.97390727	0 \n \
15	P	33	32.9717255	0 \n \
15	P	34	33.973636	0 \n \
15	P	35	34.9733141	0 \n \
15	P	36	35.97826	0 \n \
15	P	37	36.97961	0 \n \
15	P	38	37.98416	0 \n \
15	P	39	38.98618	0 \n \
15	P	40	39.9913	0 \n \
15	P	41	40.99434	0 \n \
15	P	42	42.00101	0 \n \
15	P	43	43.00619	0 \n \
15	P	44	44.01299	0 \n \
15	P	45	45.01922	0 \n \
15	P	46	46.02738	0 \n \
16	S	26	26.02788	0 \n \
16	S	27	27.01883	0 \n \
16	S	28	28.00437	0 \n \
16	S	29	28.99661	0 \n \
16	S	30	29.984903	0 \n \
16	S	31	30.9795547	0 \n \
16	S	32	31.972071	94.93 \n",
    "16	S	33	32.97145876	0.76 \n \
16	S	34	33.9678669	4.29 \n \
16	S	35	34.96903216	0 \n \
16	S	36	35.96708076	0.02 \n \
16	S	37	36.97112557	0 \n \
16	S	38	37.971163	0 \n \
16	S	39	38.97513	0 \n \
16	S	40	39.97545	0 \n \
16	S	41	40.97958	0 \n \
16	S	42	41.98102	0 \n \
16	S	43	42.98715	0 \n \
16	S	44	43.99021	0 \n \
16	S	45	44.99651	0 \n \
16	S	46	46.00075	0 \n \
16	S	47	47.00859	0 \n \
16	S	48	48.01417	0 \n \
16	S	49	49.02362	0 \n \
17	Cl	28	28.02851	0 \n \
17	Cl	29	29.01411	0 \n \
17	Cl	30	30.00477	0 \n \
17	Cl	31	30.99241	0 \n \
17	Cl	32	31.98569	0 \n \
17	Cl	33	32.9774519	0 \n \
17	Cl	34	33.97376282	0 \n \
17	Cl	35	34.96885268	75.78 \n \
17	Cl	36	35.96830698	0 \n \
17	Cl	37	36.96590259	24.22 \n \
17	Cl	38	37.96801043	0 \n \
17	Cl	39	38.9680082	0 \n \
17	Cl	40	39.97042	0 \n \
17	Cl	41	40.97068	0 \n \
17	Cl	42	41.97325	0 \n \
17	Cl	43	42.97405	0 \n \
17	Cl	44	43.97828	0 \n \
17	Cl	45	44.98029	0 \n \
17	Cl	46	45.98421	0 \n \
17	Cl	47	46.98871	0 \n \
17	Cl	48	47.99495	0 \n \
17	Cl	49	49.00032	0 \n \
17	Cl	50	50.00784	0 \n \
17	Cl	51	51.01449	0 \n \
18	Ar	30	30.02156	0 \n \
18	Ar	31	31.01212	0 \n \
18	Ar	32	31.997638	0 \n \
18	Ar	33	32.9899257	0 \n \
18	Ar	34	33.9802712	0 \n \
18	Ar	35	34.9752576	0 \n \
18	Ar	36	35.96754511	0.3365 \n \
18	Ar	37	36.96677632	0 \n \
18	Ar	38	37.9627324	0.0632 \n \
18	Ar	39	38.964313	0 \n",
    "18	Ar	40	39.96238312	99.6003 \n \
18	Ar	41	40.9645006	0 \n \
18	Ar	42	41.963046	0 \n \
18	Ar	43	42.965636	0 \n \
18	Ar	44	43.964924	0 \n \
18	Ar	45	44.96804	0 \n \
18	Ar	46	45.96809	0 \n \
18	Ar	47	46.97219	0 \n \
18	Ar	48	47.97454	0 \n \
18	Ar	49	48.98052	0 \n \
18	Ar	50	49.98443	0 \n \
18	Ar	51	50.99163	0 \n \
18	Ar	52	51.99678	0 \n \
18	Ar	53	53.00494	0 \n \
19	K	32	32.02192	0 \n \
19	K	33	33.00726	0 \n \
19	K	34	33.99841	0 \n \
19	K	35	34.98801	0 \n \
19	K	36	35.981292	0 \n \
19	K	37	36.97337589	0 \n \
19	K	38	37.9690812	0 \n \
19	K	39	38.96370668	93.2581 \n \
19	K	40	39.96399848	0.0117 \n \
19	K	41	40.96182576	6.7302 \n \
19	K	42	41.96240281	0 \n \
19	K	43	42.960716	0 \n \
19	K	44	43.96156	0 \n \
19	K	45	44.960699	0 \n \
19	K	46	45.961977	0 \n \
19	K	47	46.961678	0 \n \
19	K	48	47.965514	0 \n \
19	K	49	48.96745	0 \n \
19	K	50	49.97278	0 \n \
19	K	51	50.97638	0 \n \
19	K	52	51.98261	0 \n \
19	K	53	52.98712	0 \n \
19	K	54	53.9942	0 \n \
19	K	55	54.99971	0 \n \
20	Ca	34	34.01412	0 \n \
20	Ca	35	35.00494	0 \n \
20	Ca	36	35.99309	0 \n \
20	Ca	37	36.98587	0 \n \
20	Ca	38	37.976318	0 \n \
20	Ca	39	38.9707197	0 \n \
20	Ca	40	39.96259098	96.941 \n \
20	Ca	41	40.96227806	0 \n \
20	Ca	42	41.95861801	0.647 \n \
20	Ca	43	42.9587666	0.135 \n \
20	Ca	44	43.9554818	2.086 \n \
20	Ca	45	44.9561866	0 \n \
20	Ca	46	45.9536926	0.004 \n",
    "20	Ca	47	46.954546	0 \n \
20	Ca	48	47.952534	0.187 \n \
20	Ca	49	48.955674	0 \n \
20	Ca	50	49.957519	0 \n \
20	Ca	51	50.9615	0 \n \
20	Ca	52	51.9651	0 \n \
20	Ca	53	52.97005	0 \n \
20	Ca	54	53.97435	0 \n \
20	Ca	55	54.98055	0 \n \
20	Ca	56	55.98557	0 \n \
20	Ca	57	56.99236	0 \n \
21	Sc	36	36.01492	0 \n \
21	Sc	37	37.00305	0 \n \
21	Sc	38	37.9947	0 \n \
21	Sc	39	38.98479	0 \n \
21	Sc	40	39.977967	0 \n \
21	Sc	41	40.96925113	0 \n \
21	Sc	42	41.96551643	0 \n \
21	Sc	43	42.9611507	0 \n \
21	Sc	44	43.9594028	0 \n \
21	Sc	45	44.9559119	100 \n \
21	Sc	46	45.9551719	0 \n \
21	Sc	47	46.9524075	0 \n \
21	Sc	48	47.952231	0 \n \
21	Sc	49	48.950024	0 \n \
21	Sc	50	49.952188	0 \n \
21	Sc	51	50.953603	0 \n \
21	Sc	52	51.95668	0 \n \
21	Sc	53	52.95961	0 \n \
21	Sc	54	53.96326	0 \n \
21	Sc	55	54.96824	0 \n \
21	Sc	56	55.97287	0 \n \
21	Sc	57	56.97779	0 \n \
21	Sc	58	57.98371	0 \n \
21	Sc	59	58.98922	0 \n \
21	Sc	60	59.99571	0 \n \
22	Ti	38	38.00977	0 \n \
22	Ti	39	39.00161	0 \n \
22	Ti	40	39.9905	0 \n \
22	Ti	41	40.98315	0 \n \
22	Ti	42	41.973031	0 \n \
22	Ti	43	42.968522	0 \n \
22	Ti	44	43.9596901	0 \n \
22	Ti	45	44.9581256	0 \n \
22	Ti	46	45.9526316	8.25 \n \
22	Ti	47	46.9517631	7.44 \n \
22	Ti	48	47.9479463	73.72 \n \
22	Ti	49	48.94787	5.41 \n \
22	Ti	50	49.9447912	5.18 \n \
22	Ti	51	50.946615	0 \n \
22	Ti	52	51.946897	0 \n",
    "22	Ti	53	52.94973	0 \n \
22	Ti	54	53.95105	0 \n \
22	Ti	55	54.95527	0 \n \
22	Ti	56	55.9582	0 \n \
22	Ti	57	56.96399	0 \n \
22	Ti	58	57.96697	0 \n \
22	Ti	59	58.97293	0 \n \
22	Ti	60	59.97676	0 \n \
22	Ti	61	60.9832	0 \n \
22	Ti	62	61.98749	0 \n \
22	Ti	63	62.99442	0 \n \
23	V	40	40.01109	0 \n \
23	V	41	40.99978	0 \n \
23	V	42	41.99123	0 \n \
23	V	43	42.98065	0 \n \
23	V	44	43.97411	0 \n \
23	V	45	44.965776	0 \n \
23	V	46	45.9602005	0 \n \
23	V	47	46.9549089	0 \n \
23	V	48	47.9522537	0 \n \
23	V	49	48.9485161	0 \n \
23	V	50	49.9471585	0.250 \n \
23	V	51	50.9439595	99.750 \n \
23	V	52	51.9447755	0 \n \
23	V	53	52.944338	0 \n \
23	V	54	53.94644	0 \n \
23	V	55	54.94723	0 \n \
23	V	56	55.95053	0 \n \
23	V	57	56.95256	0 \n \
23	V	58	57.95683	0 \n \
23	V	59	58.96021	0 \n \
23	V	60	59.96503	0 \n \
23	V	61	60.96848	0 \n \
23	V	62	61.97378	0 \n \
23	V	63	62.97755	0 \n \
23	V	64	63.98347	0 \n \
23	V	65	64.98792	0 \n \
24	Cr	42	42.00643	0 \n \
24	Cr	43	42.99771	0 \n \
24	Cr	44	43.98555	0 \n \
24	Cr	45	44.97964	0 \n \
24	Cr	46	45.968359	0 \n \
24	Cr	47	46.9629	0 \n \
24	Cr	48	47.954032	0 \n \
24	Cr	49	48.9513357	0 \n \
24	Cr	50	49.9460442	4.345 \n \
24	Cr	51	50.9447674	0 \n \
24	Cr	52	51.9405075	83.789 \n \
24	Cr	53	52.9406494	9.501 \n \
24	Cr	54	53.9388804	2.365 \n \
24	Cr	55	54.9408397	0 \n",
    "24	Cr	56	55.9406531	0 \n \
24	Cr	57	56.943613	0 \n \
24	Cr	58	57.94435	0 \n \
24	Cr	59	58.94859	0 \n \
24	Cr	60	59.95008	0 \n \
24	Cr	61	60.95472	0 \n \
24	Cr	62	61.95661	0 \n \
24	Cr	63	62.96186	0 \n \
24	Cr	64	63.96441	0 \n \
24	Cr	65	64.97016	0 \n \
24	Cr	66	65.97338	0 \n \
24	Cr	67	66.97955	0 \n \
25	Mn	44	44.00687	0 \n \
25	Mn	45	44.99451	0 \n \
25	Mn	46	45.98672	0 \n \
25	Mn	47	46.9761	0 \n \
25	Mn	48	47.96852	0 \n \
25	Mn	49	48.959618	0 \n \
25	Mn	50	49.9542382	0 \n \
25	Mn	51	50.9482108	0 \n \
25	Mn	52	51.9455655	0 \n \
25	Mn	53	52.9412901	0 \n \
25	Mn	54	53.9403589	0 \n \
25	Mn	55	54.9380451	100 \n \
25	Mn	56	55.9389049	0 \n \
25	Mn	57	56.9382854	0 \n \
25	Mn	58	57.93998	0 \n \
25	Mn	59	58.94044	0 \n \
25	Mn	60	59.94291	0 \n \
25	Mn	61	60.94465	0 \n \
25	Mn	62	61.94843	0 \n \
25	Mn	63	62.95024	0 \n \
25	Mn	64	63.95425	0 \n \
25	Mn	65	64.95634	0 \n \
25	Mn	66	65.96108	0 \n \
25	Mn	67	66.96414	0 \n \
25	Mn	68	67.9693	0 \n \
25	Mn	69	68.97284	0 \n \
26	Fe	45	45.01458	0 \n \
26	Fe	46	46.00081	0 \n \
26	Fe	47	46.99289	0 \n \
26	Fe	48	47.9805	0 \n \
26	Fe	49	48.97361	0 \n \
26	Fe	50	49.96299	0 \n \
26	Fe	51	50.95682	0 \n \
26	Fe	52	51.948114	0 \n \
26	Fe	53	52.9453079	0 \n \
26	Fe	54	53.9396105	5.845 \n \
26	Fe	55	54.9382934	0 \n \
26	Fe	56	55.9349375	91.754 \n \
26	Fe	57	56.935394	2.119 \n",
    "26	Fe	58	57.9332756	0.282 \n \
26	Fe	59	58.9348755	0 \n \
26	Fe	60	59.934072	0 \n \
26	Fe	61	60.936745	0 \n \
26	Fe	62	61.936767	0 \n \
26	Fe	63	62.94037	0 \n \
26	Fe	64	63.9412	0 \n \
26	Fe	65	64.94538	0 \n \
26	Fe	66	65.94678	0 \n \
26	Fe	67	66.95095	0 \n \
26	Fe	68	67.9537	0 \n \
26	Fe	69	68.95878	0 \n \
26	Fe	70	69.96146	0 \n \
26	Fe	71	70.96672	0 \n \
26	Fe	72	71.96962	0 \n \
27	Co	47	47.01149	0 \n \
27	Co	48	48.00176	0 \n \
27	Co	49	48.98972	0 \n \
27	Co	50	49.98154	0 \n \
27	Co	51	50.97072	0 \n \
27	Co	52	51.96359	0 \n \
27	Co	53	52.954219	0 \n \
27	Co	54	53.9484596	0 \n \
27	Co	55	54.941999	0 \n \
27	Co	56	55.9398393	0 \n \
27	Co	57	56.9362914	0 \n \
27	Co	58	57.9357528	0 \n \
27	Co	59	58.933195	100 \n \
27	Co	60	59.9338171	0 \n \
27	Co	61	60.9324758	0 \n \
27	Co	62	61.934051	0 \n \
27	Co	63	62.933612	0 \n \
27	Co	64	63.93581	0 \n \
27	Co	65	64.936478	0 \n \
27	Co	66	65.93976	0 \n \
27	Co	67	66.94089	0 \n \
27	Co	68	67.94487	0 \n \
27	Co	69	68.94632	0 \n \
27	Co	70	69.951	0 \n \
27	Co	71	70.9529	0 \n \
27	Co	72	71.95781	0 \n \
27	Co	73	72.96024	0 \n \
27	Co	74	73.96538	0 \n \
27	Co	75	74.96833	0 \n \
28	Ni	48	48.01975	0 \n \
28	Ni	49	49.00966	0 \n \
28	Ni	50	49.99593	0 \n \
28	Ni	51	50.98772	0 \n \
28	Ni	52	51.97568	0 \n \
28	Ni	53	52.96847	0 \n \
28	Ni	54	53.95791	0 \n",
    "28	Ni	55	54.95133	0 \n \
28	Ni	56	55.942132	0 \n \
28	Ni	57	56.9397935	0 \n \
28	Ni	58	57.9353429	68.0769 \n \
28	Ni	59	58.9343467	0 \n \
28	Ni	60	59.9307864	26.2231 \n \
28	Ni	61	60.931056	1.1399 \n \
28	Ni	62	61.9283451	3.6345 \n \
28	Ni	63	62.9296694	0 \n \
28	Ni	64	63.927966	0.9256 \n \
28	Ni	65	64.9300843	0 \n \
28	Ni	66	65.9291393	0 \n \
28	Ni	67	66.931569	0 \n \
28	Ni	68	67.931869	0 \n \
28	Ni	69	68.93561	0 \n \
28	Ni	70	69.9365	0 \n \
28	Ni	71	70.94074	0 \n \
28	Ni	72	71.94209	0 \n \
28	Ni	73	72.94647	0 \n \
28	Ni	74	73.94807	0 \n \
28	Ni	75	74.95287	0 \n \
28	Ni	76	75.95533	0 \n \
28	Ni	77	76.96055	0 \n \
28	Ni	78	77.96318	0 \n \
29	Cu	52	51.99718	0 \n \
29	Cu	53	52.98555	0 \n \
29	Cu	54	53.97671	0 \n \
29	Cu	55	54.96605	0 \n \
29	Cu	56	55.95856	0 \n \
29	Cu	57	56.949211	0 \n \
29	Cu	58	57.9445385	0 \n \
29	Cu	59	58.939498	0 \n \
29	Cu	60	59.937365	0 \n \
29	Cu	61	60.9334578	0 \n \
29	Cu	62	61.932584	0 \n \
29	Cu	63	62.9295975	69.17 \n \
29	Cu	64	63.9297642	0 \n \
29	Cu	65	64.9277895	30.83 \n \
29	Cu	66	65.9288688	0 \n \
29	Cu	67	66.9277303	0 \n \
29	Cu	68	67.9296109	0 \n \
29	Cu	69	68.9294293	0 \n \
29	Cu	70	69.9323923	0 \n \
29	Cu	71	70.9326768	0 \n \
29	Cu	72	71.9358203	0 \n \
29	Cu	73	72.936675	0 \n \
29	Cu	74	73.939875	0 \n \
29	Cu	75	74.9419	0 \n \
29	Cu	76	75.945275	0 \n \
29	Cu	77	76.94785	0 \n \
29	Cu	78	77.95196	0 \n",
    "29	Cu	79	78.95456	0 \n \
29	Cu	80	79.96087	0 \n \
30	Zn	54	53.99295	0 \n \
30	Zn	55	54.98398	0 \n \
30	Zn	56	55.97238	0 \n \
30	Zn	57	56.96479	0 \n \
30	Zn	58	57.95459	0 \n \
30	Zn	59	58.94926	0 \n \
30	Zn	60	59.941827	0 \n \
30	Zn	61	60.939511	0 \n \
30	Zn	62	61.93433	0 \n \
30	Zn	63	62.9332116	0 \n \
30	Zn	64	63.9291422	48.63 \n \
30	Zn	65	64.929241	0 \n \
30	Zn	66	65.9260334	27.90 \n \
30	Zn	67	66.9271273	4.10 \n \
30	Zn	68	67.9248442	18.75 \n \
30	Zn	69	68.9265503	0 \n \
30	Zn	70	69.9253193	0.62 \n \
30	Zn	71	70.927722	0 \n \
30	Zn	72	71.926858	0 \n \
30	Zn	73	72.92978	0 \n \
30	Zn	74	73.92946	0 \n \
30	Zn	75	74.93294	0 \n \
30	Zn	76	75.93329	0 \n \
30	Zn	77	76.93696	0 \n \
30	Zn	78	77.93844	0 \n \
30	Zn	79	78.94265	0 \n \
30	Zn	80	79.94434	0 \n \
30	Zn	81	80.95048	0 \n \
30	Zn	82	81.95442	0 \n \
30	Zn	83	82.96103	0 \n \
31	Ga	56	55.99491	0 \n \
31	Ga	57	56.98293	0 \n \
31	Ga	58	57.97425	0 \n \
31	Ga	59	58.96337	0 \n \
31	Ga	60	59.95706	0 \n \
31	Ga	61	60.94945	0 \n \
31	Ga	62	61.944175	0 \n \
31	Ga	63	62.9392942	0 \n \
31	Ga	64	63.9368387	0 \n \
31	Ga	65	64.9327348	0 \n \
31	Ga	66	65.931589	0 \n \
31	Ga	67	66.9282017	0 \n \
31	Ga	68	67.9279801	0 \n \
31	Ga	69	68.9255736	60.108 \n \
31	Ga	70	69.926022	0 \n \
31	Ga	71	70.9247013	39.892 \n \
31	Ga	72	71.9263663	0 \n \
31	Ga	73	72.9251747	0 \n \
31	Ga	74	73.926946	0 \n",
    "31	Ga	75	74.9265002	0 \n \
31	Ga	76	75.9288276	0 \n \
31	Ga	77	76.9291543	0 \n \
31	Ga	78	77.9316082	0 \n \
31	Ga	79	78.93289	0 \n \
31	Ga	80	79.93652	0 \n \
31	Ga	81	80.93775	0 \n \
31	Ga	82	81.94299	0 \n \
31	Ga	83	82.94698	0 \n \
31	Ga	84	83.95265	0 \n \
31	Ga	85	84.957	0 \n \
31	Ga	86	85.96312	0 \n \
32	Ge	58	57.99101	0 \n \
32	Ge	59	58.98175	0 \n \
32	Ge	60	59.97019	0 \n \
32	Ge	61	60.96379	0 \n \
32	Ge	62	61.95465	0 \n \
32	Ge	63	62.94964	0 \n \
32	Ge	64	63.94165	0 \n \
32	Ge	65	64.93944	0 \n \
32	Ge	66	65.93384	0 \n \
32	Ge	67	66.932734	0 \n \
32	Ge	68	67.928094	0 \n \
32	Ge	69	68.9279645	0 \n \
32	Ge	70	69.9242474	20.84 \n \
32	Ge	71	70.924951	0 \n \
32	Ge	72	71.9220758	27.54 \n \
32	Ge	73	72.9234589	7.73 \n \
32	Ge	74	73.9211778	36.28 \n \
32	Ge	75	74.9228589	0 \n \
32	Ge	76	75.9214026	7.61 \n \
32	Ge	77	76.9235486	0 \n \
32	Ge	78	77.922853	0 \n \
32	Ge	79	78.9254	0 \n \
32	Ge	80	79.92537	0 \n \
32	Ge	81	80.92882	0 \n \
32	Ge	82	81.92955	0 \n \
32	Ge	83	82.93462	0 \n \
32	Ge	84	83.93747	0 \n \
32	Ge	85	84.94303	0 \n \
32	Ge	86	85.94649	0 \n \
32	Ge	87	86.95251	0 \n \
32	Ge	88	87.95691	0 \n \
32	Ge	89	88.96383	0 \n \
33	As	60	59.99313	0 \n \
33	As	61	60.98062	0 \n \
33	As	62	61.9732	0 \n \
33	As	63	62.96369	0 \n \
33	As	64	63.95757	0 \n \
33	As	65	64.94956	0 \n \
33	As	66	65.94471	0 \n",
    "33	As	67	66.93919	0 \n \
33	As	68	67.93677	0 \n \
33	As	69	68.93227	0 \n \
33	As	70	69.93092	0 \n \
33	As	71	70.927112	0 \n \
33	As	72	71.926752	0 \n \
33	As	73	72.923825	0 \n \
33	As	74	73.9239287	0 \n \
33	As	75	74.9215965	100 \n \
33	As	76	75.922394	0 \n \
33	As	77	76.9206473	0 \n \
33	As	78	77.921827	0 \n \
33	As	79	78.920948	0 \n \
33	As	80	79.922534	0 \n \
33	As	81	80.922132	0 \n \
33	As	82	81.9245	0 \n \
33	As	83	82.92498	0 \n \
33	As	84	83.92906	0 \n \
33	As	85	84.93202	0 \n \
33	As	86	85.9365	0 \n \
33	As	87	86.9399	0 \n \
33	As	88	87.94494	0 \n \
33	As	89	88.94939	0 \n \
33	As	90	89.9555	0 \n \
33	As	91	90.96043	0 \n \
33	As	92	91.9668	0 \n \
34	Se	65	64.96466	0 \n \
34	Se	66	65.95521	0 \n \
34	Se	67	66.95009	0 \n \
34	Se	68	67.9418	0 \n \
34	Se	69	68.93956	0 \n \
34	Se	70	69.93339	0 \n \
34	Se	71	70.93224	0 \n \
34	Se	72	71.927112	0 \n \
34	Se	73	72.926765	0 \n \
34	Se	74	73.9224764	0.89 \n \
34	Se	75	74.9225234	0 \n \
34	Se	76	75.9192136	9.37 \n \
34	Se	77	76.919914	7.63 \n \
34	Se	78	77.9173091	23.77 \n \
34	Se	79	78.9184991	0 \n \
34	Se	80	79.9165213	49.61 \n \
34	Se	81	80.9179925	0 \n \
34	Se	82	81.9166994	8.73 \n \
34	Se	83	82.919118	0 \n \
34	Se	84	83.918462	0 \n \
34	Se	85	84.92225	0 \n \
34	Se	86	85.924272	0 \n \
34	Se	87	86.92852	0 \n \
34	Se	88	87.93142	0 \n \
34	Se	89	88.93645	0 \n",
    "34	Se	90	89.93996	0 \n \
34	Se	91	90.94596	0 \n \
34	Se	92	91.94992	0 \n \
34	Se	93	92.95629	0 \n \
34	Se	94	93.96049	0 \n \
35	Br	67	66.96479	0 \n \
35	Br	68	67.95852	0 \n \
35	Br	69	68.95011	0 \n \
35	Br	70	69.94479	0 \n \
35	Br	71	70.93874	0 \n \
35	Br	72	71.93664	0 \n \
35	Br	73	72.93169	0 \n \
35	Br	74	73.929891	0 \n \
35	Br	75	74.925776	0 \n \
35	Br	76	75.924541	0 \n \
35	Br	77	76.921379	0 \n \
35	Br	78	77.921146	0 \n \
35	Br	79	78.9183371	50.69 \n \
35	Br	80	79.9185293	0 \n \
35	Br	81	80.9162906	49.31 \n \
35	Br	82	81.9168041	0 \n \
35	Br	83	82.91518	0 \n \
35	Br	84	83.916479	0 \n \
35	Br	85	84.915608	0 \n \
35	Br	86	85.918798	0 \n \
35	Br	87	86.920711	0 \n \
35	Br	88	87.92407	0 \n \
35	Br	89	88.92639	0 \n \
35	Br	90	89.93063	0 \n \
35	Br	91	90.93397	0 \n \
35	Br	92	91.93926	0 \n \
35	Br	93	92.94305	0 \n \
35	Br	94	93.94868	0 \n \
35	Br	95	94.95287	0 \n \
35	Br	96	95.95853	0 \n \
35	Br	97	96.9628	0 \n \
36	Kr	69	68.96518	0 \n \
36	Kr	70	69.95526	0 \n \
36	Kr	71	70.94963	0 \n \
36	Kr	72	71.942092	0 \n \
36	Kr	73	72.939289	0 \n \
36	Kr	74	73.9330844	0 \n \
36	Kr	75	74.930946	0 \n \
36	Kr	76	75.92591	0 \n \
36	Kr	77	76.92467	0 \n \
36	Kr	78	77.9203648	0.35 \n \
36	Kr	79	78.920082	0 \n \
36	Kr	80	79.916379	2.28 \n \
36	Kr	81	80.916592	0 \n \
36	Kr	82	81.9134836	11.58 \n \
36	Kr	83	82.914136	11.49 \n",
    "36	Kr	84	83.911507	57.00 \n \
36	Kr	85	84.9125273	0 \n \
36	Kr	86	85.91061073	17.30 \n \
36	Kr	87	86.91335486	0 \n \
36	Kr	88	87.914447	0 \n \
36	Kr	89	88.91763	0 \n \
36	Kr	90	89.919517	0 \n \
36	Kr	91	90.92345	0 \n \
36	Kr	92	91.926156	0 \n \
36	Kr	93	92.93127	0 \n \
36	Kr	94	93.93436	0 \n \
36	Kr	95	94.93984	0 \n \
36	Kr	96	95.94307	0 \n \
36	Kr	97	96.94856	0 \n \
36	Kr	98	97.95191	0 \n \
36	Kr	99	98.9576	0 \n \
36	Kr	100	99.96114	0 \n \
37	Rb	71	70.96532	0 \n \
37	Rb	72	71.95908	0 \n \
37	Rb	73	72.95056	0 \n \
37	Rb	74	73.944265	0 \n \
37	Rb	75	74.93857	0 \n \
37	Rb	76	75.9350722	0 \n \
37	Rb	77	76.930408	0 \n \
37	Rb	78	77.928141	0 \n \
37	Rb	79	78.923989	0 \n \
37	Rb	80	79.922519	0 \n \
37	Rb	81	80.918996	0 \n \
37	Rb	82	81.9182086	0 \n \
37	Rb	83	82.91511	0 \n \
37	Rb	84	83.914385	0 \n \
37	Rb	85	84.91178974	72.17 \n \
37	Rb	86	85.91116742	0 \n \
37	Rb	87	86.90918053	27.83 \n \
37	Rb	88	87.91131559	0 \n \
37	Rb	89	88.912278	0 \n \
37	Rb	90	89.914802	0 \n \
37	Rb	91	90.916537	0 \n \
37	Rb	92	91.919729	0 \n \
37	Rb	93	92.922042	0 \n \
37	Rb	94	93.926405	0 \n \
37	Rb	95	94.929303	0 \n \
37	Rb	96	95.93427	0 \n \
37	Rb	97	96.93735	0 \n \
37	Rb	98	97.94179	0 \n \
37	Rb	99	98.94538	0 \n \
37	Rb	100	99.94987	0 \n \
37	Rb	101	100.9532	0 \n \
37	Rb	102	101.95887	0 \n \
38	Sr	73	72.96597	0 \n \
38	Sr	74	73.95631	0 \n",
    "38	Sr	75	74.94995	0 \n \
38	Sr	76	75.94177	0 \n \
38	Sr	77	76.937945	0 \n \
38	Sr	78	77.93218	0 \n \
38	Sr	79	78.929708	0 \n \
38	Sr	80	79.924521	0 \n \
38	Sr	81	80.923212	0 \n \
38	Sr	82	81.918402	0 \n \
38	Sr	83	82.917557	0 \n \
38	Sr	84	83.913425	0.56 \n \
38	Sr	85	84.912933	0 \n \
38	Sr	86	85.9092602	9.86 \n \
38	Sr	87	86.9088771	7.00 \n \
38	Sr	88	87.9056121	82.58 \n \
38	Sr	89	88.9074507	0 \n \
38	Sr	90	89.907738	0 \n \
38	Sr	91	90.910203	0 \n \
38	Sr	92	91.911038	0 \n \
38	Sr	93	92.914026	0 \n \
38	Sr	94	93.915361	0 \n \
38	Sr	95	94.919359	0 \n \
38	Sr	96	95.921697	0 \n \
38	Sr	97	96.926153	0 \n \
38	Sr	98	97.928453	0 \n \
38	Sr	99	98.93324	0 \n \
38	Sr	100	99.93535	0 \n \
38	Sr	101	100.94052	0 \n \
38	Sr	102	101.94302	0 \n \
38	Sr	103	102.94895	0 \n \
38	Sr	104	103.95233	0 \n \
38	Sr	105	104.95858	0 \n \
39	Y	76	75.95845	0 \n \
39	Y	77	76.94965	0 \n \
39	Y	78	77.94361	0 \n \
39	Y	79	78.93735	0 \n \
39	Y	80	79.93428	0 \n \
39	Y	81	80.92913	0 \n \
39	Y	82	81.92679	0 \n \
39	Y	83	82.92235	0 \n \
39	Y	84	83.92039	0 \n \
39	Y	85	84.916433	0 \n \
39	Y	86	85.914886	0 \n \
39	Y	87	86.9108757	0 \n \
39	Y	88	87.9095011	0 \n \
39	Y	89	88.9058483	100 \n \
39	Y	90	89.9071519	0 \n \
39	Y	91	90.907305	0 \n \
39	Y	92	91.908949	0 \n \
39	Y	93	92.909583	0 \n \
39	Y	94	93.911595	0 \n \
39	Y	95	94.912821	0 \n",
    "39	Y	96	95.915891	0 \n \
39	Y	97	96.918134	0 \n \
39	Y	98	97.922203	0 \n \
39	Y	99	98.924636	0 \n \
39	Y	100	99.92776	0 \n \
39	Y	101	100.93031	0 \n \
39	Y	102	101.93356	0 \n \
39	Y	103	102.93673	0 \n \
39	Y	104	103.94105	0 \n \
39	Y	105	104.94487	0 \n \
39	Y	106	105.94979	0 \n \
39	Y	107	106.95414	0 \n \
39	Y	108	107.95948	0 \n \
40	Zr	78	77.95523	0 \n \
40	Zr	79	78.94916	0 \n \
40	Zr	80	79.9404	0 \n \
40	Zr	81	80.93721	0 \n \
40	Zr	82	81.93109	0 \n \
40	Zr	83	82.92865	0 \n \
40	Zr	84	83.92325	0 \n \
40	Zr	85	84.92147	0 \n \
40	Zr	86	85.91647	0 \n \
40	Zr	87	86.914816	0 \n \
40	Zr	88	87.910227	0 \n \
40	Zr	89	88.90889	0 \n \
40	Zr	90	89.9047044	51.45 \n \
40	Zr	91	90.9056458	11.22 \n \
40	Zr	92	91.9050408	17.15 \n \
40	Zr	93	92.906476	0 \n \
40	Zr	94	93.9063152	17.38 \n \
40	Zr	95	94.9080426	0 \n \
40	Zr	96	95.9082734	2.80 \n \
40	Zr	97	96.9109531	0 \n \
40	Zr	98	97.912735	0 \n \
40	Zr	99	98.916512	0 \n \
40	Zr	100	99.91776	0 \n \
40	Zr	101	100.92114	0 \n \
40	Zr	102	101.92298	0 \n \
40	Zr	103	102.9266	0 \n \
40	Zr	104	103.92878	0 \n \
40	Zr	105	104.93305	0 \n \
40	Zr	106	105.93591	0 \n \
40	Zr	107	106.94075	0 \n \
40	Zr	108	107.94396	0 \n \
40	Zr	109	108.94924	0 \n \
40	Zr	110	109.95287	0 \n \
41	Nb	81	80.94903	0 \n \
41	Nb	82	81.94313	0 \n \
41	Nb	83	82.93671	0 \n \
41	Nb	84	83.93357	0 \n \
41	Nb	85	84.92791	0 \n",
    "41	Nb	86	85.92504	0 \n \
41	Nb	87	86.92036	0 \n \
41	Nb	88	87.91833	0 \n \
41	Nb	89	88.913418	0 \n \
41	Nb	90	89.911265	0 \n \
41	Nb	91	90.906996	0 \n \
41	Nb	92	91.907194	0 \n \
41	Nb	93	92.9063781	100 \n \
41	Nb	94	93.9072839	0 \n \
41	Nb	95	94.9068358	0 \n \
41	Nb	96	95.908101	0 \n \
41	Nb	97	96.9080986	0 \n \
41	Nb	98	97.910328	0 \n \
41	Nb	99	98.911618	0 \n \
41	Nb	100	99.914182	0 \n \
41	Nb	101	100.915252	0 \n \
41	Nb	102	101.91804	0 \n \
41	Nb	103	102.91914	0 \n \
41	Nb	104	103.92246	0 \n \
41	Nb	105	104.92394	0 \n \
41	Nb	106	105.92797	0 \n \
41	Nb	107	106.93031	0 \n \
41	Nb	108	107.93484	0 \n \
41	Nb	109	108.93763	0 \n \
41	Nb	110	109.94244	0 \n \
41	Nb	111	110.94565	0 \n \
41	Nb	112	111.95083	0 \n \
41	Nb	113	112.9547	0 \n \
42	Mo	83	82.94874	0 \n \
42	Mo	84	83.94009	0 \n \
42	Mo	85	84.93655	0 \n \
42	Mo	86	85.9307	0 \n \
42	Mo	87	86.92733	0 \n \
42	Mo	88	87.921953	0 \n \
42	Mo	89	88.91948	0 \n \
42	Mo	90	89.913937	0 \n \
42	Mo	91	90.91175	0 \n \
42	Mo	92	91.906811	14.84 \n \
42	Mo	93	92.906813	0 \n \
42	Mo	94	93.9050883	9.25 \n \
42	Mo	95	94.9058421	15.92 \n \
42	Mo	96	95.9046795	16.68 \n \
42	Mo	97	96.9060215	9.55 \n \
42	Mo	98	97.9054082	24.13 \n \
42	Mo	99	98.9077119	0 \n \
42	Mo	100	99.907477	9.63 \n \
42	Mo	101	100.910347	0 \n \
42	Mo	102	101.910297	0 \n \
42	Mo	103	102.91321	0 \n \
42	Mo	104	103.91376	0 \n \
42	Mo	105	104.91697	0 \n",
    "42	Mo	106	105.918137	0 \n \
42	Mo	107	106.92169	0 \n \
42	Mo	108	107.92345	0 \n \
42	Mo	109	108.92781	0 \n \
42	Mo	110	109.92973	0 \n \
42	Mo	111	110.93441	0 \n \
42	Mo	112	111.93684	0 \n \
42	Mo	113	112.94188	0 \n \
42	Mo	114	113.94492	0 \n \
42	Mo	115	114.95029	0 \n \
43	Tc	85	84.94883	0 \n \
43	Tc	86	85.94288	0 \n \
43	Tc	87	86.93653	0 \n \
43	Tc	88	87.93268	0 \n \
43	Tc	89	88.92717	0 \n \
43	Tc	90	89.92356	0 \n \
43	Tc	91	90.91843	0 \n \
43	Tc	92	91.91526	0 \n \
43	Tc	93	92.910249	0 \n \
43	Tc	94	93.909657	0 \n \
43	Tc	95	94.907657	0 \n \
43	Tc	96	95.907871	0 \n \
43	Tc	97	96.906365	0 \n \
43	Tc	98	97.907216	0 \n \
43	Tc	99	98.9062547	0 \n \
43	Tc	100	99.9076578	0 \n \
43	Tc	101	100.907315	0 \n \
43	Tc	102	101.909215	0 \n \
43	Tc	103	102.909181	0 \n \
43	Tc	104	103.91145	0 \n \
43	Tc	105	104.91166	0 \n \
43	Tc	106	105.914358	0 \n \
43	Tc	107	106.91508	0 \n \
43	Tc	108	107.91846	0 \n \
43	Tc	109	108.91998	0 \n \
43	Tc	110	109.92382	0 \n \
43	Tc	111	110.92569	0 \n \
43	Tc	112	111.92915	0 \n \
43	Tc	113	112.93159	0 \n \
43	Tc	114	113.93588	0 \n \
43	Tc	115	114.93869	0 \n \
43	Tc	116	115.94337	0 \n \
43	Tc	117	116.94648	0 \n \
43	Tc	118	117.95148	0 \n \
44	Ru	87	86.94918	0 \n \
44	Ru	88	87.94026	0 \n \
44	Ru	89	88.93611	0 \n \
44	Ru	90	89.92989	0 \n \
44	Ru	91	90.92629	0 \n \
44	Ru	92	91.92012	0 \n \
44	Ru	93	92.91705	0 \n",
    "44	Ru	94	93.91136	0 \n \
44	Ru	95	94.910413	0 \n \
44	Ru	96	95.907598	5.54 \n \
44	Ru	97	96.907555	0 \n \
44	Ru	98	97.905287	1.87 \n \
44	Ru	99	98.9059393	12.76 \n \
44	Ru	100	99.9042195	12.60 \n \
44	Ru	101	100.9055821	17.06 \n \
44	Ru	102	101.9043493	31.55 \n \
44	Ru	103	102.9063238	0 \n \
44	Ru	104	103.905433	18.62 \n \
44	Ru	105	104.907753	0 \n \
44	Ru	106	105.907329	0 \n \
44	Ru	107	106.90991	0 \n \
44	Ru	108	107.91017	0 \n \
44	Ru	109	108.9132	0 \n \
44	Ru	110	109.91414	0 \n \
44	Ru	111	110.9177	0 \n \
44	Ru	112	111.91897	0 \n \
44	Ru	113	112.92249	0 \n \
44	Ru	114	113.92428	0 \n \
44	Ru	115	114.92869	0 \n \
44	Ru	116	115.93081	0 \n \
44	Ru	117	116.93558	0 \n \
44	Ru	118	117.93782	0 \n \
44	Ru	119	118.94284	0 \n \
44	Ru	120	119.94531	0 \n \
45	Rh	89	88.94884	0 \n \
45	Rh	90	89.94287	0 \n \
45	Rh	91	90.93655	0 \n \
45	Rh	92	91.93198	0 \n \
45	Rh	93	92.92574	0 \n \
45	Rh	94	93.9217	0 \n \
45	Rh	95	94.9159	0 \n \
45	Rh	96	95.914461	0 \n \
45	Rh	97	96.91134	0 \n \
45	Rh	98	97.910708	0 \n \
45	Rh	99	98.908132	0 \n \
45	Rh	100	99.908122	0 \n \
45	Rh	101	100.906164	0 \n \
45	Rh	102	101.906843	0 \n \
45	Rh	103	102.905504	100 \n \
45	Rh	104	103.906656	0 \n \
45	Rh	105	104.905694	0 \n \
45	Rh	106	105.907287	0 \n \
45	Rh	107	106.906748	0 \n \
45	Rh	108	107.90873	0 \n \
45	Rh	109	108.908737	0 \n \
45	Rh	110	109.91114	0 \n \
45	Rh	111	110.91159	0 \n \
45	Rh	112	111.91439	0 \n",
    "45	Rh	113	112.91553	0 \n \
45	Rh	114	113.91881	0 \n \
45	Rh	115	114.92033	0 \n \
45	Rh	116	115.92406	0 \n \
45	Rh	117	116.92598	0 \n \
45	Rh	118	117.93007	0 \n \
45	Rh	119	118.93211	0 \n \
45	Rh	120	119.93641	0 \n \
45	Rh	121	120.93872	0 \n \
45	Rh	122	121.94321	0 \n \
46	Pd	91	90.94911	0 \n \
46	Pd	92	91.94042	0 \n \
46	Pd	93	92.93591	0 \n \
46	Pd	94	93.92877	0 \n \
46	Pd	95	94.92469	0 \n \
46	Pd	96	95.91816	0 \n \
46	Pd	97	96.91648	0 \n \
46	Pd	98	97.912721	0 \n \
46	Pd	99	98.911768	0 \n \
46	Pd	100	99.908506	0 \n \
46	Pd	101	100.908289	0 \n \
46	Pd	102	101.905609	1.02 \n \
46	Pd	103	102.906087	0 \n \
46	Pd	104	103.904036	11.14 \n \
46	Pd	105	104.905085	22.33 \n \
46	Pd	106	105.903486	27.33 \n \
46	Pd	107	106.905133	0 \n \
46	Pd	108	107.903892	26.46 \n \
46	Pd	109	108.90595	0 \n \
46	Pd	110	109.905153	11.72 \n \
46	Pd	111	110.907671	0 \n \
46	Pd	112	111.907314	0 \n \
46	Pd	113	112.91015	0 \n \
46	Pd	114	113.910363	0 \n \
46	Pd	115	114.91368	0 \n \
46	Pd	116	115.91416	0 \n \
46	Pd	117	116.91784	0 \n \
46	Pd	118	117.91898	0 \n \
46	Pd	119	118.92311	0 \n \
46	Pd	120	119.92469	0 \n \
46	Pd	121	120.92887	0 \n \
46	Pd	122	121.93055	0 \n \
46	Pd	123	122.93493	0 \n \
46	Pd	124	123.93688	0 \n \
47	Ag	93	92.94978	0 \n \
47	Ag	94	93.94278	0 \n \
47	Ag	95	94.93548	0 \n \
47	Ag	96	95.93068	0 \n \
47	Ag	97	96.92397	0 \n \
47	Ag	98	97.92157	0 \n \
47	Ag	99	98.9176	0 \n",
    "47	Ag	100	99.9161	0 \n \
47	Ag	101	100.9128	0 \n \
47	Ag	102	101.91169	0 \n \
47	Ag	103	102.908973	0 \n \
47	Ag	104	103.908629	0 \n \
47	Ag	105	104.906529	0 \n \
47	Ag	106	105.906669	0 \n \
47	Ag	107	106.905097	51.839 \n \
47	Ag	108	107.905956	0 \n \
47	Ag	109	108.904752	48.161 \n \
47	Ag	110	109.906107	0 \n \
47	Ag	111	110.905291	0 \n \
47	Ag	112	111.907005	0 \n \
47	Ag	113	112.906567	0 \n \
47	Ag	114	113.908804	0 \n \
47	Ag	115	114.90876	0 \n \
47	Ag	116	115.91136	0 \n \
47	Ag	117	116.91168	0 \n \
47	Ag	118	117.91458	0 \n \
47	Ag	119	118.91567	0 \n \
47	Ag	120	119.91879	0 \n \
47	Ag	121	120.91985	0 \n \
47	Ag	122	121.92353	0 \n \
47	Ag	123	122.9249	0 \n \
47	Ag	124	123.92864	0 \n \
47	Ag	125	124.93043	0 \n \
47	Ag	126	125.9345	0 \n \
47	Ag	127	126.93677	0 \n \
47	Ag	128	127.94117	0 \n \
47	Ag	129	128.94369	0 \n \
47	Ag	130	129.95045	0 \n \
48	Cd	95	94.94987	0 \n \
48	Cd	96	95.93977	0 \n \
48	Cd	97	96.93494	0 \n \
48	Cd	98	97.9274	0 \n \
48	Cd	99	98.92501	0 \n \
48	Cd	100	99.92029	0 \n \
48	Cd	101	100.91868	0 \n \
48	Cd	102	101.91446	0 \n \
48	Cd	103	102.913419	0 \n \
48	Cd	104	103.909849	0 \n \
48	Cd	105	104.909468	0 \n \
48	Cd	106	105.906459	1.25 \n \
48	Cd	107	106.906618	0 \n \
48	Cd	108	107.904184	0.89 \n \
48	Cd	109	108.904982	0 \n \
48	Cd	110	109.9030021	12.49 \n \
48	Cd	111	110.9041781	12.80 \n \
48	Cd	112	111.9027578	24.13 \n \
48	Cd	113	112.9044017	12.22 \n \
48	Cd	114	113.9033585	28.73 \n",
    "48	Cd	115	114.905431	0 \n \
48	Cd	116	115.904756	7.49 \n \
48	Cd	117	116.907219	0 \n \
48	Cd	118	117.906915	0 \n \
48	Cd	119	118.90992	0 \n \
48	Cd	120	119.90985	0 \n \
48	Cd	121	120.91298	0 \n \
48	Cd	122	121.91333	0 \n \
48	Cd	123	122.917	0 \n \
48	Cd	124	123.91765	0 \n \
48	Cd	125	124.92125	0 \n \
48	Cd	126	125.92235	0 \n \
48	Cd	127	126.92644	0 \n \
48	Cd	128	127.92776	0 \n \
48	Cd	129	128.93215	0 \n \
48	Cd	130	129.9339	0 \n \
48	Cd	131	130.94067	0 \n \
48	Cd	132	131.94555	0 \n \
49	In	97	96.94954	0 \n \
49	In	98	97.94214	0 \n \
49	In	99	98.93422	0 \n \
49	In	100	99.93111	0 \n \
49	In	101	100.92634	0 \n \
49	In	102	101.92409	0 \n \
49	In	103	102.919914	0 \n \
49	In	104	103.9183	0 \n \
49	In	105	104.914674	0 \n \
49	In	106	105.913465	0 \n \
49	In	107	106.910295	0 \n \
49	In	108	107.909698	0 \n \
49	In	109	108.907151	0 \n \
49	In	110	109.907165	0 \n \
49	In	111	110.905103	0 \n \
49	In	112	111.905532	0 \n \
49	In	113	112.904058	4.29 \n \
49	In	114	113.904914	0 \n \
49	In	115	114.903878	95.71 \n \
49	In	116	115.90526	0 \n \
49	In	117	116.904514	0 \n \
49	In	118	117.906354	0 \n \
49	In	119	118.905845	0 \n \
49	In	120	119.90796	0 \n \
49	In	121	120.907846	0 \n \
49	In	122	121.91028	0 \n \
49	In	123	122.910438	0 \n \
49	In	124	123.91318	0 \n \
49	In	125	124.9136	0 \n \
49	In	126	125.91646	0 \n \
49	In	127	126.91735	0 \n \
49	In	128	127.92017	0 \n \
49	In	129	128.9217	0 \n",
    "49	In	130	129.92497	0 \n \
49	In	131	130.92685	0 \n \
49	In	132	131.93299	0 \n \
49	In	133	132.93781	0 \n \
49	In	134	133.94415	0 \n \
49	In	135	134.94933	0 \n \
50	Sn	99	98.94933	0 \n \
50	Sn	100	99.93904	0 \n \
50	Sn	101	100.93606	0 \n \
50	Sn	102	101.9303	0 \n \
50	Sn	103	102.9281	0 \n \
50	Sn	104	103.92314	0 \n \
50	Sn	105	104.92135	0 \n \
50	Sn	106	105.91688	0 \n \
50	Sn	107	106.91564	0 \n \
50	Sn	108	107.911925	0 \n \
50	Sn	109	108.911283	0 \n \
50	Sn	110	109.907843	0 \n \
50	Sn	111	110.907734	0 \n \
50	Sn	112	111.904818	0.97 \n \
50	Sn	113	112.905171	0 \n \
50	Sn	114	113.902779	0.66 \n \
50	Sn	115	114.903342	0.34 \n \
50	Sn	116	115.901741	14.54 \n \
50	Sn	117	116.902952	7.68 \n \
50	Sn	118	117.901603	24.22 \n \
50	Sn	119	118.903308	8.59 \n \
50	Sn	120	119.9021947	32.58 \n \
50	Sn	121	120.9042355	0 \n \
50	Sn	122	121.903439	4.63 \n \
50	Sn	123	122.9057208	0 \n \
50	Sn	124	123.9052739	5.79 \n \
50	Sn	125	124.9077841	0 \n \
50	Sn	126	125.907653	0 \n \
50	Sn	127	126.91036	0 \n \
50	Sn	128	127.910537	0 \n \
50	Sn	129	128.91348	0 \n \
50	Sn	130	129.913967	0 \n \
50	Sn	131	130.917	0 \n \
50	Sn	132	131.917816	0 \n \
50	Sn	133	132.92383	0 \n \
50	Sn	134	133.92829	0 \n \
50	Sn	135	134.93473	0 \n \
50	Sn	136	135.93934	0 \n \
50	Sn	137	136.94599	0 \n \
51	Sb	103	102.93969	0 \n \
51	Sb	104	103.93647	0 \n \
51	Sb	105	104.93149	0 \n \
51	Sb	106	105.92879	0 \n \
51	Sb	107	106.92415	0 \n \
51	Sb	108	107.92216	0 \n",
    "51	Sb	109	108.918132	0 \n \
51	Sb	110	109.91675	0 \n \
51	Sb	111	110.91316	0 \n \
51	Sb	112	111.912398	0 \n \
51	Sb	113	112.909372	0 \n \
51	Sb	114	113.90927	0 \n \
51	Sb	115	114.906598	0 \n \
51	Sb	116	115.906794	0 \n \
51	Sb	117	116.904836	0 \n \
51	Sb	118	117.905529	0 \n \
51	Sb	119	118.903942	0 \n \
51	Sb	120	119.905072	0 \n \
51	Sb	121	120.9038157	57.21 \n \
51	Sb	122	121.9051737	0 \n \
51	Sb	123	122.904214	42.79 \n \
51	Sb	124	123.9059357	0 \n \
51	Sb	125	124.9052538	0 \n \
51	Sb	126	125.90725	0 \n \
51	Sb	127	126.906924	0 \n \
51	Sb	128	127.909169	0 \n \
51	Sb	129	128.909148	0 \n \
51	Sb	130	129.911656	0 \n \
51	Sb	131	130.911982	0 \n \
51	Sb	132	131.914467	0 \n \
51	Sb	133	132.915252	0 \n \
51	Sb	134	133.92038	0 \n \
51	Sb	135	134.92517	0 \n \
51	Sb	136	135.93035	0 \n \
51	Sb	137	136.93531	0 \n \
51	Sb	138	137.94079	0 \n \
51	Sb	139	138.94598	0 \n \
52	Te	105	104.94364	0 \n \
52	Te	106	105.9375	0 \n \
52	Te	107	106.93501	0 \n \
52	Te	108	107.92944	0 \n \
52	Te	109	108.92742	0 \n \
52	Te	110	109.92241	0 \n \
52	Te	111	110.92111	0 \n \
52	Te	112	111.91701	0 \n \
52	Te	113	112.91589	0 \n \
52	Te	114	113.91209	0 \n \
52	Te	115	114.9119	0 \n \
52	Te	116	115.90846	0 \n \
52	Te	117	116.908645	0 \n \
52	Te	118	117.905828	0 \n \
52	Te	119	118.906404	0 \n \
52	Te	120	119.90402	0.09 \n \
52	Te	121	120.904936	0 \n \
52	Te	122	121.9030439	2.55 \n \
52	Te	123	122.90427	0.89 \n \
52	Te	124	123.9028179	4.74 \n",
    "52	Te	125	124.9044307	7.07 \n \
52	Te	126	125.9033117	18.84 \n \
52	Te	127	126.9052263	0 \n \
52	Te	128	127.9044631	31.74 \n \
52	Te	129	128.9065982	0 \n \
52	Te	130	129.9062244	34.08 \n \
52	Te	131	130.9085239	0 \n \
52	Te	132	131.908553	0 \n \
52	Te	133	132.910955	0 \n \
52	Te	134	133.911369	0 \n \
52	Te	135	134.91645	0 \n \
52	Te	136	135.9201	0 \n \
52	Te	137	136.92532	0 \n \
52	Te	138	137.92922	0 \n \
52	Te	139	138.93473	0 \n \
52	Te	140	139.93885	0 \n \
52	Te	141	140.94465	0 \n \
52	Te	142	141.94908	0 \n \
53	I	108	107.94348	0 \n \
53	I	109	108.93815	0 \n \
53	I	110	109.93524	0 \n \
53	I	111	110.93028	0 \n \
53	I	112	111.92797	0 \n \
53	I	113	112.92364	0 \n \
53	I	114	113.92185	0 \n \
53	I	115	114.91805	0 \n \
53	I	116	115.91681	0 \n \
53	I	117	116.91365	0 \n \
53	I	118	117.913074	0 \n \
53	I	119	118.91007	0 \n \
53	I	120	119.910048	0 \n \
53	I	121	120.907367	0 \n \
53	I	122	121.907589	0 \n \
53	I	123	122.905589	0 \n \
53	I	124	123.9062099	0 \n \
53	I	125	124.9046302	0 \n \
53	I	126	125.905624	0 \n \
53	I	127	126.904473	100 \n \
53	I	128	127.905809	0 \n \
53	I	129	128.904988	0 \n \
53	I	130	129.906674	0 \n \
53	I	131	130.9061246	0 \n \
53	I	132	131.907997	0 \n \
53	I	133	132.907797	0 \n \
53	I	134	133.909744	0 \n \
53	I	135	134.910048	0 \n \
53	I	136	135.91465	0 \n \
53	I	137	136.917871	0 \n \
53	I	138	137.92235	0 \n \
53	I	139	138.9261	0 \n \
53	I	140	139.931	0 \n",
    "53	I	141	140.93503	0 \n \
53	I	142	141.94018	0 \n \
53	I	143	142.94456	0 \n \
53	I	144	143.94999	0 \n \
54	Xe	110	109.94428	0 \n \
54	Xe	111	110.9416	0 \n \
54	Xe	112	111.93562	0 \n \
54	Xe	113	112.93334	0 \n \
54	Xe	114	113.92798	0 \n \
54	Xe	115	114.926294	0 \n \
54	Xe	116	115.921581	0 \n \
54	Xe	117	116.920359	0 \n \
54	Xe	118	117.916179	0 \n \
54	Xe	119	118.915411	0 \n \
54	Xe	120	119.911784	0 \n \
54	Xe	121	120.911462	0 \n \
54	Xe	122	121.908368	0 \n \
54	Xe	123	122.908482	0 \n \
54	Xe	124	123.905893	0.09 \n \
54	Xe	125	124.9063955	0 \n \
54	Xe	126	125.904274	0.09 \n \
54	Xe	127	126.905184	0 \n \
54	Xe	128	127.9035313	1.92 \n \
54	Xe	129	128.9047794	26.44 \n \
54	Xe	130	129.903508	4.08 \n \
54	Xe	131	130.9050824	21.18 \n \
54	Xe	132	131.9041535	26.89 \n \
54	Xe	133	132.9059107	0 \n \
54	Xe	134	133.9053945	10.44 \n \
54	Xe	135	134.907227	0 \n \
54	Xe	136	135.907219	8.87 \n \
54	Xe	137	136.911562	0 \n \
54	Xe	138	137.91395	0 \n \
54	Xe	139	138.918793	0 \n \
54	Xe	140	139.92164	0 \n \
54	Xe	141	140.92665	0 \n \
54	Xe	142	141.92971	0 \n \
54	Xe	143	142.93511	0 \n \
54	Xe	144	143.93851	0 \n \
54	Xe	145	144.94407	0 \n \
54	Xe	146	145.94775	0 \n \
54	Xe	147	146.95356	0 \n \
55	Cs	112	111.9503	0 \n \
55	Cs	113	112.94449	0 \n \
55	Cs	114	113.94145	0 \n \
55	Cs	115	114.93591	0 \n \
55	Cs	116	115.93337	0 \n \
55	Cs	117	116.92867	0 \n \
55	Cs	118	117.926559	0 \n \
55	Cs	119	118.922377	0 \n \
55	Cs	120	119.920677	0 \n",
    "55	Cs	121	120.917229	0 \n \
55	Cs	122	121.91611	0 \n \
55	Cs	123	122.912996	0 \n \
55	Cs	124	123.912258	0 \n \
55	Cs	125	124.909728	0 \n \
55	Cs	126	125.909452	0 \n \
55	Cs	127	126.907418	0 \n \
55	Cs	128	127.907749	0 \n \
55	Cs	129	128.906064	0 \n \
55	Cs	130	129.906709	0 \n \
55	Cs	131	130.905464	0 \n \
55	Cs	132	131.9064343	0 \n \
55	Cs	133	132.9054519	100 \n \
55	Cs	134	133.9067185	0 \n \
55	Cs	135	134.905977	0 \n \
55	Cs	136	135.9073116	0 \n \
55	Cs	137	136.9070895	0 \n \
55	Cs	138	137.911017	0 \n \
55	Cs	139	138.913364	0 \n \
55	Cs	140	139.917282	0 \n \
55	Cs	141	140.920046	0 \n \
55	Cs	142	141.924299	0 \n \
55	Cs	143	142.927352	0 \n \
55	Cs	144	143.932077	0 \n \
55	Cs	145	144.935526	0 \n \
55	Cs	146	145.94029	0 \n \
55	Cs	147	146.94416	0 \n \
55	Cs	148	147.94922	0 \n \
55	Cs	149	148.95293	0 \n \
55	Cs	150	149.95817	0 \n \
55	Cs	151	150.96219	0 \n \
56	Ba	114	113.95068	0 \n \
56	Ba	115	114.94737	0 \n \
56	Ba	116	115.94138	0 \n \
56	Ba	117	116.9385	0 \n \
56	Ba	118	117.93304	0 \n \
56	Ba	119	118.93066	0 \n \
56	Ba	120	119.92604	0 \n \
56	Ba	121	120.92405	0 \n \
56	Ba	122	121.9199	0 \n \
56	Ba	123	122.918781	0 \n \
56	Ba	124	123.915094	0 \n \
56	Ba	125	124.914473	0 \n \
56	Ba	126	125.91125	0 \n \
56	Ba	127	126.911094	0 \n \
56	Ba	128	127.908318	0 \n \
56	Ba	129	128.908679	0 \n \
56	Ba	130	129.9063208	0.106 \n \
56	Ba	131	130.906941	0 \n \
56	Ba	132	131.9050613	0.101 \n \
56	Ba	133	132.9060075	0 \n",
    "56	Ba	134	133.9045084	2.417 \n \
56	Ba	135	134.9056886	6.592 \n \
56	Ba	136	135.9045759	7.854 \n \
56	Ba	137	136.9058274	11.232 \n \
56	Ba	138	137.9052472	71.698 \n \
56	Ba	139	138.9088413	0 \n \
56	Ba	140	139.910605	0 \n \
56	Ba	141	140.914411	0 \n \
56	Ba	142	141.916453	0 \n \
56	Ba	143	142.920627	0 \n \
56	Ba	144	143.922953	0 \n \
56	Ba	145	144.92763	0 \n \
56	Ba	146	145.93022	0 \n \
56	Ba	147	146.93495	0 \n \
56	Ba	148	147.93772	0 \n \
56	Ba	149	148.94258	0 \n \
56	Ba	150	149.94568	0 \n \
56	Ba	151	150.95081	0 \n \
56	Ba	152	151.95427	0 \n \
56	Ba	153	152.95961	0 \n \
57	La	117	116.95007	0 \n \
57	La	118	117.94673	0 \n \
57	La	119	118.94099	0 \n \
57	La	120	119.93807	0 \n \
57	La	121	120.93301	0 \n \
57	La	122	121.93071	0 \n \
57	La	123	122.92624	0 \n \
57	La	124	123.92457	0 \n \
57	La	125	124.920816	0 \n \
57	La	126	125.91951	0 \n \
57	La	127	126.916375	0 \n \
57	La	128	127.91559	0 \n \
57	La	129	128.912693	0 \n \
57	La	130	129.912369	0 \n \
57	La	131	130.91007	0 \n \
57	La	132	131.9101	0 \n \
57	La	133	132.90822	0 \n \
57	La	134	133.908514	0 \n \
57	La	135	134.906977	0 \n \
57	La	136	135.90764	0 \n \
57	La	137	136.906494	0 \n \
57	La	138	137.907112	0.090 \n \
57	La	139	138.9063533	99.910 \n \
57	La	140	139.9094776	0 \n \
57	La	141	140.910962	0 \n \
57	La	142	141.914079	0 \n \
57	La	143	142.916063	0 \n \
57	La	144	143.9196	0 \n \
57	La	145	144.92165	0 \n \
57	La	146	145.92579	0 \n \
57	La	147	146.92824	0 \n",
    "57	La	148	147.93223	0 \n \
57	La	149	148.93473	0 \n \
57	La	150	149.93877	0 \n \
57	La	151	150.94172	0 \n \
57	La	152	151.94625	0 \n \
57	La	153	152.94962	0 \n \
57	La	154	153.9545	0 \n \
57	La	155	154.95835	0 \n \
58	Ce	119	118.95276	0 \n \
58	Ce	120	119.94664	0 \n \
58	Ce	121	120.94342	0 \n \
58	Ce	122	121.93791	0 \n \
58	Ce	123	122.9354	0 \n \
58	Ce	124	123.93041	0 \n \
58	Ce	125	124.92844	0 \n \
58	Ce	126	125.92397	0 \n \
58	Ce	127	126.92273	0 \n \
58	Ce	128	127.91891	0 \n \
58	Ce	129	128.9181	0 \n \
58	Ce	130	129.91474	0 \n \
58	Ce	131	130.91442	0 \n \
58	Ce	132	131.91146	0 \n \
58	Ce	133	132.911515	0 \n \
58	Ce	134	133.908925	0 \n \
58	Ce	135	134.909151	0 \n \
58	Ce	136	135.907172	0.185 \n \
58	Ce	137	136.907806	0 \n \
58	Ce	138	137.905991	0.251 \n \
58	Ce	139	138.906653	0 \n \
58	Ce	140	139.9054387	88.450 \n \
58	Ce	141	140.9082763	0 \n \
58	Ce	142	141.909244	11.114 \n \
58	Ce	143	142.912386	0 \n \
58	Ce	144	143.913647	0 \n \
58	Ce	145	144.91723	0 \n \
58	Ce	146	145.91876	0 \n \
58	Ce	147	146.92267	0 \n \
58	Ce	148	147.92443	0 \n \
58	Ce	149	148.9284	0 \n \
58	Ce	150	149.93041	0 \n \
58	Ce	151	150.93398	0 \n \
58	Ce	152	151.93654	0 \n \
58	Ce	153	152.94058	0 \n \
58	Ce	154	153.94342	0 \n \
58	Ce	155	154.94804	0 \n \
58	Ce	156	155.95126	0 \n \
58	Ce	157	156.95634	0 \n \
59	Pr	121	120.95536	0 \n \
59	Pr	122	121.95181	0 \n \
59	Pr	123	122.94596	0 \n \
59	Pr	124	123.94296	0 \n",
    "59	Pr	125	124.93783	0 \n \
59	Pr	126	125.93531	0 \n \
59	Pr	127	126.93083	0 \n \
59	Pr	128	127.92879	0 \n \
59	Pr	129	128.9251	0 \n \
59	Pr	130	129.92359	0 \n \
59	Pr	131	130.92026	0 \n \
59	Pr	132	131.91926	0 \n \
59	Pr	133	132.916331	0 \n \
59	Pr	134	133.91571	0 \n \
59	Pr	135	134.913112	0 \n \
59	Pr	136	135.912692	0 \n \
59	Pr	137	136.910705	0 \n \
59	Pr	138	137.910755	0 \n \
59	Pr	139	138.908938	0 \n \
59	Pr	140	139.909076	0 \n \
59	Pr	141	140.9076528	100 \n \
59	Pr	142	141.9100448	0 \n \
59	Pr	143	142.9108169	0 \n \
59	Pr	144	143.913305	0 \n \
59	Pr	145	144.914512	0 \n \
59	Pr	146	145.91764	0 \n \
59	Pr	147	146.918996	0 \n \
59	Pr	148	147.922135	0 \n \
59	Pr	149	148.92372	0 \n \
59	Pr	150	149.926673	0 \n \
59	Pr	151	150.928319	0 \n \
59	Pr	152	151.9315	0 \n \
59	Pr	153	152.93384	0 \n \
59	Pr	154	153.93752	0 \n \
59	Pr	155	154.94012	0 \n \
59	Pr	156	155.94427	0 \n \
59	Pr	157	156.94743	0 \n \
59	Pr	158	157.95198	0 \n \
59	Pr	159	158.9555	0 \n \
60	Nd	124	123.95223	0 \n \
60	Nd	125	124.94888	0 \n \
60	Nd	126	125.94322	0 \n \
60	Nd	127	126.9405	0 \n \
60	Nd	128	127.93539	0 \n \
60	Nd	129	128.93319	0 \n \
60	Nd	130	129.92851	0 \n \
60	Nd	131	130.92725	0 \n \
60	Nd	132	131.923321	0 \n \
60	Nd	133	132.92235	0 \n \
60	Nd	134	133.91879	0 \n \
60	Nd	135	134.918181	0 \n \
60	Nd	136	135.914976	0 \n \
60	Nd	137	136.914567	0 \n \
60	Nd	138	137.91195	0 \n \
60	Nd	139	138.911978	0 \n",
    "60	Nd	140	139.90955	0 \n \
60	Nd	141	140.90961	0 \n \
60	Nd	142	141.9077233	27.2 \n \
60	Nd	143	142.9098143	12.2 \n \
60	Nd	144	143.9100873	23.8 \n \
60	Nd	145	144.9125736	8.3 \n \
60	Nd	146	145.9131169	17.2 \n \
60	Nd	147	146.9161004	0 \n \
60	Nd	148	147.916893	5.7 \n \
60	Nd	149	148.920149	0 \n \
60	Nd	150	149.920891	5.6 \n \
60	Nd	151	150.923829	0 \n \
60	Nd	152	151.924682	0 \n \
60	Nd	153	152.927698	0 \n \
60	Nd	154	153.92948	0 \n \
60	Nd	155	154.93293	0 \n \
60	Nd	156	155.93502	0 \n \
60	Nd	157	156.93903	0 \n \
60	Nd	158	157.9416	0 \n \
60	Nd	159	158.94609	0 \n \
60	Nd	160	159.94909	0 \n \
60	Nd	161	160.95388	0 \n \
61	Pm	126	125.95752	0 \n \
61	Pm	127	126.95163	0 \n \
61	Pm	128	127.94842	0 \n \
61	Pm	129	128.94316	0 \n \
61	Pm	130	129.94045	0 \n \
61	Pm	131	130.93587	0 \n \
61	Pm	132	131.93375	0 \n \
61	Pm	133	132.92978	0 \n \
61	Pm	134	133.92835	0 \n \
61	Pm	135	134.92488	0 \n \
61	Pm	136	135.92357	0 \n \
61	Pm	137	136.920479	0 \n \
61	Pm	138	137.919548	0 \n \
61	Pm	139	138.916804	0 \n \
61	Pm	140	139.91604	0 \n \
61	Pm	141	140.913555	0 \n \
61	Pm	142	141.912874	0 \n \
61	Pm	143	142.910933	0 \n \
61	Pm	144	143.912591	0 \n \
61	Pm	145	144.912749	0 \n \
61	Pm	146	145.914696	0 \n \
61	Pm	147	146.9151385	0 \n \
61	Pm	148	147.917475	0 \n \
61	Pm	149	148.918334	0 \n \
61	Pm	150	149.920984	0 \n \
61	Pm	151	150.921207	0 \n \
61	Pm	152	151.923497	0 \n \
61	Pm	153	152.924117	0 \n \
61	Pm	154	153.92646	0 \n",
    "61	Pm	155	154.9281	0 \n \
61	Pm	156	155.93106	0 \n \
61	Pm	157	156.93304	0 \n \
61	Pm	158	157.93656	0 \n \
61	Pm	159	158.93897	0 \n \
61	Pm	160	159.94299	0 \n \
61	Pm	161	160.94586	0 \n \
61	Pm	162	161.95029	0 \n \
61	Pm	163	162.95368	0 \n \
62	Sm	128	127.95808	0 \n \
62	Sm	129	128.95464	0 \n \
62	Sm	130	129.94892	0 \n \
62	Sm	131	130.94611	0 \n \
62	Sm	132	131.94069	0 \n \
62	Sm	133	132.93867	0 \n \
62	Sm	134	133.93397	0 \n \
62	Sm	135	134.93252	0 \n \
62	Sm	136	135.928276	0 \n \
62	Sm	137	136.92697	0 \n \
62	Sm	138	137.923244	0 \n \
62	Sm	139	138.922297	0 \n \
62	Sm	140	139.918995	0 \n \
62	Sm	141	140.918476	0 \n \
62	Sm	142	141.915198	0 \n \
62	Sm	143	142.914628	0 \n \
62	Sm	144	143.911999	3.07 \n \
62	Sm	145	144.91341	0 \n \
62	Sm	146	145.913041	0 \n \
62	Sm	147	146.9148979	14.99 \n \
62	Sm	148	147.9148227	11.24 \n \
62	Sm	149	148.9171847	13.82 \n \
62	Sm	150	149.9172755	7.38 \n \
62	Sm	151	150.9199324	0 \n \
62	Sm	152	151.9197324	26.75 \n \
62	Sm	153	152.9220974	0 \n \
62	Sm	154	153.9222093	22.75 \n \
62	Sm	155	154.9246402	0 \n \
62	Sm	156	155.925528	0 \n \
62	Sm	157	156.92836	0 \n \
62	Sm	158	157.92999	0 \n \
62	Sm	159	158.93321	0 \n \
62	Sm	160	159.93514	0 \n \
62	Sm	161	160.93883	0 \n \
62	Sm	162	161.94122	0 \n \
62	Sm	163	162.94536	0 \n \
62	Sm	164	163.94828	0 \n \
62	Sm	165	164.95298	0 \n \
63	Eu	130	129.96357	0 \n \
63	Eu	131	130.95775	0 \n \
63	Eu	132	131.95437	0 \n \
63	Eu	133	132.94924	0 \n",
    "63	Eu	134	133.94651	0 \n \
63	Eu	135	134.94182	0 \n \
63	Eu	136	135.9396	0 \n \
63	Eu	137	136.93557	0 \n \
63	Eu	138	137.93371	0 \n \
63	Eu	139	138.929792	0 \n \
63	Eu	140	139.92809	0 \n \
63	Eu	141	140.924931	0 \n \
63	Eu	142	141.92343	0 \n \
63	Eu	143	142.920298	0 \n \
63	Eu	144	143.918817	0 \n \
63	Eu	145	144.916265	0 \n \
63	Eu	146	145.917206	0 \n \
63	Eu	147	146.916746	0 \n \
63	Eu	148	147.918086	0 \n \
63	Eu	149	148.917931	0 \n \
63	Eu	150	149.919702	0 \n \
63	Eu	151	150.9198502	47.81 \n \
63	Eu	152	151.9217445	0 \n \
63	Eu	153	152.9212303	52.19 \n \
63	Eu	154	153.9229792	0 \n \
63	Eu	155	154.9228933	0 \n \
63	Eu	156	155.924752	0 \n \
63	Eu	157	156.925424	0 \n \
63	Eu	158	157.92785	0 \n \
63	Eu	159	158.929089	0 \n \
63	Eu	160	159.93197	0 \n \
63	Eu	161	160.93368	0 \n \
63	Eu	162	161.93704	0 \n \
63	Eu	163	162.93921	0 \n \
63	Eu	164	163.94299	0 \n \
63	Eu	165	164.94572	0 \n \
63	Eu	166	165.94997	0 \n \
63	Eu	167	166.95321	0 \n \
64	Gd	134	133.95537	0 \n \
64	Gd	135	134.95257	0 \n \
64	Gd	136	135.94734	0 \n \
64	Gd	137	136.94502	0 \n \
64	Gd	138	137.94012	0 \n \
64	Gd	139	138.93824	0 \n \
64	Gd	140	139.93367	0 \n \
64	Gd	141	140.932126	0 \n \
64	Gd	142	141.92812	0 \n \
64	Gd	143	142.92675	0 \n \
64	Gd	144	143.92296	0 \n \
64	Gd	145	144.921709	0 \n \
64	Gd	146	145.918311	0 \n \
64	Gd	147	146.919094	0 \n \
64	Gd	148	147.918115	0 \n \
64	Gd	149	148.919341	0 \n \
64	Gd	150	149.918659	0 \n",
    "64	Gd	151	150.920348	0 \n \
64	Gd	152	151.919791	0.20 \n \
64	Gd	153	152.9217495	0 \n \
64	Gd	154	153.9208656	2.18 \n \
64	Gd	155	154.922622	14.80 \n \
64	Gd	156	155.9221227	20.47 \n \
64	Gd	157	156.9239601	15.65 \n \
64	Gd	158	157.9241039	24.84 \n \
64	Gd	159	158.9263887	0 \n \
64	Gd	160	159.9270541	21.86 \n \
64	Gd	161	160.9296692	0 \n \
64	Gd	162	161.930985	0 \n \
64	Gd	163	162.93399	0 \n \
64	Gd	164	163.93586	0 \n \
64	Gd	165	164.93938	0 \n \
64	Gd	166	165.9416	0 \n \
64	Gd	167	166.94557	0 \n \
64	Gd	168	167.94836	0 \n \
64	Gd	169	168.95287	0 \n \
65	Tb	136	135.96138	0 \n \
65	Tb	137	136.95598	0 \n \
65	Tb	138	137.95316	0 \n \
65	Tb	139	138.94829	0 \n \
65	Tb	140	139.94581	0 \n \
65	Tb	141	140.94145	0 \n \
65	Tb	142	141.93874	0 \n \
65	Tb	143	142.93512	0 \n \
65	Tb	144	143.93305	0 \n \
65	Tb	145	144.92927	0 \n \
65	Tb	146	145.92725	0 \n \
65	Tb	147	146.924045	0 \n \
65	Tb	148	147.924272	0 \n \
65	Tb	149	148.923246	0 \n \
65	Tb	150	149.92366	0 \n \
65	Tb	151	150.923103	0 \n \
65	Tb	152	151.92407	0 \n \
65	Tb	153	152.923435	0 \n \
65	Tb	154	153.92468	0 \n \
65	Tb	155	154.923505	0 \n \
65	Tb	156	155.924747	0 \n \
65	Tb	157	156.9240246	0 \n \
65	Tb	158	157.9254131	0 \n \
65	Tb	159	158.9253468	100 \n \
65	Tb	160	159.9271676	0 \n \
65	Tb	161	160.9275699	0 \n \
65	Tb	162	161.92949	0 \n \
65	Tb	163	162.930648	0 \n \
65	Tb	164	163.93335	0 \n \
65	Tb	165	164.93488	0 \n \
65	Tb	166	165.93799	0 \n \
65	Tb	167	166.94005	0 \n",
    "65	Tb	168	167.94364	0 \n \
65	Tb	169	168.94622	0 \n \
65	Tb	170	169.95025	0 \n \
65	Tb	171	170.9533	0 \n \
66	Dy	138	137.96249	0 \n \
66	Dy	139	138.95954	0 \n \
66	Dy	140	139.95401	0 \n \
66	Dy	141	140.95135	0 \n \
66	Dy	142	141.94637	0 \n \
66	Dy	143	142.94383	0 \n \
66	Dy	144	143.93925	0 \n \
66	Dy	145	144.93743	0 \n \
66	Dy	146	145.932845	0 \n \
66	Dy	147	146.931092	0 \n \
66	Dy	148	147.92715	0 \n \
66	Dy	149	148.927305	0 \n \
66	Dy	150	149.925585	0 \n \
66	Dy	151	150.926185	0 \n \
66	Dy	152	151.924718	0 \n \
66	Dy	153	152.925765	0 \n \
66	Dy	154	153.924424	0 \n \
66	Dy	155	154.925754	0 \n \
66	Dy	156	155.924283	0.06 \n \
66	Dy	157	156.925466	0 \n \
66	Dy	158	157.924409	0.10 \n \
66	Dy	159	158.9257392	0 \n \
66	Dy	160	159.9251975	2.34 \n \
66	Dy	161	160.9269334	18.91 \n \
66	Dy	162	161.9267984	25.51 \n \
66	Dy	163	162.9287312	24.90 \n \
66	Dy	164	163.9291748	28.18 \n \
66	Dy	165	164.9317033	0 \n \
66	Dy	166	165.9328067	0 \n \
66	Dy	167	166.93566	0 \n \
66	Dy	168	167.93713	0 \n \
66	Dy	169	168.94031	0 \n \
66	Dy	170	169.94239	0 \n \
66	Dy	171	170.9462	0 \n \
66	Dy	172	171.94876	0 \n \
66	Dy	173	172.953	0 \n \
67	Ho	140	139.96854	0 \n \
67	Ho	141	140.9631	0 \n \
67	Ho	142	141.95977	0 \n \
67	Ho	143	142.95461	0 \n \
67	Ho	144	143.95148	0 \n \
67	Ho	145	144.9472	0 \n \
67	Ho	146	145.94464	0 \n \
67	Ho	147	146.94006	0 \n \
67	Ho	148	147.93772	0 \n \
67	Ho	149	148.933775	0 \n \
67	Ho	150	149.933496	0 \n",
    "67	Ho	151	150.931688	0 \n \
67	Ho	152	151.931714	0 \n \
67	Ho	153	152.930199	0 \n \
67	Ho	154	153.930602	0 \n \
67	Ho	155	154.929103	0 \n \
67	Ho	156	155.92984	0 \n \
67	Ho	157	156.928256	0 \n \
67	Ho	158	157.928941	0 \n \
67	Ho	159	158.927712	0 \n \
67	Ho	160	159.928729	0 \n \
67	Ho	161	160.927855	0 \n \
67	Ho	162	161.929096	0 \n \
67	Ho	163	162.9287339	0 \n \
67	Ho	164	163.9302335	0 \n \
67	Ho	165	164.9303221	100 \n \
67	Ho	166	165.9322842	0 \n \
67	Ho	167	166.933133	0 \n \
67	Ho	168	167.93552	0 \n \
67	Ho	169	168.936872	0 \n \
67	Ho	170	169.93962	0 \n \
67	Ho	171	170.94147	0 \n \
67	Ho	172	171.94482	0 \n \
67	Ho	173	172.94729	0 \n \
67	Ho	174	173.95115	0 \n \
67	Ho	175	174.95405	0 \n \
68	Er	143	142.96634	0 \n \
68	Er	144	143.96038	0 \n \
68	Er	145	144.95739	0 \n \
68	Er	146	145.952	0 \n \
68	Er	147	146.94949	0 \n \
68	Er	148	147.94455	0 \n \
68	Er	149	148.94231	0 \n \
68	Er	150	149.937914	0 \n \
68	Er	151	150.937449	0 \n \
68	Er	152	151.93505	0 \n \
68	Er	153	152.935063	0 \n \
68	Er	154	153.932783	0 \n \
68	Er	155	154.933209	0 \n \
68	Er	156	155.931065	0 \n \
68	Er	157	156.93192	0 \n \
68	Er	158	157.929893	0 \n \
68	Er	159	158.930684	0 \n \
68	Er	160	159.929083	0 \n \
68	Er	161	160.929995	0 \n \
68	Er	162	161.928778	0.14 \n \
68	Er	163	162.930033	0 \n \
68	Er	164	163.9292	1.61 \n \
68	Er	165	164.930726	0 \n \
68	Er	166	165.9302931	33.61 \n \
68	Er	167	166.9320482	22.93 \n \
68	Er	168	167.9323702	26.78 \n",
    "68	Er	169	168.9345904	0 \n \
68	Er	170	169.9354643	14.93 \n \
68	Er	171	170.9380298	0 \n \
68	Er	172	171.939356	0 \n \
68	Er	173	172.9424	0 \n \
68	Er	174	173.94423	0 \n \
68	Er	175	174.94777	0 \n \
68	Er	176	175.95008	0 \n \
68	Er	177	176.95405	0 \n \
69	Tm	145	144.97007	0 \n \
69	Tm	146	145.96643	0 \n \
69	Tm	147	146.96096	0 \n \
69	Tm	148	147.95784	0 \n \
69	Tm	149	148.95272	0 \n \
69	Tm	150	149.94996	0 \n \
69	Tm	151	150.945483	0 \n \
69	Tm	152	151.94442	0 \n \
69	Tm	153	152.942012	0 \n \
69	Tm	154	153.941568	0 \n \
69	Tm	155	154.939199	0 \n \
69	Tm	156	155.93898	0 \n \
69	Tm	157	156.93697	0 \n \
69	Tm	158	157.93698	0 \n \
69	Tm	159	158.93498	0 \n \
69	Tm	160	159.93526	0 \n \
69	Tm	161	160.93355	0 \n \
69	Tm	162	161.933995	0 \n \
69	Tm	163	162.932651	0 \n \
69	Tm	164	163.93356	0 \n \
69	Tm	165	164.932435	0 \n \
69	Tm	166	165.933554	0 \n \
69	Tm	167	166.9328516	0 \n \
69	Tm	168	167.934173	0 \n \
69	Tm	169	168.9342133	100 \n \
69	Tm	170	169.9358014	0 \n \
69	Tm	171	170.9364294	0 \n \
69	Tm	172	171.9384	0 \n \
69	Tm	173	172.939604	0 \n \
69	Tm	174	173.94217	0 \n \
69	Tm	175	174.94384	0 \n \
69	Tm	176	175.94699	0 \n \
69	Tm	177	176.94904	0 \n \
69	Tm	178	177.95264	0 \n \
69	Tm	179	178.95534	0 \n \
70	Yb	148	147.96742	0 \n \
70	Yb	149	148.96404	0 \n \
70	Yb	150	149.95842	0 \n \
70	Yb	151	150.9554	0 \n \
70	Yb	152	151.95029	0 \n \
70	Yb	153	152.94948	0 \n \
70	Yb	154	153.946394	0 \n",
    "70	Yb	155	154.945782	0 \n \
70	Yb	156	155.942818	0 \n \
70	Yb	157	156.942628	0 \n \
70	Yb	158	157.939866	0 \n \
70	Yb	159	158.94005	0 \n \
70	Yb	160	159.937552	0 \n \
70	Yb	161	160.937902	0 \n \
70	Yb	162	161.935768	0 \n \
70	Yb	163	162.936334	0 \n \
70	Yb	164	163.934489	0 \n \
70	Yb	165	164.93528	0 \n \
70	Yb	166	165.933882	0 \n \
70	Yb	167	166.93495	0 \n \
70	Yb	168	167.933897	0.13 \n \
70	Yb	169	168.93519	0 \n \
70	Yb	170	169.9347618	3.04 \n \
70	Yb	171	170.9363258	14.28 \n \
70	Yb	172	171.9363815	21.83 \n \
70	Yb	173	172.9382108	16.13 \n \
70	Yb	174	173.9388621	31.83 \n \
70	Yb	175	174.9412765	0 \n \
70	Yb	176	175.9425717	12.76 \n \
70	Yb	177	176.9452608	0 \n \
70	Yb	178	177.946647	0 \n \
70	Yb	179	178.95017	0 \n \
70	Yb	180	179.95233	0 \n \
70	Yb	181	180.95615	0 \n \
71	Lu	150	149.97323	0 \n \
71	Lu	151	150.96758	0 \n \
71	Lu	152	151.96412	0 \n \
71	Lu	153	152.95877	0 \n \
71	Lu	154	153.95752	0 \n \
71	Lu	155	154.954316	0 \n \
71	Lu	156	155.95303	0 \n \
71	Lu	157	156.950098	0 \n \
71	Lu	158	157.949313	0 \n \
71	Lu	159	158.94663	0 \n \
71	Lu	160	159.94603	0 \n \
71	Lu	161	160.94357	0 \n \
71	Lu	162	161.94328	0 \n \
71	Lu	163	162.94118	0 \n \
71	Lu	164	163.94134	0 \n \
71	Lu	165	164.939407	0 \n \
71	Lu	166	165.93986	0 \n \
71	Lu	167	166.93827	0 \n \
71	Lu	168	167.93874	0 \n \
71	Lu	169	168.937651	0 \n \
71	Lu	170	169.938475	0 \n \
71	Lu	171	170.9379131	0 \n \
71	Lu	172	171.939086	0 \n \
71	Lu	173	172.9389306	0 \n",
    "71	Lu	174	173.9403375	0 \n \
71	Lu	175	174.9407718	97.41 \n \
71	Lu	176	175.9426863	2.59 \n \
71	Lu	177	176.9437581	0 \n \
71	Lu	178	177.945955	0 \n \
71	Lu	179	178.947327	0 \n \
71	Lu	180	179.94988	0 \n \
71	Lu	181	180.95197	0 \n \
71	Lu	182	181.95504	0 \n \
71	Lu	183	182.95757	0 \n \
71	Lu	184	183.96091	0 \n \
72	Hf	153	152.97069	0 \n \
72	Hf	154	153.96486	0 \n \
72	Hf	155	154.96339	0 \n \
72	Hf	156	155.95936	0 \n \
72	Hf	157	156.9584	0 \n \
72	Hf	158	157.954799	0 \n \
72	Hf	159	158.953995	0 \n \
72	Hf	160	159.950684	0 \n \
72	Hf	161	160.950275	0 \n \
72	Hf	162	161.94721	0 \n \
72	Hf	163	162.94709	0 \n \
72	Hf	164	163.944367	0 \n \
72	Hf	165	164.94457	0 \n \
72	Hf	166	165.94218	0 \n \
72	Hf	167	166.9426	0 \n \
72	Hf	168	167.94057	0 \n \
72	Hf	169	168.94126	0 \n \
72	Hf	170	169.93961	0 \n \
72	Hf	171	170.94049	0 \n \
72	Hf	172	171.939448	0 \n \
72	Hf	173	172.94051	0 \n \
72	Hf	174	173.940046	0.16 \n \
72	Hf	175	174.941509	0 \n \
72	Hf	176	175.9414086	5.26 \n \
72	Hf	177	176.9432207	18.60 \n \
72	Hf	178	177.9436988	27.28 \n \
72	Hf	179	178.9458161	13.62 \n \
72	Hf	180	179.94655	35.08 \n \
72	Hf	181	180.9491012	0 \n \
72	Hf	182	181.950554	0 \n \
72	Hf	183	182.95353	0 \n \
72	Hf	184	183.95545	0 \n \
72	Hf	185	184.95882	0 \n \
72	Hf	186	185.96089	0 \n \
72	Hf	187	186.96459	0 \n \
72	Hf	188	187.96685	0 \n \
73	Ta	155	154.97459	0 \n \
73	Ta	156	155.9723	0 \n \
73	Ta	157	156.96819	0 \n \
73	Ta	158	157.9667	0 \n",
    "73	Ta	159	158.963018	0 \n \
73	Ta	160	159.96149	0 \n \
73	Ta	161	160.95842	0 \n \
73	Ta	162	161.95729	0 \n \
73	Ta	163	162.95433	0 \n \
73	Ta	164	163.95353	0 \n \
73	Ta	165	164.950773	0 \n \
73	Ta	166	165.95051	0 \n \
73	Ta	167	166.94809	0 \n \
73	Ta	168	167.94805	0 \n \
73	Ta	169	168.94601	0 \n \
73	Ta	170	169.94618	0 \n \
73	Ta	171	170.94448	0 \n \
73	Ta	172	171.9449	0 \n \
73	Ta	173	172.94375	0 \n \
73	Ta	174	173.94445	0 \n \
73	Ta	175	174.94374	0 \n \
73	Ta	176	175.94486	0 \n \
73	Ta	177	176.944472	0 \n \
73	Ta	178	177.945778	0 \n \
73	Ta	179	178.9459295	0 \n \
73	Ta	180	179.9474648	0.012 \n \
73	Ta	181	180.9479958	99.988 \n \
73	Ta	182	181.9501518	0 \n \
73	Ta	183	182.9513726	0 \n \
73	Ta	184	183.954008	0 \n \
73	Ta	185	184.955559	0 \n \
73	Ta	186	185.95855	0 \n \
73	Ta	187	186.96053	0 \n \
73	Ta	188	187.9637	0 \n \
73	Ta	189	188.96583	0 \n \
73	Ta	190	189.96923	0 \n \
74	W	158	157.97456	0 \n \
74	W	159	158.97292	0 \n \
74	W	160	159.96848	0 \n \
74	W	161	160.96736	0 \n \
74	W	162	161.963497	0 \n \
74	W	163	162.96252	0 \n \
74	W	164	163.958954	0 \n \
74	W	165	164.95828	0 \n \
74	W	166	165.955027	0 \n \
74	W	167	166.954816	0 \n \
74	W	168	167.951808	0 \n \
74	W	169	168.951779	0 \n \
74	W	170	169.949228	0 \n \
74	W	171	170.94945	0 \n \
74	W	172	171.94729	0 \n \
74	W	173	172.94769	0 \n \
74	W	174	173.94608	0 \n \
74	W	175	174.94672	0 \n \
74	W	176	175.94563	0 \n",
    "74	W	177	176.94664	0 \n \
74	W	178	177.945876	0 \n \
74	W	179	178.94707	0 \n \
74	W	180	179.946704	0.12 \n \
74	W	181	180.948197	0 \n \
74	W	182	181.9482042	26.50 \n \
74	W	183	182.950223	14.31 \n \
74	W	184	183.9509312	30.64 \n \
74	W	185	184.9534193	0 \n \
74	W	186	185.9543641	28.43 \n \
74	W	187	186.9571605	0 \n \
74	W	188	187.958489	0 \n \
74	W	189	188.96191	0 \n \
74	W	190	189.96318	0 \n \
74	W	191	190.9666	0 \n \
74	W	192	191.96817	0 \n \
75	Re	160	159.98212	0 \n \
75	Re	161	160.97759	0 \n \
75	Re	162	161.976	0 \n \
75	Re	163	162.972081	0 \n \
75	Re	164	163.97032	0 \n \
75	Re	165	164.967089	0 \n \
75	Re	166	165.96581	0 \n \
75	Re	167	166.9626	0 \n \
75	Re	168	167.96157	0 \n \
75	Re	169	168.95879	0 \n \
75	Re	170	169.95822	0 \n \
75	Re	171	170.95572	0 \n \
75	Re	172	171.95542	0 \n \
75	Re	173	172.95324	0 \n \
75	Re	174	173.95312	0 \n \
75	Re	175	174.95138	0 \n \
75	Re	176	175.95162	0 \n \
75	Re	177	176.95033	0 \n \
75	Re	178	177.95099	0 \n \
75	Re	179	178.949988	0 \n \
75	Re	180	179.950789	0 \n \
75	Re	181	180.950068	0 \n \
75	Re	182	181.95121	0 \n \
75	Re	183	182.95082	0 \n \
75	Re	184	183.952521	0 \n \
75	Re	185	184.952955	37.40 \n \
75	Re	186	185.9549861	0 \n \
75	Re	187	186.9557531	62.60 \n \
75	Re	188	187.9581144	0 \n \
75	Re	189	188.959229	0 \n \
75	Re	190	189.96182	0 \n \
75	Re	191	190.963125	0 \n \
75	Re	192	191.96596	0 \n \
75	Re	193	192.96747	0 \n \
75	Re	194	193.97042	0 \n",
    "76	Os	162	161.98443	0 \n \
76	Os	163	162.98269	0 \n \
76	Os	164	163.97804	0 \n \
76	Os	165	164.97676	0 \n \
76	Os	166	165.972691	0 \n \
76	Os	167	166.97155	0 \n \
76	Os	168	167.967804	0 \n \
76	Os	169	168.967019	0 \n \
76	Os	170	169.963577	0 \n \
76	Os	171	170.963185	0 \n \
76	Os	172	171.960023	0 \n \
76	Os	173	172.959808	0 \n \
76	Os	174	173.957062	0 \n \
76	Os	175	174.956946	0 \n \
76	Os	176	175.95481	0 \n \
76	Os	177	176.954965	0 \n \
76	Os	178	177.953251	0 \n \
76	Os	179	178.953816	0 \n \
76	Os	180	179.952379	0 \n \
76	Os	181	180.95324	0 \n \
76	Os	182	181.95211	0 \n \
76	Os	183	182.95313	0 \n \
76	Os	184	183.9524891	0.02 \n \
76	Os	185	184.9540423	0 \n \
76	Os	186	185.9538382	1.59 \n \
76	Os	187	186.9557505	1.96 \n \
76	Os	188	187.9558382	13.24 \n \
76	Os	189	188.9581475	16.15 \n \
76	Os	190	189.958447	26.26 \n \
76	Os	191	190.9609297	0 \n \
76	Os	192	191.9614807	40.78 \n \
76	Os	193	192.9641516	0 \n \
76	Os	194	193.9651821	0 \n \
76	Os	195	194.96813	0 \n \
76	Os	196	195.96964	0 \n \
77	Ir	164	163.9922	0 \n \
77	Ir	165	164.98752	0 \n \
77	Ir	166	165.98582	0 \n \
77	Ir	167	166.981665	0 \n \
77	Ir	168	167.97988	0 \n \
77	Ir	169	168.976295	0 \n \
77	Ir	170	169.97497	0 \n \
77	Ir	171	170.97163	0 \n \
77	Ir	172	171.97046	0 \n \
77	Ir	173	172.967502	0 \n \
77	Ir	174	173.966861	0 \n \
77	Ir	175	174.964113	0 \n \
77	Ir	176	175.963649	0 \n \
77	Ir	177	176.961302	0 \n \
77	Ir	178	177.961082	0 \n \
77	Ir	179	178.959122	0 \n",
    "77	Ir	180	179.959229	0 \n \
77	Ir	181	180.957625	0 \n \
77	Ir	182	181.958076	0 \n \
77	Ir	183	182.956846	0 \n \
77	Ir	184	183.95748	0 \n \
77	Ir	185	184.9567	0 \n \
77	Ir	186	185.957946	0 \n \
77	Ir	187	186.957363	0 \n \
77	Ir	188	187.958853	0 \n \
77	Ir	189	188.958719	0 \n \
77	Ir	190	189.960546	0 \n \
77	Ir	191	190.960594	37.3 \n \
77	Ir	192	191.962605	0 \n \
77	Ir	193	192.9629264	62.7 \n \
77	Ir	194	193.9650784	0 \n \
77	Ir	195	194.9659796	0 \n \
77	Ir	196	195.9684	0 \n \
77	Ir	197	196.969653	0 \n \
77	Ir	198	197.97228	0 \n \
77	Ir	199	198.9738	0 \n \
78	Pt	166	165.99486	0 \n \
78	Pt	167	166.99298	0 \n \
78	Pt	168	167.98815	0 \n \
78	Pt	169	168.98672	0 \n \
78	Pt	170	169.982495	0 \n \
78	Pt	171	170.98124	0 \n \
78	Pt	172	171.977347	0 \n \
78	Pt	173	172.97644	0 \n \
78	Pt	174	173.972819	0 \n \
78	Pt	175	174.972421	0 \n \
78	Pt	176	175.968945	0 \n \
78	Pt	177	176.968469	0 \n \
78	Pt	178	177.965649	0 \n \
78	Pt	179	178.965363	0 \n \
78	Pt	180	179.963031	0 \n \
78	Pt	181	180.963097	0 \n \
78	Pt	182	181.961171	0 \n \
78	Pt	183	182.961597	0 \n \
78	Pt	184	183.959922	0 \n \
78	Pt	185	184.96062	0 \n \
78	Pt	186	185.959351	0 \n \
78	Pt	187	186.96059	0 \n \
78	Pt	188	187.959395	0 \n \
78	Pt	189	188.960834	0 \n \
78	Pt	190	189.959932	0.014 \n \
78	Pt	191	190.961677	0 \n \
78	Pt	192	191.961038	0.782 \n \
78	Pt	193	192.9629874	0 \n \
78	Pt	194	193.9626803	32.967 \n \
78	Pt	195	194.9647911	33.832 \n \
78	Pt	196	195.9649515	25.242 \n \
78	Pt	197	196.9673402	0 \n \
78	Pt	198	197.967893	7.163 \n",
    "78	Pt	199	198.970593	0 \n \
78	Pt	200	199.971441	0 \n \
78	Pt	201	200.97451	0 \n \
78	Pt	202	201.97574	0 \n \
79	Au	169	168.99808	0 \n \
79	Au	170	169.99612	0 \n \
79	Au	171	170.991879	0 \n \
79	Au	172	171.99004	0 \n \
79	Au	173	172.986237	0 \n \
79	Au	174	173.98476	0 \n \
79	Au	175	174.98127	0 \n \
79	Au	176	175.9801	0 \n \
79	Au	177	176.976865	0 \n \
79	Au	178	177.97603	0 \n \
79	Au	179	178.973213	0 \n \
79	Au	180	179.972521	0 \n \
79	Au	181	180.970079	0 \n \
79	Au	182	181.969618	0 \n \
79	Au	183	182.967593	0 \n \
79	Au	184	183.967452	0 \n \
79	Au	185	184.965789	0 \n \
79	Au	186	185.965953	0 \n \
79	Au	187	186.964568	0 \n \
79	Au	188	187.965324	0 \n \
79	Au	189	188.963948	0 \n \
79	Au	190	189.9647	0 \n \
79	Au	191	190.9637	0 \n \
79	Au	192	191.964813	0 \n \
79	Au	193	192.96415	0 \n \
79	Au	194	193.965365	0 \n \
79	Au	195	194.9650346	0 \n \
79	Au	196	195.96657	0 \n \
79	Au	197	196.9665687	100 \n \
79	Au	198	197.9682423	0 \n \
79	Au	199	198.9687652	0 \n \
79	Au	200	199.97073	0 \n \
79	Au	201	200.971657	0 \n \
79	Au	202	201.97381	0 \n \
79	Au	203	202.975155	0 \n \
79	Au	204	203.97772	0 \n \
79	Au	205	204.97987	0 \n \
80	Hg	171	171.00376	0 \n \
80	Hg	172	171.99883	0 \n \
80	Hg	173	172.99724	0 \n \
80	Hg	174	173.992864	0 \n \
80	Hg	175	174.99142	0 \n \
80	Hg	176	175.987355	0 \n \
80	Hg	177	176.98628	0 \n \
80	Hg	178	177.982483	0 \n \
80	Hg	179	178.981834	0 \n \
80	Hg	180	179.978266	0 \n",
    "80	Hg	181	180.977819	0 \n \
80	Hg	182	181.97469	0 \n \
80	Hg	183	182.97445	0 \n \
80	Hg	184	183.971713	0 \n \
80	Hg	185	184.971899	0 \n \
80	Hg	186	185.969362	0 \n \
80	Hg	187	186.969814	0 \n \
80	Hg	188	187.967577	0 \n \
80	Hg	189	188.96819	0 \n \
80	Hg	190	189.966322	0 \n \
80	Hg	191	190.967157	0 \n \
80	Hg	192	191.965634	0 \n \
80	Hg	193	192.966665	0 \n \
80	Hg	194	193.965439	0 \n \
80	Hg	195	194.96672	0 \n \
80	Hg	196	195.965833	0.15 \n \
80	Hg	197	196.967213	0 \n \
80	Hg	198	197.966769	9.97 \n \
80	Hg	199	198.9682799	16.87 \n \
80	Hg	200	199.968326	23.10 \n \
80	Hg	201	200.9703023	13.18 \n \
80	Hg	202	201.970643	29.86 \n \
80	Hg	203	202.9728725	0 \n \
80	Hg	204	203.9734939	6.87 \n \
80	Hg	205	204.976073	0 \n \
80	Hg	206	205.977514	0 \n \
80	Hg	207	206.98259	0 \n \
80	Hg	208	207.98594	0 \n \
80	Hg	209	208.99104	0 \n \
80	Hg	210	209.99451	0 \n \
81	Tl	176	176.00059	0 \n \
81	Tl	177	176.996427	0 \n \
81	Tl	178	177.9949	0 \n \
81	Tl	179	178.99109	0 \n \
81	Tl	180	179.98991	0 \n \
81	Tl	181	180.986257	0 \n \
81	Tl	182	181.98567	0 \n \
81	Tl	183	182.982193	0 \n \
81	Tl	184	183.98187	0 \n \
81	Tl	185	184.97879	0 \n \
81	Tl	186	185.97833	0 \n \
81	Tl	187	186.975906	0 \n \
81	Tl	188	187.97601	0 \n \
81	Tl	189	188.973588	0 \n \
81	Tl	190	189.97388	0 \n \
81	Tl	191	190.971786	0 \n \
81	Tl	192	191.97223	0 \n \
81	Tl	193	192.97067	0 \n \
81	Tl	194	193.9712	0 \n \
81	Tl	195	194.969774	0 \n \
81	Tl	196	195.970481	0 \n",
    "81	Tl	197	196.969575	0 \n \
81	Tl	198	197.97048	0 \n \
81	Tl	199	198.96988	0 \n \
81	Tl	200	199.970963	0 \n \
81	Tl	201	200.970819	0 \n \
81	Tl	202	201.972106	0 \n \
81	Tl	203	202.9723442	29.524 \n \
81	Tl	204	203.9738635	0 \n \
81	Tl	205	204.9744275	70.476 \n \
81	Tl	206	205.9761103	0 \n \
81	Tl	207	206.977419	0 \n \
81	Tl	208	207.9820187	0 \n \
81	Tl	209	208.985359	0 \n \
81	Tl	210	209.990074	0 \n \
81	Tl	211	210.99348	0 \n \
81	Tl	212	211.99823	0 \n \
82	Pb	178	178.00383	0 \n \
82	Pb	179	179.00215	0 \n \
82	Pb	180	179.997918	0 \n \
82	Pb	181	180.99662	0 \n \
82	Pb	182	181.992672	0 \n \
82	Pb	183	182.99187	0 \n \
82	Pb	184	183.988142	0 \n \
82	Pb	185	184.98761	0 \n \
82	Pb	186	185.984239	0 \n \
82	Pb	187	186.983918	0 \n \
82	Pb	188	187.980874	0 \n \
82	Pb	189	188.98081	0 \n \
82	Pb	190	189.978082	0 \n \
82	Pb	191	190.97827	0 \n \
82	Pb	192	191.975785	0 \n \
82	Pb	193	192.97617	0 \n \
82	Pb	194	193.974012	0 \n \
82	Pb	195	194.974542	0 \n \
82	Pb	196	195.972774	0 \n \
82	Pb	197	196.973431	0 \n \
82	Pb	198	197.972034	0 \n \
82	Pb	199	198.972917	0 \n \
82	Pb	200	199.971827	0 \n \
82	Pb	201	200.972885	0 \n \
82	Pb	202	201.972159	0 \n \
82	Pb	203	202.973391	0 \n \
82	Pb	204	203.9730436	1.4 \n \
82	Pb	205	204.9744818	0 \n \
82	Pb	206	205.9744653	24.1 \n \
82	Pb	207	206.9758969	22.1 \n \
82	Pb	208	207.9766521	52.4 \n \
82	Pb	209	208.9810901	0 \n \
82	Pb	210	209.9841885	0 \n \
82	Pb	211	210.988737	0 \n \
82	Pb	212	211.9918975	0 \n",
    "82	Pb	213	212.996581	0 \n \
82	Pb	214	213.9998054	0 \n \
82	Pb	215	215.00481	0 \n \
83	Bi	184	184.00112	0 \n \
83	Bi	185	184.99763	0 \n \
83	Bi	186	185.9966	0 \n \
83	Bi	187	186.993158	0 \n \
83	Bi	188	187.99227	0 \n \
83	Bi	189	188.9892	0 \n \
83	Bi	190	189.9883	0 \n \
83	Bi	191	190.985786	0 \n \
83	Bi	192	191.98546	0 \n \
83	Bi	193	192.98296	0 \n \
83	Bi	194	193.98283	0 \n \
83	Bi	195	194.980651	0 \n \
83	Bi	196	195.980667	0 \n \
83	Bi	197	196.978864	0 \n \
83	Bi	198	197.97921	0 \n \
83	Bi	199	198.977672	0 \n \
83	Bi	200	199.978132	0 \n \
83	Bi	201	200.977009	0 \n \
83	Bi	202	201.977742	0 \n \
83	Bi	203	202.976876	0 \n \
83	Bi	204	203.977813	0 \n \
83	Bi	205	204.977389	0 \n \
83	Bi	206	205.978499	0 \n \
83	Bi	207	206.9784707	0 \n \
83	Bi	208	207.9797422	0 \n \
83	Bi	209	208.9803987	100 \n \
83	Bi	210	209.9841204	0 \n \
83	Bi	211	210.987269	0 \n \
83	Bi	212	211.9912857	0 \n \
83	Bi	213	212.994385	0 \n \
83	Bi	214	213.998712	0 \n \
83	Bi	215	215.00177	0 \n \
83	Bi	216	216.006306	0 \n \
83	Bi	217	217.00947	0 \n \
83	Bi	218	218.01432	0 \n \
84	Po	188	187.999422	0 \n \
84	Po	189	188.998481	0 \n \
84	Po	190	189.995101	0 \n \
84	Po	191	190.994574	0 \n \
84	Po	192	191.991335	0 \n \
84	Po	193	192.99103	0 \n \
84	Po	194	193.988186	0 \n \
84	Po	195	194.98811	0 \n \
84	Po	196	195.985535	0 \n \
84	Po	197	196.98566	0 \n \
84	Po	198	197.983389	0 \n \
84	Po	199	198.983666	0 \n \
84	Po	200	199.981799	0 \n",
    "84	Po	201	200.98226	0 \n \
84	Po	202	201.980758	0 \n \
84	Po	203	202.98142	0 \n \
84	Po	204	203.980318	0 \n \
84	Po	205	204.981203	0 \n \
84	Po	206	205.980481	0 \n \
84	Po	207	206.981593	0 \n \
84	Po	208	207.9812457	0 \n \
84	Po	209	208.9824304	0 \n \
84	Po	210	209.9828737	0 \n \
84	Po	211	210.9866532	0 \n \
84	Po	212	211.988868	0 \n \
84	Po	213	212.992857	0 \n \
84	Po	214	213.9952014	0 \n \
84	Po	215	214.99942	0 \n \
84	Po	216	216.001915	0 \n \
84	Po	217	217.006335	0 \n \
84	Po	218	218.008973	0 \n \
84	Po	219	219.01374	0 \n \
84	Po	220	220.0166	0 \n \
85	At	193	192.99984	0 \n \
85	At	194	193.99873	0 \n \
85	At	195	194.996268	0 \n \
85	At	196	195.99579	0 \n \
85	At	197	196.99319	0 \n \
85	At	198	197.99284	0 \n \
85	At	199	198.99053	0 \n \
85	At	200	199.990351	0 \n \
85	At	201	200.988417	0 \n \
85	At	202	201.98863	0 \n \
85	At	203	202.986942	0 \n \
85	At	204	203.987251	0 \n \
85	At	205	204.986074	0 \n \
85	At	206	205.986667	0 \n \
85	At	207	206.985784	0 \n \
85	At	208	207.98659	0 \n \
85	At	209	208.986173	0 \n \
85	At	210	209.987148	0 \n \
85	At	211	210.9874963	0 \n \
85	At	212	211.990745	0 \n \
85	At	213	212.992937	0 \n \
85	At	214	213.996372	0 \n \
85	At	215	214.998653	0 \n \
85	At	216	216.002423	0 \n \
85	At	217	217.004719	0 \n \
85	At	218	218.008694	0 \n \
85	At	219	219.011162	0 \n \
85	At	220	220.01541	0 \n \
85	At	221	221.01805	0 \n \
85	At	222	222.02233	0 \n \
85	At	223	223.02519	0 \n",
    "86	Rn	195	195.00544	0 \n \
86	Rn	196	196.002115	0 \n \
86	Rn	197	197.00158	0 \n \
86	Rn	198	197.998679	0 \n \
86	Rn	199	198.99837	0 \n \
86	Rn	200	199.995699	0 \n \
86	Rn	201	200.99563	0 \n \
86	Rn	202	201.993263	0 \n \
86	Rn	203	202.993387	0 \n \
86	Rn	204	203.991429	0 \n \
86	Rn	205	204.99172	0 \n \
86	Rn	206	205.990214	0 \n \
86	Rn	207	206.990734	0 \n \
86	Rn	208	207.989642	0 \n \
86	Rn	209	208.990415	0 \n \
86	Rn	210	209.989696	0 \n \
86	Rn	211	210.990601	0 \n \
86	Rn	212	211.990704	0 \n \
86	Rn	213	212.993883	0 \n \
86	Rn	214	213.995363	0 \n \
86	Rn	215	214.998745	0 \n \
86	Rn	216	216.000274	0 \n \
86	Rn	217	217.003928	0 \n \
86	Rn	218	218.0056013	0 \n \
86	Rn	219	219.0094802	0 \n \
86	Rn	220	220.011394	0 \n \
86	Rn	221	221.015537	0 \n \
86	Rn	222	222.0175777	0 \n \
86	Rn	223	223.02179	0 \n \
86	Rn	224	224.02409	0 \n \
86	Rn	225	225.02844	0 \n \
86	Rn	226	226.03089	0 \n \
86	Rn	227	227.03541	0 \n \
86	Rn	228	228.03799	0 \n \
87	Fr	199	199.00726	0 \n \
87	Fr	200	200.00657	0 \n \
87	Fr	201	201.00386	0 \n \
87	Fr	202	202.00337	0 \n \
87	Fr	203	203.000925	0 \n \
87	Fr	204	204.000653	0 \n \
87	Fr	205	204.998594	0 \n \
87	Fr	206	205.99867	0 \n \
87	Fr	207	206.99695	0 \n \
87	Fr	208	207.99714	0 \n \
87	Fr	209	208.995954	0 \n \
87	Fr	210	209.996408	0 \n \
87	Fr	211	210.995537	0 \n \
87	Fr	212	211.996202	0 \n \
87	Fr	213	212.996189	0 \n \
87	Fr	214	213.998971	0 \n \
87	Fr	215	215.000341	0 \n",
    "87	Fr	216	216.003198	0 \n \
87	Fr	217	217.004632	0 \n \
87	Fr	218	218.007578	0 \n \
87	Fr	219	219.009252	0 \n \
87	Fr	220	220.012327	0 \n \
87	Fr	221	221.014255	0 \n \
87	Fr	222	222.017552	0 \n \
87	Fr	223	223.0197359	0 \n \
87	Fr	224	224.02325	0 \n \
87	Fr	225	225.02557	0 \n \
87	Fr	226	226.02939	0 \n \
87	Fr	227	227.03184	0 \n \
87	Fr	228	228.03573	0 \n \
87	Fr	229	229.03845	0 \n \
87	Fr	230	230.04251	0 \n \
87	Fr	231	231.04544	0 \n \
87	Fr	232	232.04977	0 \n \
88	Ra	202	202.00989	0 \n \
88	Ra	203	203.00927	0 \n \
88	Ra	204	204.0065	0 \n \
88	Ra	205	205.00627	0 \n \
88	Ra	206	206.003827	0 \n \
88	Ra	207	207.0038	0 \n \
88	Ra	208	208.00184	0 \n \
88	Ra	209	209.00199	0 \n \
88	Ra	210	210.000495	0 \n \
88	Ra	211	211.000898	0 \n \
88	Ra	212	211.999794	0 \n \
88	Ra	213	213.000384	0 \n \
88	Ra	214	214.000108	0 \n \
88	Ra	215	215.00272	0 \n \
88	Ra	216	216.003533	0 \n \
88	Ra	217	217.00632	0 \n \
88	Ra	218	218.00714	0 \n \
88	Ra	219	219.010085	0 \n \
88	Ra	220	220.011028	0 \n \
88	Ra	221	221.013917	0 \n \
88	Ra	222	222.015375	0 \n \
88	Ra	223	223.0185022	0 \n \
88	Ra	224	224.0202118	0 \n \
88	Ra	225	225.023612	0 \n \
88	Ra	226	226.0254098	0 \n \
88	Ra	227	227.0291778	0 \n \
88	Ra	228	228.0310703	0 \n \
88	Ra	229	229.034958	0 \n \
88	Ra	230	230.037056	0 \n \
88	Ra	231	231.04122	0 \n \
88	Ra	232	232.04364	0 \n \
88	Ra	233	233.04806	0 \n \
88	Ra	234	234.0507	0 \n \
89	Ac	206	206.0145	0 \n",
    "89	Ac	207	207.01195	0 \n \
89	Ac	208	208.01155	0 \n \
89	Ac	209	209.00949	0 \n \
89	Ac	210	210.00944	0 \n \
89	Ac	211	211.00773	0 \n \
89	Ac	212	212.00781	0 \n \
89	Ac	213	213.00661	0 \n \
89	Ac	214	214.006902	0 \n \
89	Ac	215	215.006454	0 \n \
89	Ac	216	216.00872	0 \n \
89	Ac	217	217.009347	0 \n \
89	Ac	218	218.01164	0 \n \
89	Ac	219	219.01242	0 \n \
89	Ac	220	220.014763	0 \n \
89	Ac	221	221.01559	0 \n \
89	Ac	222	222.017844	0 \n \
89	Ac	223	223.019137	0 \n \
89	Ac	224	224.021723	0 \n \
89	Ac	225	225.02323	0 \n \
89	Ac	226	226.026098	0 \n \
89	Ac	227	227.0277521	0 \n \
89	Ac	228	228.0310211	0 \n \
89	Ac	229	229.03302	0 \n \
89	Ac	230	230.03629	0 \n \
89	Ac	231	231.03856	0 \n \
89	Ac	232	232.04203	0 \n \
89	Ac	233	233.04455	0 \n \
89	Ac	234	234.04842	0 \n \
89	Ac	235	235.05123	0 \n \
89	Ac	236	236.0553	0 \n \
90	Th	209	209.01772	0 \n \
90	Th	210	210.015075	0 \n \
90	Th	211	211.01493	0 \n \
90	Th	212	212.01298	0 \n \
90	Th	213	213.01301	0 \n \
90	Th	214	214.0115	0 \n \
90	Th	215	215.01173	0 \n \
90	Th	216	216.011062	0 \n \
90	Th	217	217.013114	0 \n \
90	Th	218	218.013284	0 \n \
90	Th	219	219.01554	0 \n \
90	Th	220	220.015748	0 \n \
90	Th	221	221.018184	0 \n \
90	Th	222	222.018468	0 \n \
90	Th	223	223.020811	0 \n \
90	Th	224	224.021467	0 \n \
90	Th	225	225.023951	0 \n \
90	Th	226	226.024903	0 \n \
90	Th	227	227.0277041	0 \n \
90	Th	228	228.0287411	0 \n \
90	Th	229	229.031762	0 \n",
    "90	Th	230	230.0331338	0 \n \
90	Th	231	231.0363043	0 \n \
90	Th	232	232.0380553	100 \n \
90	Th	233	233.0415818	0 \n \
90	Th	234	234.043601	0 \n \
90	Th	235	235.04751	0 \n \
90	Th	236	236.04987	0 \n \
90	Th	237	237.05389	0 \n \
90	Th	238	238.0565	0 \n \
91	Pa	212	212.0232	0 \n \
91	Pa	213	213.02111	0 \n \
91	Pa	214	214.02092	0 \n \
91	Pa	215	215.01919	0 \n \
91	Pa	216	216.01911	0 \n \
91	Pa	217	217.01832	0 \n \
91	Pa	218	218.020042	0 \n \
91	Pa	219	219.01988	0 \n \
91	Pa	220	220.02188	0 \n \
91	Pa	221	221.02188	0 \n \
91	Pa	222	222.02374	0 \n \
91	Pa	223	223.02396	0 \n \
91	Pa	224	224.025626	0 \n \
91	Pa	225	225.02613	0 \n \
91	Pa	226	226.027948	0 \n \
91	Pa	227	227.028805	0 \n \
91	Pa	228	228.031051	0 \n \
91	Pa	229	229.0320968	0 \n \
91	Pa	230	230.034541	0 \n \
91	Pa	231	231.035884	100 \n \
91	Pa	232	232.038592	0 \n \
91	Pa	233	233.0402473	0 \n \
91	Pa	234	234.043308	0 \n \
91	Pa	235	235.04544	0 \n \
91	Pa	236	236.04868	0 \n \
91	Pa	237	237.05115	0 \n \
91	Pa	238	238.0545	0 \n \
91	Pa	239	239.05726	0 \n \
91	Pa	240	240.06098	0 \n \
92	U	217	217.02437	0 \n \
92	U	218	218.02354	0 \n \
92	U	219	219.02492	0 \n \
92	U	220	220.02472	0 \n \
92	U	221	221.0264	0 \n \
92	U	222	222.02609	0 \n \
92	U	223	223.02774	0 \n \
92	U	224	224.027605	0 \n \
92	U	225	225.029391	0 \n \
92	U	226	226.029339	0 \n \
92	U	227	227.031156	0 \n \
92	U	228	228.031374	0 \n \
92	U	229	229.033506	0 \n",
    "92	U	230	230.03394	0 \n \
92	U	231	231.036294	0 \n \
92	U	232	232.0371562	0 \n \
92	U	233	233.0396352	0 \n \
92	U	234	234.0409521	0.0055 \n \
92	U	235	235.0439299	0.7200 \n \
92	U	236	236.045568	0 \n \
92	U	237	237.0487302	0 \n \
92	U	238	238.0507882	99.2745 \n \
92	U	239	239.0542933	0 \n \
92	U	240	240.056592	0 \n \
92	U	241	241.06033	0 \n \
92	U	242	242.06293	0 \n \
93	Np	225	225.03391	0 \n \
93	Np	226	226.03515	0 \n \
93	Np	227	227.03496	0 \n \
93	Np	228	228.03618	0 \n \
93	Np	229	229.03626	0 \n \
93	Np	230	230.03783	0 \n \
93	Np	231	231.03825	0 \n \
93	Np	232	232.04011	0 \n \
93	Np	233	233.04074	0 \n \
93	Np	234	234.042895	0 \n \
93	Np	235	235.0440633	0 \n \
93	Np	236	236.04657	0 \n \
93	Np	237	237.0481734	0 \n \
93	Np	238	238.0509464	0 \n \
93	Np	239	239.052939	0 \n \
93	Np	240	240.056162	0 \n \
93	Np	241	241.05825	0 \n \
93	Np	242	242.06164	0 \n \
93	Np	243	243.06428	0 \n \
93	Np	244	244.06785	0 \n \
94	Pu	228	228.03874	0 \n \
94	Pu	229	229.04015	0 \n \
94	Pu	230	230.03965	0 \n \
94	Pu	231	231.041101	0 \n \
94	Pu	232	232.041187	0 \n \
94	Pu	233	233.043	0 \n \
94	Pu	234	234.043317	0 \n \
94	Pu	235	235.045286	0 \n \
94	Pu	236	236.046058	0 \n \
94	Pu	237	237.0484097	0 \n \
94	Pu	238	238.0495599	0 \n \
94	Pu	239	239.0521634	0 \n \
94	Pu	240	240.0538135	0 \n \
94	Pu	241	241.0568515	0 \n \
94	Pu	242	242.0587426	0 \n \
94	Pu	243	243.062003	0 \n \
94	Pu	244	244.064204	0 \n \
94	Pu	245	245.067747	0 \n",
    "94	Pu	246	246.070205	0 \n \
94	Pu	247	247.07407	0 \n \
95	Am	231	231.04556	0 \n \
95	Am	232	232.04659	0 \n \
95	Am	233	233.04635	0 \n \
95	Am	234	234.04781	0 \n \
95	Am	235	235.04795	0 \n \
95	Am	236	236.04958	0 \n \
95	Am	237	237.05	0 \n \
95	Am	238	238.05198	0 \n \
95	Am	239	239.0530245	0 \n \
95	Am	240	240.0553	0 \n \
95	Am	241	241.0568291	0 \n \
95	Am	242	242.0595492	0 \n \
95	Am	243	243.0613811	0 \n \
95	Am	244	244.0642848	0 \n \
95	Am	245	245.066452	0 \n \
95	Am	246	246.069775	0 \n \
95	Am	247	247.07209	0 \n \
95	Am	248	248.07575	0 \n \
95	Am	249	249.07848	0 \n \
96	Cm	233	233.05077	0 \n \
96	Cm	234	234.05016	0 \n \
96	Cm	235	235.05143	0 \n \
96	Cm	236	236.05141	0 \n \
96	Cm	237	237.0529	0 \n \
96	Cm	238	238.05303	0 \n \
96	Cm	239	239.05496	0 \n \
96	Cm	240	240.0555295	0 \n \
96	Cm	241	241.057653	0 \n \
96	Cm	242	242.0588358	0 \n \
96	Cm	243	243.0613891	0 \n \
96	Cm	244	244.0627526	0 \n \
96	Cm	245	245.0654912	0 \n \
96	Cm	246	246.0672237	0 \n \
96	Cm	247	247.070354	0 \n \
96	Cm	248	248.072349	0 \n \
96	Cm	249	249.075953	0 \n \
96	Cm	250	250.078357	0 \n \
96	Cm	251	251.082285	0 \n \
96	Cm	252	252.08487	0 \n \
97	Bk	235	235.05658	0 \n \
97	Bk	236	236.05733	0 \n \
97	Bk	237	237.057	0 \n \
97	Bk	238	238.05828	0 \n \
97	Bk	239	239.05828	0 \n \
97	Bk	240	240.05976	0 \n \
97	Bk	241	241.06023	0 \n \
97	Bk	242	242.06198	0 \n \
97	Bk	243	243.063008	0 \n \
97	Bk	244	244.065181	0 \n",
    "97	Bk	245	245.0663616	0 \n \
97	Bk	246	246.06867	0 \n \
97	Bk	247	247.070307	0 \n \
97	Bk	248	248.07309	0 \n \
97	Bk	249	249.0749867	0 \n \
97	Bk	250	250.078317	0 \n \
97	Bk	251	251.08076	0 \n \
97	Bk	252	252.08431	0 \n \
97	Bk	253	253.08688	0 \n \
97	Bk	254	254.0906	0 \n \
98	Cf	237	237.06207	0 \n \
98	Cf	238	238.06141	0 \n \
98	Cf	239	239.06242	0 \n \
98	Cf	240	240.0623	0 \n \
98	Cf	241	241.06373	0 \n \
98	Cf	242	242.0637	0 \n \
98	Cf	243	243.06543	0 \n \
98	Cf	244	244.066001	0 \n \
98	Cf	245	245.068049	0 \n \
98	Cf	246	246.0688053	0 \n \
98	Cf	247	247.071001	0 \n \
98	Cf	248	248.072185	0 \n \
98	Cf	249	249.0748535	0 \n \
98	Cf	250	250.0764061	0 \n \
98	Cf	251	251.079587	0 \n \
98	Cf	252	252.081626	0 \n \
98	Cf	253	253.085133	0 \n \
98	Cf	254	254.087323	0 \n \
98	Cf	255	255.09105	0 \n \
98	Cf	256	256.09344	0 \n \
99	Es	240	240.06892	0 \n \
99	Es	241	241.06854	0 \n \
99	Es	242	242.06975	0 \n \
99	Es	243	243.06955	0 \n \
99	Es	244	244.07088	0 \n \
99	Es	245	245.07132	0 \n \
99	Es	246	246.0729	0 \n \
99	Es	247	247.07366	0 \n \
99	Es	248	248.07547	0 \n \
99	Es	249	249.07641	0 \n \
99	Es	250	250.07861	0 \n \
99	Es	251	251.079992	0 \n \
99	Es	252	252.08298	0 \n \
99	Es	253	253.0848247	0 \n \
99	Es	254	254.088022	0 \n \
99	Es	255	255.090273	0 \n \
99	Es	256	256.0936	0 \n \
99	Es	257	257.09598	0 \n \
99	Es	258	258.09952	0 \n \
100	Fm	242	242.07343	0 \n \
100	Fm	243	243.07435	0 \n",
    "100	Fm	244	244.07408	0 \n \
100	Fm	245	245.07539	0 \n \
100	Fm	246	246.0753	0 \n \
100	Fm	247	247.07685	0 \n \
100	Fm	248	248.077195	0 \n \
100	Fm	249	249.07903	0 \n \
100	Fm	250	250.079521	0 \n \
100	Fm	251	251.081575	0 \n \
100	Fm	252	252.082467	0 \n \
100	Fm	253	253.085185	0 \n \
100	Fm	254	254.0868542	0 \n \
100	Fm	255	255.089962	0 \n \
100	Fm	256	256.091773	0 \n \
100	Fm	257	257.095105	0 \n \
100	Fm	258	258.09708	0 \n \
100	Fm	259	259.1006	0 \n \
100	Fm	260	260.10268	0 \n \
101	Md	245	245.08083	0 \n \
101	Md	246	246.08189	0 \n \
101	Md	247	247.08164	0 \n \
101	Md	248	248.08282	0 \n \
101	Md	249	249.08301	0 \n \
101	Md	250	250.08442	0 \n \
101	Md	251	251.08484	0 \n \
101	Md	252	252.08656	0 \n \
101	Md	253	253.08728	0 \n \
101	Md	254	254.08966	0 \n \
101	Md	255	255.091083	0 \n \
101	Md	256	256.09406	0 \n \
101	Md	257	257.095541	0 \n \
101	Md	258	258.098431	0 \n \
101	Md	259	259.10051	0 \n \
101	Md	260	260.10365	0 \n \
101	Md	261	261.10572	0 \n \
101	Md	262	262.10887	0 \n \
102	No	248	248.0866	0 \n \
102	No	249	249.08783	0 \n \
102	No	250	250.08751	0 \n \
102	No	251	251.08901	0 \n \
102	No	252	252.088977	0 \n \
102	No	253	253.09068	0 \n \
102	No	254	254.090955	0 \n \
102	No	255	255.093241	0 \n \
102	No	256	256.094283	0 \n \
102	No	257	257.096877	0 \n \
102	No	258	258.09821	0 \n \
102	No	259	259.10103	0 \n \
102	No	260	260.10264	0 \n \
102	No	261	261.10575	0 \n \
102	No	262	262.1073	0 \n \
102	No	263	263.11055	0 \n",
    "102	No	264	264.11235	0 \n \
103	Lr	251	251.09436	0 \n \
103	Lr	252	252.09537	0 \n \
103	Lr	253	253.09521	0 \n \
103	Lr	254	254.09645	0 \n \
103	Lr	255	255.09668	0 \n \
103	Lr	256	256.09863	0 \n \
103	Lr	257	257.09956	0 \n \
103	Lr	258	258.10181	0 \n \
103	Lr	259	259.1029	0 \n \
103	Lr	260	260.1055	0 \n \
103	Lr	261	261.10688	0 \n \
103	Lr	262	262.10963	0 \n \
103	Lr	263	263.11129	0 \n \
103	Lr	264	264.11404	0 \n \
103	Lr	265	265.11584	0 \n \
103	Lr	266	266.11931	0 \n \
104	Rf	253	253.10069	0 \n \
104	Rf	254	254.10018	0 \n \
104	Rf	255	255.10134	0 \n \
104	Rf	256	256.101166	0 \n \
104	Rf	257	257.10299	0 \n \
104	Rf	258	258.10349	0 \n \
104	Rf	259	259.10564	0 \n \
104	Rf	260	260.10644	0 \n \
104	Rf	261	261.10877	0 \n \
104	Rf	262	262.10993	0 \n \
104	Rf	263	263.11255	0 \n \
104	Rf	264	264.11399	0 \n \
104	Rf	265	265.1167	0 \n \
104	Rf	266	266.11796	0 \n \
104	Rf	267	267.12153	0 \n \
104	Rf	268	268.12364	0 \n",
    "110	Ds	267	267.14434	0 \n \
110	Ds	268	268.1438	0 \n \
110	Ds	269	269.14512	0 \n \
110	Ds	270	270.14472	0 \n \
110	Ds	271	271.14606	0 \n \
110	Ds	272	272.14632	0 \n \
110	Ds	273	273.14886	0 \n \
110	Ds	274	274.14949	0 \n \
110	Ds	275	275.15218	0 \n \
110	Ds	276	276.15303	0 \n \
110	Ds	277	277.15565	0 \n \
110	Ds	278	278.15647	0 \n \
110	Ds	279	279.15886	0 \n \
110	Ds	280	280.1598	0 \n \
110	Ds	281	281.16206	0 \n",
    // from NIST
    // https://www.nist.gov/pml/atomic-weights-and-isotopic-compositions-relative-atomic-masses
    // 17.09.2017
    "111 Rg  272  272.15327 0 \n \
111 Rg  273  273.15313 0 \n \
111 Rg  274  274.15525 0 \n \
111 Rg  275  275.15594 0 \n \
111 Rg  276  276.15833 0 \n \
111 Rg  277  277.15907 0 \n \
111 Rg  278  278.16149 0 \n \
111 Rg  279  279.16272 0 \n \
111 Rg  280  280.16514 0 \n \
111 Rg  281  281.16636 0 \n \
111 Rg  282  282.16912 0 \n \
111 Rg  283  283.17054 0 \n \
112 Cn  276  276.16141 0 \n \
112 Cn  277  277.16364 0 \n \
112 Cn  278  278.16416 0 \n \
112 Cn  279  279.16654 0 \n \
112 Cn  280  280.16715 0 \n \
112 Cn  281  281.16975 0 \n \
112 Cn  282  282.17050 0 \n \
112 Cn  283  283.17327 0 \n \
112 Cn  284  284.17416 0 \n \
112 Cn  285  285.17712 0 \n \
113 Nh  278  278.17058 0 \n \
113 Nh  279  279.17095 0 \n \
113 Nh  280  280.17293 0 \n \
113 Nh  281  281.17348 0 \n \
113 Nh  282  282.17567 0 \n \
113 Nh  283  283.17657 0 \n \
113 Nh  284  284.17873 0 \n \
113 Nh  285  285.17973 0 \n \
113 Nh  286  286.18221 0 \n \
113 Nh  287  287.18339 0 \n \
114 Fl  285  285.18364 0 \n \
114 Fl  286  286.18423 0 \n \
114 Fl  287  287.18678 0 \n \
114 Fl  288  288.18757 0 \n \
114 Fl  289  289.19042 0 \n \
115 Mc  287  287.19070 0 \n \
115 Mc  288  288.19274 0 \n \
115 Mc  289  289.19363 0 \n \
115 Mc  290  290.19598 0 \n \
115 Mc  291  291.19707 0 \n \
116 Lv  289  289.19816 0 \n \
116 Lv  290  290.19864 0 \n \
116 Lv  291  291.20108 0 \n \
116 Lv  292  292.20174 0 \n \
116 Lv  293  293.20449 0 \n \
117 Ts  291  291.20553 0 \n \
117 Ts  292  292.20746 0 \n \
117 Ts  293  293.20824 0 \n \
117 Ts  294  294.21046 0 \n \
118 Og  293  293.21356 0 \n \
118 Og  294  294.21392 0 \n \
118 Og  295  295.21624 0 \n",
    "EOS"};
}  // namespace RDKit
