from nplinker.genomics import BGC
from nplinker.genomics import GCF
from nplinker.strain import Strain


def test_default():
    bgc = BGC("BGC0000001", "Polyketide")
    assert bgc.id == "BGC0000001"
    assert bgc.product_prediction == ("Polyketide",)
    assert bgc.is_mibig() is True
    assert bgc.parents == set()
    assert bgc.bigscape_classes == set()
    assert bgc.strain is None

    strain = Strain("sample_strain")
    bgc.strain = strain
    assert bgc.strain == strain


def test_add_and_detach_parent():
    bgc = BGC("BGC0000001", "Polyketide")
    gcf = GCF("1")
    bgc.add_parent(gcf)
    assert bgc.parents == {gcf}
    bgc.detach_parent(gcf)
    assert bgc.parents == set()


def test_to_dict():
    bgc = BGC("BGC0000001", "Polyketide", "NRP")
    bgc.strain = Strain("sample_strain")
    bgc.description = "Sample description"

    dict_repr = bgc.to_dict()
    assert dict_repr["GCF_id"] == list()
    assert dict_repr["GCF_bigscape_class"] == list()
    assert dict_repr["BGC_name"] == "BGC0000001"
    assert dict_repr["product_prediction"] == ["Polyketide", "NRP"]
    assert dict_repr["mibig_bgc_class"] is None
    assert dict_repr["description"] == "Sample description"
    assert dict_repr["strain_id"] == "sample_strain"
    assert dict_repr["antismash_id"] is None
    assert dict_repr["antismash_region"] is None

    bgc.add_parent(GCF("1"))
    bgc.mibig_bgc_class = [
        "NRP",
    ]
    bgc.antismash_id = "ABC_0001"
    bgc.antismash_region = 1
    dict_repr = bgc.to_dict()
    assert dict_repr["GCF_id"] == [
        "1",
    ]
    assert dict_repr["GCF_bigscape_class"] == list()
    assert dict_repr["mibig_bgc_class"] == [
        "NRP",
    ]
    assert dict_repr["antismash_id"] == "ABC_0001"
    assert dict_repr["antismash_region"] == 1


def test__to_string():
    assert BGC._to_string([]) == ""
    assert BGC._to_string([1, 2.0, "a"]) == "1, 2.0, a"
    assert BGC._to_string(dict()) == ""
    assert BGC._to_string({"key1": 1, "key2": "value2"}) == "key1:1, key2:value2"
    assert BGC._to_string(None) == ""
    assert BGC._to_string(0) == "0"
    assert BGC._to_string(0.0) == "0.0"
    assert BGC._to_string(100.2) == "100.2"
    assert BGC._to_string(False) == "False"


def test_to_tabular():
    bgc = BGC("BGC0000001", "Polyketide", "NRP")
    bgc.strain = Strain("sample_strain")
    bgc.description = "Sample description"

    tabular_repr = bgc.to_tabular()
    assert tabular_repr["GCF_id"] == ""
    assert tabular_repr["GCF_bigscape_class"] == ""
    assert tabular_repr["BGC_name"] == "BGC0000001"
    assert tabular_repr["product_prediction"] == "Polyketide, NRP"
    assert tabular_repr["mibig_bgc_class"] == ""
    assert tabular_repr["description"] == "Sample description"
    assert tabular_repr["strain_id"] == "sample_strain"
    assert tabular_repr["antismash_id"] == ""
    assert tabular_repr["antismash_region"] == ""

    bgc.add_parent(GCF("1"))
    bgc.mibig_bgc_class = [
        "NRP",
    ]
    bgc.antismash_id = "ABC_0001"
    bgc.antismash_region = 1
    tabular_repr = bgc.to_tabular()
    assert tabular_repr["GCF_id"] == "1"
    assert tabular_repr["GCF_bigscape_class"] == ""
    assert tabular_repr["mibig_bgc_class"] == "NRP"
    assert tabular_repr["antismash_id"] == "ABC_0001"
    assert tabular_repr["antismash_region"] == "1"
