<!--
---
weight: 107
title: "Configuring PyneCore"
description: "Learn how to configure PyneCore for your specific needs"
icon: "settings"
date: "2025-03-31"
lastmod: "2025-03-31"
draft: false
toc: true
categories: ["Overview", "Configuration"]
tags: ["configuration", "settings", "toml", "workdir", "symbol-config", "script-config"]
---
-->

# Configuring PyneCore

PyneCore provides a simple yet powerful configuration system that is primarily based on TOML files. This guide will help you understand the key configuration components and how to use them effectively.

## Working Directory Structure

The working directory is where PyneCore stores data, configuration files, output files, and scripts. A typical PyneCore working directory looks like this:

```
workdir/
├── config/           # Configuration files
├── data/             # Data files (.ohlcv files and their .toml companions)
├── scripts/          # Your Pyne scripts (.py files and their .toml companions)
└── output/           # Script output (plots, strategy statistics, equity curves)
```

PyneCore automatically discovers the working directory by:
1. Looking for a `workdir` directory in the current directory
2. Searching up to 10 levels in parent directories
3. If not found, using or creating a `workdir` in the current directory

## Configuration Files

PyneCore uses two types of TOML configuration files:

1. **Symbol configuration files** - Located alongside the `.ohlcv` data files
2. **Script configuration files** - Located alongside your Pyne scripts

### Symbol Configuration

Each OHLCV data file has a corresponding TOML file with the same name (but `.toml` extension) that contains information about the symbol. This information is used by:

- The PyneCore runtime system to handle timeframe and session calculations
- Your scripts via the `syminfo` module to access symbol properties

Example of a symbol configuration file (`capitalcom_EURUSD_15.toml`):

```toml
[symbol]
prefix = "CAPITALCOM"
description = "EUR/USD"
ticker = "EURUSD"
currency = "USD"
basecurrency = "EUR"
period = "15"
type = "forex"
mintick = 0.00001000
pricescale = 100000
minmove = 1.00000000
pointvalue = 1
timezone = "US/Eastern"
volumetype = "base"
avg_spread = 0.00007430

# Opening hours
[[opening_hours]]
day = 0
start = "00:00:00"
end = "16:59:50"

# Many more opening hours entries...

# Session starts
[[session_starts]]
day = 0
time = "17:05:00"

# More session entries...
```

The symbol configuration is initially created by the data provider when you download data, but you can modify it if needed. This is particularly useful when you need to:

- Adjust session times to match TradingView's behavior for backtesting consistency
- Fix timezone information for a particular exchange
- Set the correct point value or tick size for a symbol

### Script Configuration

Each script (indicator or strategy) can have a corresponding TOML file that controls its settings and input parameters. This file is automatically generated when you run a script, and you can modify it to change settings without editing the script.

Example of a script configuration file (`barupdown.toml`):

```toml
# Indicator / Strategy / Library Settings

[script]
overlay = false
behind_chart = true
# Many more script settings...

# Input Settings

[inputs.maxIdLossPcnt]
# Input metadata, cannot be modified
#       title: "Max Intraday Loss(%)"
#      inline: false
#     confirm: false
# Change here to modify the input value
#value =
```

The script configuration file has two main sections:

1. **Script settings** (`[script]`) - Control how your script executes:
   - For indicators: overlay, format, max_lines_count, etc.
   - For strategies: initial_capital, commission_value, slippage, etc.

2. **Input settings** (`[inputs.XXX]`) - Control the input values for your script:
   - Each input defined in your script (using `lib.input` functions) gets its own section
   - The metadata is displayed as comments and cannot be modified
   - You can set the `value` to override the default value

## How Configuration is Used

### Symbol Configuration Usage

The symbol configuration is used in several ways:

1. By the runtime system to determine:
   - When a bar is the first bar of a session
   - How to handle timeframes
   - Converting between ticks, points, and price

2. By your scripts via the `syminfo` module:
   ```python
   from pynecore.lib import syminfo

   def main():
       # Access symbol configuration
       print(f"Trading {syminfo.ticker} on {syminfo.prefix}")
       print(f"Minimum tick: {syminfo.mintick}")

       # Check if we're in a session
       if syminfo.session.ismarket:
           # Do something during market hours
           pass
   ```

### Script Configuration Usage

The script configuration is used to:

1. Control script behavior through the `@script.indicator` or `@script.strategy` decorator:
   ```python
   @script.indicator(
       title="My Indicator",
       overlay=True,
       format=lib.format.price,
       precision=2
   )
   def main():
       # These settings can be overridden in the TOML file
       pass
   ```

2. Provide input values through the `lib.input` functions:
   ```python
   def main():
       # This value can be overridden in the TOML file
       length = lib.input.int(14, title="MA Length")
       source = lib.input.source(lib.close, title="Source")
   ```

The script loads its configuration from the TOML file (if it exists) when it starts running. Any changes to the TOML file will take effect the next time you run the script.

## Modifying Configuration Files

When modifying symbol configuration files, be careful to maintain the correct format, especially for:
- Opening hours (which define when the market is open)
- Session starts and ends (which define when a new trading session begins)
- Technical parameters like mintick and pointvalue (which affect price calculations)

When modifying script configuration files:
- Only change the `value =` lines under input sections
- Only uncomment and change script settings that you need to modify
- The script will save its configuration with updated values when it runs

## Environment Variables

PyneCore also recognizes a few environment variables:

- `PYNE_NO_LOGO`: If set, suppresses the PyneCore logo at startup
- `PYNE_QUIET`: Suppresses all non-essential output
- `PYNE_SAVE_SCRIPT_TOML`: Set to "0" to disable saving script TOML files (default: "1")
