// Copyright (C) 2019-2025 Provable Inc.
// This file is part of the Leo library.

// The Leo library is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// The Leo library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with the Leo library. If not, see <https://www.gnu.org/licenses/>.

use super::*;
use leo_errors::{ParserError, Result};

use leo_span::sym;
use snarkvm::{
    console::account::Address,
    prelude::{CanaryV0, MainnetV0, TestnetV0},
};

const INT_TYPES: &[Token] = &[
    Token::I8,
    Token::I16,
    Token::I32,
    Token::I64,
    Token::I128,
    Token::U8,
    Token::U16,
    Token::U32,
    Token::U64,
    Token::U128,
    Token::Field,
    Token::Group,
    Token::Scalar,
];

impl ParserContext<'_> {
    /// Returns an [`Expression`] AST node if the next token is an expression.
    /// Includes struct init expressions.
    pub(crate) fn parse_expression(&mut self) -> Result<Expression> {
        // Store current parser state.
        let prior_fuzzy_state = self.disallow_struct_construction;

        // Allow struct init expressions.
        self.disallow_struct_construction = false;

        // Parse expression.
        let result = self.parse_conditional_expression();

        // Restore prior parser state.
        self.disallow_struct_construction = prior_fuzzy_state;

        result
    }

    /// Returns an [`Expression`] AST node if the next tokens represent
    /// a ternary expression. May or may not include struct init expressions.
    ///
    /// Otherwise, tries to parse the next token using [`parse_boolean_or_expression`].
    pub(super) fn parse_conditional_expression(&mut self) -> Result<Expression> {
        // Try to parse the next expression. Try BinaryOperation::Or.
        let mut expr = self.parse_boolean_or_expression()?;

        // Parse the rest of the ternary expression.
        if self.eat(&Token::Question) {
            let if_true = self.parse_expression()?;
            self.expect(&Token::Colon)?;
            let if_false = self.parse_expression()?;
            expr = TernaryExpression {
                span: expr.span() + if_false.span(),
                condition: expr,
                if_true,
                if_false,
                id: self.node_builder.next_id(),
            }
            .into();
        }
        Ok(expr)
    }

    /// Constructs a binary expression `left op right`.
    fn bin_expr(node_builder: &NodeBuilder, left: Expression, right: Expression, op: BinaryOperation) -> Expression {
        BinaryExpression { span: left.span() + right.span(), op, left, right, id: node_builder.next_id() }.into()
    }

    /// Parses a left-associative binary expression `<left> token <right>` using `f` for left/right.
    /// The `token` is translated to `op` in the AST.
    fn parse_bin_expr(
        &mut self,
        tokens: &[Token],
        mut f: impl FnMut(&mut Self) -> Result<Expression>,
    ) -> Result<Expression> {
        let mut expr = f(self)?;
        while let Some(op) = self.eat_bin_op(tokens) {
            expr = Self::bin_expr(self.node_builder, expr, f(self)?, op);
        }
        Ok(expr)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent
    /// a binary OR expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_boolean_and_expression`].
    fn parse_boolean_or_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::Or], Self::parse_boolean_and_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// binary AND expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_equality_expression`].
    fn parse_boolean_and_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::And], Self::parse_equality_expression)
    }

    fn eat_unary_op(&mut self) -> Option<UnaryOperation> {
        self.eat_any(&[Token::Not, Token::Sub]).then(|| match &self.prev_token.token {
            Token::Not => UnaryOperation::Not,
            Token::Sub => UnaryOperation::Negate,
            _ => panic!("Can't happen."),
        })
    }

    /// Eats one of binary operators matching any in `tokens`.
    fn eat_bin_op(&mut self, tokens: &[Token]) -> Option<BinaryOperation> {
        self.eat_any(tokens).then(|| match &self.prev_token.token {
            Token::Eq => BinaryOperation::Eq,
            Token::NotEq => BinaryOperation::Neq,
            Token::Lt => BinaryOperation::Lt,
            Token::LtEq => BinaryOperation::Lte,
            Token::Gt => BinaryOperation::Gt,
            Token::GtEq => BinaryOperation::Gte,
            Token::Add => BinaryOperation::Add,
            Token::Sub => BinaryOperation::Sub,
            Token::Mul => BinaryOperation::Mul,
            Token::Div => BinaryOperation::Div,
            Token::Rem => BinaryOperation::Rem,
            Token::Or => BinaryOperation::Or,
            Token::And => BinaryOperation::And,
            Token::BitOr => BinaryOperation::BitwiseOr,
            Token::BitAnd => BinaryOperation::BitwiseAnd,
            Token::Pow => BinaryOperation::Pow,
            Token::Shl => BinaryOperation::Shl,
            Token::Shr => BinaryOperation::Shr,
            Token::BitXor => BinaryOperation::Xor,
            _ => unreachable!("`eat_bin_op` shouldn't produce this"),
        })
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// binary relational expression: less than, less than or equals, greater than, greater than or equals.
    ///
    /// Otherwise, tries to parse the next token using [`parse_additive_expression`].
    fn parse_ordering_expression(&mut self) -> Result<Expression> {
        let mut expr = self.parse_bitwise_exclusive_or_expression()?;
        if let Some(op) = self.eat_bin_op(&[Token::Lt, Token::LtEq, Token::Gt, Token::GtEq]) {
            let right = self.parse_bitwise_exclusive_or_expression()?;
            expr = Self::bin_expr(self.node_builder, expr, right, op);
        }
        Ok(expr)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// binary equals or not equals expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_ordering_expression`].
    fn parse_equality_expression(&mut self) -> Result<Expression> {
        let mut expr = self.parse_ordering_expression()?;
        if let Some(op) = self.eat_bin_op(&[Token::Eq, Token::NotEq]) {
            let right = self.parse_ordering_expression()?;
            expr = Self::bin_expr(self.node_builder, expr, right, op);
        }
        Ok(expr)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// bitwise exclusive or expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_bitwise_inclusive_or_expression`].
    fn parse_bitwise_exclusive_or_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::BitXor], Self::parse_bitwise_inclusive_or_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// bitwise inclusive or expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_bitwise_and_expression`].
    fn parse_bitwise_inclusive_or_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::BitOr], Self::parse_bitwise_and_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// bitwise and expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_shift_expression`].
    fn parse_bitwise_and_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::BitAnd], Self::parse_shift_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// shift left or a shift right expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_additive_expression`].
    fn parse_shift_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::Shl, Token::Shr], Self::parse_additive_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// binary addition or subtraction expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_mul_div_pow_expression`].
    fn parse_additive_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::Add, Token::Sub], Self::parse_multiplicative_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// binary multiplication, division, or a remainder expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_exponential_expression`].
    fn parse_multiplicative_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::Mul, Token::Div, Token::Rem], Self::parse_exponential_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// binary exponentiation expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_cast_expression`].
    fn parse_exponential_expression(&mut self) -> Result<Expression> {
        self.parse_bin_expr(&[Token::Pow], Self::parse_cast_expression)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// cast expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_unary_expression`].
    fn parse_cast_expression(&mut self) -> Result<Expression> {
        let mut expr = self.parse_unary_expression()?;
        if self.eat(&Token::As) {
            let (type_, end_span) = self.parse_primitive_type()?;
            let span = expr.span() + end_span;
            expr = CastExpression { expression: expr, type_, span, id: self.node_builder.next_id() }.into();
        }

        Ok(expr)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// unary not, negate, or bitwise not expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_postfix_expression`].
    pub(super) fn parse_unary_expression(&mut self) -> Result<Expression> {
        let token_span = self.token.span;

        let Some(op) = self.eat_unary_op() else {
            return self.parse_postfix_expression();
        };

        let mut inner = self.parse_unary_expression()?;

        // Try to construct a negative literal.
        if let UnaryOperation::Negate = op {
            use LiteralVariant::*;
            if let Expression::Literal(Literal {
                variant: Integer(_, string) | Field(string) | Group(string) | Scalar(string),
                span,
                ..
            }) = &mut inner
            {
                if !string.starts_with('-') {
                    // The operation was a negation and the literal was not already negative, so fold it in.
                    string.insert(0, '-');
                    *span = token_span + *span;
                    return Ok(inner);
                }
            }
        }

        Ok(UnaryExpression { span: token_span + inner.span(), op, receiver: inner, id: self.node_builder.next_id() }
            .into())
    }

    // TODO: Parse method call expressions directly and later put them into a canonical form.
    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// method call expression.
    fn parse_method_call_expression(&mut self, receiver: Expression, method: Identifier) -> Result<Expression> {
        // Parse the argument list.
        let (mut args, _, span) = self.parse_expr_tuple()?;
        let span = receiver.span() + span;

        if let (true, Some(op)) = (args.is_empty(), UnaryOperation::from_symbol(method.name)) {
            // Found an unary operator and the argument list is empty.
            Ok(UnaryExpression { span, op, receiver, id: self.node_builder.next_id() }.into())
        } else if let (1, Some(op)) = (args.len(), BinaryOperation::from_symbol(method.name)) {
            // Found a binary operator and the argument list contains a single argument.
            Ok(BinaryExpression {
                span,
                op,
                left: receiver,
                right: args.swap_remove(0),
                id: self.node_builder.next_id(),
            }
            .into())
        } else if let (2, Some(CoreFunction::SignatureVerify)) =
            (args.len(), CoreFunction::from_symbols(sym::signature, method.name))
        {
            Ok(AssociatedFunctionExpression {
                variant: Identifier::new(sym::signature, self.node_builder.next_id()),
                name: method,
                arguments: std::iter::once(receiver).chain(args).collect(),
                span,
                id: self.node_builder.next_id(),
            }
            .into())
        } else if let (0, Some(CoreFunction::FutureAwait)) =
            (args.len(), CoreFunction::from_symbols(sym::Future, method.name))
        {
            Ok(AssociatedFunctionExpression {
                variant: Identifier::new(sym::Future, self.node_builder.next_id()),
                name: method,
                arguments: vec![receiver],
                span,
                id: self.node_builder.next_id(),
            }
            .into())
        } else {
            // Attempt to parse the method call as a mapping operation.
            match (args.len(), CoreFunction::from_symbols(sym::Mapping, method.name)) {
                (1, Some(CoreFunction::MappingGet))
                | (2, Some(CoreFunction::MappingGetOrUse))
                | (2, Some(CoreFunction::MappingSet))
                | (1, Some(CoreFunction::MappingRemove))
                | (1, Some(CoreFunction::MappingContains)) => {
                    // Found an instance of `<mapping>.get`, `<mapping>.get_or_use`, `<mapping>.set`, `<mapping>.remove`, or `<mapping>.contains`.
                    Ok(AssociatedFunctionExpression {
                        variant: Identifier::new(sym::Mapping, self.node_builder.next_id()),
                        name: method,
                        arguments: std::iter::once(receiver).chain(args).collect(),
                        span,
                        id: self.node_builder.next_id(),
                    }
                    .into())
                }
                _ => {
                    // Either an invalid unary/binary operator, or more arguments given.
                    self.emit_err(ParserError::invalid_method_call(receiver, method, args.len(), span));
                    Ok(ErrExpression { span, id: self.node_builder.next_id() }.into())
                }
            }
        }
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// static access expression.
    fn parse_associated_access_expression(&mut self, module_name: Expression) -> Result<Expression> {
        // Ensure that the preceding expression is an identifier (a named type).
        let Expression::Path(ref path) = module_name else {
            return Err(ParserError::invalid_associated_access(&module_name, module_name.span()).into());
        };

        let variant = path.identifier();
        if !path.qualifier().is_empty() {
            return Err(ParserError::invalid_associated_access(&module_name, module_name.span()).into());
        };

        // Parse the constant or function name.
        let member_name = self.expect_identifier()?;

        // Check if there are arguments.
        let expression = if self.check(&Token::LeftParen) {
            // Parse the arguments
            let (args, _, end) = self.parse_expr_tuple()?;

            // Return the associated function.
            AssociatedFunctionExpression {
                span: module_name.span() + end,
                variant,
                name: member_name,
                arguments: args,
                id: self.node_builder.next_id(),
            }
            .into()
        } else if CoreConstant::from_symbols(variant.name, member_name.name).is_some() {
            // Return the associated constant.
            AssociatedConstantExpression {
                span: module_name.span() + member_name.span(),
                ty: Type::Identifier(variant),
                name: member_name,
                id: self.node_builder.next_id(),
            }
            .into()
        } else {
            Path::new(vec![variant], member_name, None, variant.span + member_name.span, self.node_builder.next_id())
                .into()
        };

        Ok(expression)
    }

    /// Parses a tuple of `Expression` AST nodes.
    pub(crate) fn parse_expr_tuple(&mut self) -> Result<(Vec<Expression>, bool, Span)> {
        self.parse_paren_comma_list(|p| p.parse_expression().map(Some))
    }

    /// Parses an external function call `credits.aleo/transfer()` or locator `token.aleo/accounts`.
    ///
    /// In the ABNF grammar,
    /// an external function call is one of the two kinds of free function calls,
    /// namely the one that uses a locator to designate the function;
    /// a locator is a kind of primary expression.
    fn parse_external_resource(&mut self, expr: Expression, network_span: Span) -> Result<Expression> {
        // Parse `/`.
        self.expect(&Token::Div)?;

        // Parse name.
        let name = self.expect_identifier()?;

        // Ensure the preceding expression is a (program) identifier.
        let Expression::Path(ref path) = expr else {
            unreachable!("Function must be preceded by a program identifier.");
        };

        if !path.qualifier().is_empty() {
            unreachable!("Function must be preceded by a single-segment path.");
        };

        let program = path.identifier();

        // Parsing a '{' means that user is trying to illegally define an external record.
        if self.token.token == Token::LeftCurly {
            return Err(ParserError::cannot_define_external_record(expr.span() + name.span()).into());
        }

        // If there is no parenthesis, then it is a locator.
        if self.token.token != Token::LeftParen {
            // Parse an external resource locator.
            return Ok(LocatorExpression {
                program: ProgramId {
                    name: program,
                    network: Identifier { name: sym::aleo, span: network_span, id: self.node_builder.next_id() },
                },
                name: name.name,
                span: expr.span() + name.span(),
                id: self.node_builder.next_id(),
            }
            .into());
        }

        // Parse the function call.
        let (arguments, _, span) = self.parse_paren_comma_list(|p| p.parse_expression().map(Some))?;

        Ok(CallExpression {
            span: expr.span() + span,
            function: name.into(),
            program: Some(program.name),
            const_arguments: vec![], // we do not expect const arguments for external calls at this time
            arguments,
            id: self.node_builder.next_id(),
        }
        .into())
    }

    /// Returns an [`Expression`] AST node if the next tokens represent an
    /// array access, struct member access, tuple access, or method call expression.
    ///
    /// Otherwise, tries to parse the next token using [`parse_primary_expression`].
    /// Note that, as mentioned in [`parse_primary_expression`],
    /// this function also completes the parsing of some primary expressions
    /// (as defined in the ABNF grammar),
    /// which [`parse_primary_expression`] only starts to parse.
    fn parse_postfix_expression(&mut self) -> Result<Expression> {
        // We don't directly parse named types and identifiers in associated constants and functions
        // here as the ABNF states. Rather, those named types and identifiers are parsed
        // as primary expressions, and combined to form associated constants and functions here.
        let mut expr = self.parse_primary_expression()?;
        loop {
            if self.eat(&Token::Dot) {
                if self.check_int() {
                    // Eat a tuple member access.
                    let (index, span) = self.eat_whole_number()?;
                    expr = TupleAccess { tuple: expr, index, span, id: self.node_builder.next_id() }.into();
                } else if self.eat(&Token::Leo) {
                    return Err(ParserError::only_aleo_external_calls(expr.span()).into());
                } else if self.eat(&Token::Aleo) {
                    if self.token.token == Token::Div {
                        expr = self.parse_external_resource(expr, self.prev_token.span)?;
                    } else {
                        // Parse as address literal, e.g. `hello.aleo`.
                        if !matches!(expr, Expression::Path(ref path) if path.qualifier().is_empty()) {
                            self.emit_err(ParserError::unexpected(expr.to_string(), "an identifier", expr.span()));
                        }

                        expr =
                            Literal::address(format!("{expr}.aleo"), expr.span(), self.node_builder.next_id()).into();
                    }
                } else {
                    // Parse instances of `self.address`.
                    // This needs to be handled as a special case because `address` is a keyword in Leo,
                    if matches!(expr, Expression::Path(ref path) if path.identifier().name == sym::SelfLower && path.qualifier().is_empty())
                        && self.token.token == Token::Address
                    {
                        // Eat the address token.
                        let span = self.expect(&Token::Address)?;
                        // Return a member access expression.
                        expr = MemberAccess {
                            span: expr.span() + span,
                            inner: expr,
                            name: Identifier { name: sym::address, span, id: self.node_builder.next_id() },
                            id: self.node_builder.next_id(),
                        }
                        .into();
                        continue;
                    }

                    // Parse identifier name.
                    let name = self.expect_identifier()?;

                    if self.check(&Token::LeftParen) {
                        // Eat a method call on a type
                        expr = self.parse_method_call_expression(expr, name)?
                    } else {
                        // Eat a struct member access.
                        let expr_span = expr.span();
                        expr = MemberAccess {
                            inner: expr,
                            name,
                            span: expr_span + name.span(),
                            id: self.node_builder.next_id(),
                        }
                        .into();
                    }
                }
            } else if self.eat(&Token::DoubleColon) {
                // If we see a `::`, then we either expect a core associated expression or a list of const arguments in
                // square brackets.
                if self.check(&Token::LeftSquare) {
                    // Check that the expression is a path.
                    let Expression::Path(path) = &expr else {
                        return Err(leo_errors::LeoError::ParserError(ParserError::unexpected(
                            expr.to_string(),
                            "a path",
                            expr.span(),
                        )));
                    };

                    // Parse a list of const arguments in between `[..]`
                    let const_arguments = self.parse_bracket_comma_list(|p| p.parse_expression().map(Some))?.0;

                    // If the next token is a `(` then we parse a call expression.
                    // If the next token is a `{`, then we parse a struct init expression.
                    if self.check(&Token::LeftParen) {
                        // Parse a list of input arguments in between `(..)`
                        let (arguments, _, span) = self.parse_paren_comma_list(|p| p.parse_expression().map(Some))?;

                        // Now form a `CallExpression`
                        expr = CallExpression {
                            span: expr.span() + span,
                            function: path.clone(),
                            program: self.program_name,
                            const_arguments,
                            arguments,
                            id: self.node_builder.next_id(),
                        }
                        .into();
                    } else if !self.disallow_struct_construction && self.check(&Token::LeftCurly) {
                        // Parse struct and records inits as struct expressions with const arguments.
                        // Enforce struct or record type later at type checking.
                        expr = self.parse_struct_init_expression(path.clone(), const_arguments)?;
                    } else {
                        self.emit_err(ParserError::unexpected(expr.to_string(), "( or {{", expr.span()))
                    }
                } else {
                    // Eat a core associated constant or core associated function call.
                    expr = self.parse_associated_access_expression(expr)?;
                }
            } else if self.eat(&Token::LeftSquare) {
                // Eat an array access.
                let index = self.parse_expression()?;
                // Eat the closing bracket.
                let span = self.expect(&Token::RightSquare)?;
                let expr_span = expr.span();
                expr =
                    ArrayAccess { array: expr, index, span: expr_span + span, id: self.node_builder.next_id() }.into();
            } else if self.check(&Token::LeftParen) {
                // Check that the expression is a path.
                let Expression::Path(path) = &expr else {
                    return Err(leo_errors::LeoError::ParserError(ParserError::unexpected(
                        expr.to_string(),
                        "a path",
                        expr.span(),
                    )));
                };

                // Parse a function call that's by itself.
                let (arguments, _, span) = self.parse_paren_comma_list(|p| p.parse_expression().map(Some))?;

                // Check for a core function otherwise treat as regular call.
                expr = if path.qualifier().len() == 1
                    && CoreFunction::from_symbols(path.qualifier()[0].name, path.identifier().name).is_some()
                {
                    AssociatedFunctionExpression {
                        variant: path.qualifier()[0],
                        name: path.identifier(),
                        arguments,
                        span: expr.span() + span,
                        id: self.node_builder.next_id(),
                    }
                    .into()
                } else {
                    CallExpression {
                        span: expr.span() + span,
                        function: path.clone(),
                        program: self.program_name,
                        const_arguments: vec![],
                        arguments,
                        id: self.node_builder.next_id(),
                    }
                    .into()
                };
            }
            // Stop parsing the postfix expression unless a dot or square bracket follows.
            if !(self.check(&Token::Dot) || self.check(&Token::LeftSquare)) {
                break;
            }
        }
        Ok(expr)
    }

    /// Returns an [`Expression`] AST node if the next tokens represent
    /// a parenthesized expression or a unit expression
    /// or a tuple initialization expression or an affine group literal.
    fn parse_tuple_expression(&mut self) -> Result<Expression> {
        let (mut elements, trailing, span) = self.parse_expr_tuple()?;

        match (elements.len(), trailing) {
            (0, _) | (1, true) => {
                // A tuple with 0 or 1 elements - emit an error since tuples must have at least two elements.
                Err(ParserError::tuple_must_have_at_least_two_elements("expression", span).into())
            }
            (1, false) => {
                // If there is one element in the tuple but no trailing comma, e.g `(foo)`, return the element.
                Ok(elements.remove(0))
            }
            _ => {
                // Otherwise, return a tuple expression.
                // Note: This is the only place where `TupleExpression` is constructed in the parser.
                Ok(TupleExpression { elements, span, id: self.node_builder.next_id() }.into())
            }
        }
    }

    /// Attempts to parse an array initialization expression and returns an [`Expression`] AST node if successful.
    fn parse_array_or_repeat_expression(&mut self) -> Result<Expression> {
        let (open, close) = Delimiter::Bracket.open_close_pair();
        let open_span = self.expect(&open)?;

        // Attempt to parse the first expression in the array.
        let Ok(first_expr) = self.parse_expression() else {
            // If we're unable to parse an expression, just expect a `]` and error out on empty array.
            let close_span = self.expect(&close)?;
            return Err(ParserError::array_must_have_at_least_one_element("expression", open_span + close_span).into());
        };

        // Handle array repetition syntax: [expr; count]
        if self.eat(&Token::Semicolon) {
            let count = self.parse_expression()?;
            let span = open_span + self.expect(&close)?;
            return Ok(RepeatExpression { expr: first_expr, count, span, id: self.node_builder.next_id() }.into());
        }

        // Handle array with multiple elements: [expr1, expr2, ...] or single element with or without trailing comma:
        // [expr,]
        let mut elements = vec![first_expr];
        while self.eat(&Token::Comma) && !self.check(&close) {
            elements.push(self.parse_expression()?);
        }
        let span = open_span + self.expect(&close)?;
        Ok(ArrayExpression { elements, span, id: self.node_builder.next_id() }.into())
    }

    fn parse_struct_member(&mut self) -> Result<StructVariableInitializer> {
        let identifier = self.expect_identifier()?;

        let (expression, span) = if self.eat(&Token::Colon) {
            // Parse individual struct variable declarations.
            let expression = self.parse_expression()?;
            let span = identifier.span + expression.span();
            (Some(expression), span)
        } else {
            (None, identifier.span)
        };

        Ok(StructVariableInitializer { identifier, expression, id: self.node_builder.next_id(), span })
    }

    /// Returns an [`Expression`] AST node if the next tokens represent a
    /// struct initialization expression.
    /// let foo = Foo { x: 1u8 };
    pub fn parse_struct_init_expression(&mut self, path: Path, const_arguments: Vec<Expression>) -> Result<Expression> {
        let (members, _, end) =
            self.parse_list(Delimiter::Brace, Some(Token::Comma), |p| p.parse_struct_member().map(Some))?;

        Ok(StructExpression { span: path.span + end, path, const_arguments, members, id: self.node_builder.next_id() }
            .into())
    }

    /// Returns an [`Expression`] AST node if the next token is a primary expression:
    /// - Literals: field, group, unsigned integer, signed integer, boolean, address, string
    /// - Aggregate type constructors: array, tuple, structs
    /// - Identifiers: variables, keywords
    ///
    /// This function only parses some of the primary expressions defined in the ABNF grammar;
    /// for the others, it parses their initial parts,
    /// leaving it to the [self.parse_postfix_expression] function to complete the parsing.
    /// For example, of the primary expression `u8::c`, this function only parses the `u8` part,
    /// leaving it to [self.parse_postfix_expression] to parse the `::c` part.
    /// So technically the expression returned by this function may not quite be
    /// an expression as defined in the ABNF grammar,
    /// but it is only a temporary expression that is combined into a larger one
    /// by [self.parse_postfix_expression], yielding an actual expression according to the grammar.
    ///
    /// Returns an expression error if the token cannot be matched.
    fn parse_primary_expression(&mut self) -> Result<Expression> {
        if let Token::LeftParen = self.token.token {
            return self.parse_tuple_expression();
        } else if let Token::LeftSquare = self.token.token {
            return self.parse_array_or_repeat_expression();
        } else if let Token::Async = self.token.token {
            let async_keyword_span = self.expect(&Token::Async)?;
            let block = self.parse_block()?;
            let span = async_keyword_span + block.span;
            return Ok(AsyncExpression { block, span, id: self.node_builder.next_id() }.into());
        }

        let SpannedToken { token, span } = self.token.clone();
        self.bump();

        Ok(match token {
            Token::Integer(value) => {
                let suffix_span = self.token.span;
                let full_span = span + suffix_span;
                let assert_no_whitespace = |x| assert_no_whitespace(span, suffix_span, &value, x);

                match self.eat_any(INT_TYPES).then_some(&self.prev_token.token) {
                    // Hex, octal, binary literal on a noninteger is an error.
                    Some(Token::Field) | Some(Token::Group) | Some(Token::Scalar)
                        if value.starts_with("0x")
                            || value.starts_with("0o")
                            || value.starts_with("0b")
                            || value.starts_with("-0x")
                            || value.starts_with("-0o")
                            || value.starts_with("-0b") =>
                    {
                        return Err(ParserError::hexbin_literal_nonintegers(span).into());
                    }
                    // Literal followed by `field`, e.g., `42field`.
                    Some(Token::Field) => {
                        assert_no_whitespace("field")?;
                        Literal::field(value, full_span, self.node_builder.next_id()).into()
                    }
                    // Literal followed by `group`, e.g., `42group`.
                    Some(Token::Group) => {
                        assert_no_whitespace("group")?;
                        Literal::group(value, full_span, self.node_builder.next_id()).into()
                    }
                    // Literal followed by `scalar` e.g., `42scalar`.
                    Some(Token::Scalar) => {
                        assert_no_whitespace("scalar")?;
                        Literal::scalar(value, full_span, self.node_builder.next_id()).into()
                    }
                    // Literal followed by other type suffix, e.g., `42u8`.
                    Some(suffix) => {
                        assert_no_whitespace(&suffix.to_string())?;
                        let int_ty = Self::token_to_int_type(suffix).expect("unknown int type token");
                        Literal::integer(int_ty, value, full_span, self.node_builder.next_id()).into()
                    }
                    None => {
                        // `Integer` tokens with no suffix are `unsuffixed`. We try to infer their
                        // type in the type inference phase of the type checker.
                        Literal::unsuffixed(value, span, self.node_builder.next_id()).into()
                    }
                }
            }
            Token::True => Literal::boolean(true, span, self.node_builder.next_id()).into(),
            Token::False => Literal::boolean(false, span, self.node_builder.next_id()).into(),
            Token::AddressLit(address_string) => {
                if match self.network {
                    NetworkName::MainnetV0 => address_string.parse::<Address<MainnetV0>>().is_err(),
                    NetworkName::TestnetV0 => address_string.parse::<Address<TestnetV0>>().is_err(),
                    NetworkName::CanaryV0 => address_string.parse::<Address<CanaryV0>>().is_err(),
                } {
                    self.emit_err(ParserError::invalid_address_lit(&address_string, span));
                }
                Literal::address(address_string, span, self.node_builder.next_id()).into()
            }
            Token::StaticString(value) => {
                Literal { span, id: self.node_builder.next_id(), variant: LiteralVariant::String(value) }.into()
            }
            Token::Identifier(first_ident) => {
                let mut path_span = span;
                let identifier = Identifier { name: first_ident, span, id: self.node_builder.next_id() };
                let mut segments = vec![identifier];

                // Parse `::`-separated path segments
                while self.check(&Token::DoubleColon) {
                    // Look ahead to see if the next token after `::` is a `[` — if so, stop for const generics
                    if self.look_ahead(1, |next| matches!(next.token, Token::LeftSquare)) {
                        break;
                    }

                    self.bump(); // consume `::`

                    let next_ident = self.expect_identifier()?;
                    path_span = path_span + next_ident.span;
                    segments.push(next_ident);
                }

                let (identifier, qualifier) = segments.split_last().expect("guaranateed to have at least one segment");

                let path = Path::new(qualifier.to_vec(), *identifier, None, path_span, self.node_builder.next_id());

                // Check for struct initializer
                if !self.disallow_struct_construction && self.check(&Token::LeftCurly) {
                    // Parse struct and records inits as struct expressions without const arguments.
                    // Enforce struct or record type later at type checking.
                    self.parse_struct_init_expression(path.clone(), Vec::new())?
                } else {
                    path.into()
                }
            }
            Token::SelfLower => Identifier { name: sym::SelfLower, span, id: self.node_builder.next_id() }.into(),
            Token::Block => Identifier { name: sym::block, span, id: self.node_builder.next_id() }.into(),
            Token::Future => Identifier { name: sym::Future, span, id: self.node_builder.next_id() }.into(),
            Token::Network => Identifier { name: sym::network, span, id: self.node_builder.next_id() }.into(),
            t if crate::type_::TYPE_TOKENS.contains(&t) => {
                Identifier { name: t.keyword_to_symbol().unwrap(), span, id: self.node_builder.next_id() }.into()
            }
            token => {
                return Err(ParserError::unexpected_str(token, "expression", span).into());
            }
        })
    }
}

fn assert_no_whitespace(left_span: Span, right_span: Span, left: &str, right: &str) -> Result<()> {
    if left_span.hi != right_span.lo {
        let error_span = Span::new(left_span.hi, right_span.lo); // The span between them.
        return Err(ParserError::unexpected_whitespace(left, right, error_span).into());
    }

    Ok(())
}
