// Copyright (C) 2019-2025 Provable Inc.
// This file is part of the Leo library.

// The Leo library is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// The Leo library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with the Leo library. If not, see <https://www.gnu.org/licenses/>.

//! The Static Single Assignment pass traverses the AST and converts it into SSA form.
//! See https://en.wikipedia.org/wiki/Static_single-assignment_form for more information.
//! The pass also replaces `DefinitionStatement`s with `AssignmentStatement`s.
//! The pass also simplifies complex expressions into a sequence of `AssignStatement`s. For example, `(a + b) * c` is rewritten into `$var$1 = a + b; $var$2 = $var$1 * c`.
//!
//! Consider the following Leo code.
//! ```leo
//! function main(flag: u8, value: u8) -> u8 {
//!     if (flag == 0u8) {
//!         value += 1u8;
//!         return value;
//!     } else {
//!         value += 2u8;
//!     }
//!     return value;
//! }
//! ```
//!
//! The SSA pass produces the following code.
//! ```leo
//! function main(flag: u8, value: u8) -> u8 {
//!     $var$0 = flag == 0u8;
//!     if ($var$0) {
//!         $var$1 = value + 1u8;
//!         value$1 = $var$1;
//!         return value$1;
//!     } else {
//!         $var$2 = value + 2u8;
//!         value$2 = $var$2;
//!     }
//!     value$3 = $var$0 ? value$1 : value$2;
//!     return value$3;
//! }
//! ```
//! Note that the redundant assignments have no effect on the bytecode generated by the compiler.

use crate::{Pass, RenameTable};

use leo_ast::ProgramConsumer;
use leo_errors::Result;
use leo_span::Symbol;

mod expression;

mod program;

mod statement;

mod visitor;
use visitor::*;

pub struct SsaFormingInput {
    pub rename_defs: bool,
}

pub struct SsaForming;

impl Pass for SsaForming {
    type Input = SsaFormingInput;
    type Output = ();

    const NAME: &str = "SsaForming";

    fn do_pass(input: Self::Input, state: &mut crate::CompilerState) -> Result<Self::Output> {
        let mut ast = std::mem::take(&mut state.ast);
        let mut visitor = SsaFormingVisitor {
            state,
            rename_table: RenameTable::new(None),
            program: Symbol::intern(""),
            rename_defs: input.rename_defs,
        };
        ast.ast = visitor.consume_program(ast.ast);
        visitor.state.handler.last_err().map_err(|e| *e)?;
        visitor.state.ast = ast;
        Ok(())
    }
}
