use anyhow::*;
use itertools::Itertools;
use serde::{Deserialize, Serialize};
use std::{collections::HashMap, fmt, iter::FromIterator};

use super::super::*;

/// A value assigned to an attribute in a widget.
/// This can be a primitive String that contains any amount of variable
/// references, as would be generated by the string "foo {{var}} bar".
#[derive(Serialize, Deserialize, Clone, PartialEq, derive_more::Into, derive_more::From, Default)]
pub struct AttrVal(Vec<AttrValElement>);

impl fmt::Display for AttrVal {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.iter().map(|x| format!("{}", x)).join(""))
    }
}

impl fmt::Debug for AttrVal {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "AttrValue({:?})", self.0)
    }
}

impl IntoIterator for AttrVal {
    type IntoIter = std::vec::IntoIter<Self::Item>;
    type Item = AttrValElement;

    fn into_iter(self) -> Self::IntoIter {
        self.0.into_iter()
    }
}

impl FromIterator<AttrValElement> for AttrVal {
    fn from_iter<T: IntoIterator<Item = AttrValElement>>(iter: T) -> Self {
        AttrVal(iter.into_iter().collect())
    }
}

impl AttrVal {
    pub fn from_primitive<T: Into<PrimVal>>(v: T) -> Self {
        AttrVal(vec![AttrValElement::Primitive(v.into())])
    }

    pub fn from_var_ref<T: Into<VarName>>(v: T) -> Self {
        AttrVal(vec![AttrValElement::Expr(AttrValExpr::VarRef(v.into()))])
    }

    pub fn iter(&self) -> std::slice::Iter<AttrValElement> {
        self.0.iter()
    }

    pub fn var_refs(&self) -> impl Iterator<Item = &VarName> {
        self.0.iter().filter_map(|x| Some(x.as_expr()?.var_refs())).flatten()
    }

    /// resolve partially.
    /// If a var-ref links to another var-ref, that other var-ref is used.
    /// If a referenced variable is not found in the given hashmap, returns the var-ref unchanged.
    pub fn resolve_one_level(self, variables: &HashMap<VarName, AttrVal>) -> AttrVal {
        self.into_iter()
            .map(|entry| match entry {
                AttrValElement::Expr(expr) => AttrValElement::Expr(expr.map_terminals_into(|child_expr| match child_expr {
                    AttrValExpr::VarRef(var_name) => match variables.get(&var_name) {
                        Some(value) => AttrValExpr::Literal(value.clone()),
                        None => AttrValExpr::VarRef(var_name),
                    },
                    other => other,
                })),

                _ => entry,
            })
            .collect()
    }

    /// resolve fully.
    /// As the variables here have to be primitive values,
    /// this enforces that var-refs are not linking to other variables.
    pub fn resolve_fully(self, variables: &HashMap<VarName, PrimVal>) -> Result<PrimVal> {
        self.into_iter()
            .map(|element| match element {
                AttrValElement::Primitive(x) => Ok(x),
                AttrValElement::Expr(expr) => expr.eval(variables),
            })
            .collect()
    }

    // TODO this could be a fancy Iterator implementation, ig
    pub fn parse_string(s: &str) -> AttrVal {
        let mut elements = Vec::new();

        let mut cur_word = "".to_owned();
        let mut cur_varref: Option<String> = None;
        let mut curly_count = 0;
        for c in s.chars() {
            if let Some(ref mut varref) = cur_varref {
                if c == '}' {
                    curly_count -= 1;
                    if curly_count == 0 {
                        elements.push(AttrValElement::Expr(AttrValExpr::parse(varref).unwrap()));
                        cur_varref = None
                    }
                } else {
                    curly_count = 2;
                    varref.push(c);
                }
            } else if c == '{' {
                curly_count += 1;
                if curly_count == 2 {
                    if !cur_word.is_empty() {
                        elements.push(AttrValElement::primitive(std::mem::take(&mut cur_word)));
                    }
                    cur_varref = Some(String::new())
                }
            } else {
                if curly_count == 1 {
                    cur_word.push('{');
                }
                curly_count = 0;
                cur_word.push(c);
            }
        }
        if let Some(unfinished_varref) = cur_varref.take() {
            elements.push(AttrValElement::primitive(unfinished_varref));
        } else if !cur_word.is_empty() {
            elements.push(AttrValElement::primitive(cur_word));
        }
        AttrVal(elements)
    }
}

#[derive(Clone, PartialEq, Serialize, Deserialize)]
pub enum AttrValElement {
    Primitive(PrimVal),
    Expr(AttrValExpr),
}
impl fmt::Display for AttrValElement {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            AttrValElement::Primitive(x) => write!(f, "{}", x),
            AttrValElement::Expr(x) => write!(f, "{{{{{}}}}}", x),
        }
    }
}

impl fmt::Debug for AttrValElement {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            AttrValElement::Primitive(x) => write!(f, "Primitive({:?})", x),
            AttrValElement::Expr(x) => write!(f, "Expr({:?})", x),
        }
    }
}

impl AttrValElement {
    pub fn primitive(s: String) -> Self {
        AttrValElement::Primitive(PrimVal::from_string(s))
    }

    pub fn as_expr(&self) -> Option<&AttrValExpr> {
        match self {
            AttrValElement::Expr(x) => Some(x),
            _ => None,
        }
    }

    pub fn as_primitive(&self) -> Option<&PrimVal> {
        match self {
            AttrValElement::Primitive(x) => Some(x),
            _ => None,
        }
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use pretty_assertions::assert_eq;
    #[test]
    fn test_parse_string_or_var_ref_list() {
        let input = "{{foo}}{{bar}}b{}azb{a}z{{bat}}{}quok{{test}}";
        let output = AttrVal::parse_string(input);
        assert_eq!(
            output,
            AttrVal(vec![
                AttrValElement::Expr(AttrValExpr::VarRef(VarName("foo".to_owned()))),
                AttrValElement::Expr(AttrValExpr::VarRef(VarName("bar".to_owned()))),
                AttrValElement::primitive("b{}azb{a}z".to_owned()),
                AttrValElement::Expr(AttrValExpr::VarRef(VarName("bat".to_owned()))),
                AttrValElement::primitive("{}quok".to_owned()),
                AttrValElement::Expr(AttrValExpr::VarRef(VarName("test".to_owned()))),
            ]),
        )
    }
    #[test]
    fn test_parse_string_with_var_refs_attr_value() {
        assert_eq!(
            AttrVal(
                vec![
                    AttrValElement::Expr(AttrValExpr::VarRef(VarName("var".to_owned()))),
                    AttrValElement::primitive("something".to_owned())
                ]
                .into()
            ),
            AttrVal::parse_string("{{var}}something")
        );
    }
}
