---
title: Installation
description: 'Learn how to install and configure Nuxt UI in your Nuxt application.'
framework: nuxt
links:
  - label: Template
    to: https://codesandbox.io/p/devbox/nuxt-ui-xgrzw5
    icon: i-lucide-codesandbox
navigation.icon: i-lucide-square-play
---

::callout{to="/docs/getting-started/installation/vue" icon="i-logos-vue" class="hidden"}
Looking for the **Vue** version?
::

## Setup

### Add to a Nuxt project

::steps{level="4"}

#### Install the Nuxt UI package

::code-group{sync="pm"}

```bash [pnpm]
pnpm add @nuxt/ui
```

```bash [yarn]
yarn add @nuxt/ui
```

```bash [npm]
npm install @nuxt/ui
```

```bash [bun]
bun add @nuxt/ui
```

::

::warning
If you're using **pnpm**, ensure that you either set [`shamefully-hoist=true`](https://pnpm.io/npmrc#shamefully-hoist) in your `.npmrc` file or install `tailwindcss` in your project's root directory.
::

#### Add the Nuxt UI module in your `nuxt.config.ts`{lang="ts-type"}

```ts [nuxt.config.ts]
export default defineNuxtConfig({
  modules: ['@nuxt/ui']
})
```

#### Import Tailwind CSS and Nuxt UI in your CSS

::code-group

```css [app/assets/css/main.css]
@import "tailwindcss";
@import "@nuxt/ui";
```

```ts [nuxt.config.ts] {3}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css']
})
```

::

::callout{icon="i-simple-icons-visualstudiocode"}
It's recommended to install the [Tailwind CSS IntelliSense](https://marketplace.visualstudio.com/items?itemName=bradlc.vscode-tailwindcss) extension for VSCode and add the following settings:

```json [.vscode/settings.json]
{
  "files.associations": {
    "*.css": "tailwindcss"
  },
  "editor.quickSuggestions": {
    "strings": "on"
  },
  "tailwindCSS.classAttributes": ["class", "ui"],
  "tailwindCSS.experimental.classRegex": [
    ["ui:\\s*{([^)]*)\\s*}", "(?:'|\"|`)([^']*)(?:'|\"|`)"]
  ]
}
```

::

#### Wrap your app with App component

```vue [app.vue]
<template>
  <UApp>
    <NuxtPage />
  </UApp>
</template>
```

::note{to="/docs/components/app"}
The `App` component provides global configurations and is required for **Toast**, **Tooltip** components to work as well as **Programmatic Overlays**.
::

::

### Use a Nuxt template

To quickly get started with Nuxt UI, you can use the [starter template](https://github.com/nuxt-ui-templates/starter) by running:

```bash [Terminal]
npm create nuxt@latest -- -t ui
```

You can also get started with one of our [official templates](/templates):

::card-group
  ::card
  ---
  title: Starter
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/starter
  target: _blank
  color: neutral
  ---
  A minimal template to get started with Nuxt UI.
  ::

  ::card
  ---
  title: Landing
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/landing
  target: _blank
  color: neutral
  ---
  A modern landing page template powered by Nuxt Content.
  ::

  ::card
  ---
  title: Docs
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/docs
  target: _blank
  color: neutral
  ---
  A documentation template powered by Nuxt Content.
  ::

  ::card
  ---
  title: SaaS
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/saas
  target: _blank
  color: neutral
  ---
  A SaaS template with landing, pricing, docs and blog powered by Nuxt Content.
  ::

  ::card
  ---
  title: Dashboard
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/dashboard
  target: _blank
  color: neutral
  variant: subtle
  ---
  A dashboard template with multi-column layout for building sophisticated admin interfaces.
  ::

  ::card
  ---
  title: Chat
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/chat
  target: _blank
  color: neutral
  ---
  An AI chatbot template to build your own chatbot powered by Nuxt MDC and Vercel AI SDK.
  ::

  ::card
  ---
  title: Portfolio
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/portfolio
  target: _blank
  color: neutral
  ---
  A sleek portfolio template to showcase your work, skills and blog powered by Nuxt Content.
  ::

  ::card
  ---
  title: Changelog
  icon: i-simple-icons-github
  to: https://github.com/nuxt-ui-templates/changelog
  target: _blank
  color: neutral
  ---
  A changelog template to display your repository releases notes from GitHub powered by Nuxt MDC.
  ::
::

You can use the `Use this template` button on GitHub to create a new repository or use the CLI:

::code-group

```bash [Starter]
npm create nuxt@latest -- -t ui
```

```bash [Landing]
npm create nuxt@latest -- -t ui/landing
```

```bash [Docs]
npm create nuxt@latest -- -t ui/docs
```

```bash [SaaS]
npm create nuxt@latest -- -t ui/saas
```

```bash [Dashboard]
npm create nuxt@latest -- -t ui/dashboard
```

```bash [Chat]
npm create nuxt@latest -- -t ui/chat
```

```bash [Portfolio]
npm create nuxt@latest -- -t ui/portfolio
```

```bash [Changelog]
npm create nuxt@latest -- -t ui/changelog
```

::

## Options

You can customize Nuxt UI by providing options in your `nuxt.config.ts`.

### `prefix`

Use the `prefix` option to change the prefix of the components.

- Default: `U`{lang="ts-type"}

```ts [nuxt.config.ts] {4-6}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    prefix: 'Nuxt'
  }
})
```

### `fonts`

Use the `fonts` option to enable or disable the [`@nuxt/fonts`](https://github.com/nuxt/fonts) module.

- Default: `true`{lang="ts-type"}

```ts [nuxt.config.ts] {4-6}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    fonts: false
  }
})
```

### `colorMode`

Use the `colorMode` option to enable or disable the [`@nuxt/color-mode`](https://github.com/nuxt-modules/color-mode) module.

- Default: `true`{lang="ts-type"}

```ts [nuxt.config.ts] {4-6}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    colorMode: false
  }
})
```

### `theme.colors`

Use the `theme.colors` option to define the dynamic color aliases used to generate components theme.

- Default: `['primary', 'secondary', 'success', 'info', 'warning', 'error']`{lang="ts-type" class="inline"}

```ts [nuxt.config.ts] {4-8}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    theme: {
      colors: ['primary', 'error']
    }
  }
})
```

::tip{to="/docs/getting-started/theme/design-system#colors"}
Learn more about color customization and theming in the Theme section.
::

### `theme.transitions`

Use the `theme.transitions` option to enable or disable transitions on components.

- Default: `true`{lang="ts-type"}

```ts [nuxt.config.ts] {4-8}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    theme: {
      transitions: false
    }
  }
})
```

::note
This option adds the `transition-colors` class on components with hover or active states.
::

### `theme.defaultVariants`

Use the `theme.defaultVariants` option to override the default `color` and `size` variants for components.

- Default: `{ color: 'primary', size: 'md' }`{lang="ts-type"}

```ts [nuxt.config.ts] {4-11}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    theme: {
      defaultVariants: {
        color: 'neutral',
        size: 'sm'
      }
    }
  }
})
```

### `theme.prefix` :badge{label="Soon"}

Use the `theme.prefix` option to configure the same prefix you set on your Tailwind CSS import. This ensures Nuxt UI components use the correct prefixed utility classes and CSS variables.

::code-group

```ts [nuxt.config.ts] {4-8}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    theme: {
      prefix: 'tw'
    }
  }
})
```

```css [app/assets/css/main.css] {1}
@import "tailwindcss" prefix(tw);
@import "@nuxt/ui";
```

::

::warning{to="https://fonts.nuxt.com/get-started/configuration#processcssvariables"}
You might need to enable `fonts.processCSSVariables` to use the prefix option with the `@nuxt/fonts` module:

```ts [nuxt.config.ts] {9-11}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    theme: {
      prefix: 'tw'
    }
  },
  fonts: {
    processCSSVariables: true
  }
})
```
::

This will automatically prefix all Tailwind utility classes and CSS variables in Nuxt UI component themes:

```html
<!-- Without prefix -->
<button class="px-2 py-1 text-xs hover:bg-primary/75">Button</button>

<!-- With prefix: tw -->
<button class="tw:px-2 tw:py-1 tw:text-xs tw:hover:bg-primary/75">Button</button>
```

::note{to="https://tailwindcss.com/docs/styling-with-utility-classes#using-the-prefix-option" target="_blank"}
Learn more about using a prefix in the Tailwind CSS documentation.
::

### `mdc`

Use the `mdc` option to force the import of Nuxt UI `<Prose>` components even if `@nuxtjs/mdc` or `@nuxt/content` is not installed.

- Default: `false`{lang="ts-type"}

```ts [nuxt.config.ts] {4-6}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    mdc: true
  }
})
```

### `content`

Use the `content` option to force the import of Nuxt UI `<Prose>` and `<UContent>` components even if `@nuxt/content` is not installed (`@nuxtjs/mdc` is installed by `@nuxt/content`).

- Default: `false`{lang="ts-type"}

```ts [nuxt.config.ts] {4-6}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    content: true
  }
})
```

### `experimental.componentDetection` :badge{label="4.1+"}

Use the `experimental.componentDetection` option to enable automatic component detection for tree-shaking. This feature scans your source code to detect which components are actually used and only generates the necessary CSS for those components (including their dependencies).

- Default: `false`{lang="ts-type"}
- Type: `boolean | string[]`{lang="ts-type"}

**Enable automatic detection:**

```ts [nuxt.config.ts] {4-8}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    experimental: {
      componentDetection: true
    }
  }
})
```

**Include additional components for dynamic usage:**

```ts [nuxt.config.ts] {4-8}
export default defineNuxtConfig({
  modules: ['@nuxt/ui'],
  css: ['~/assets/css/main.css'],
  ui: {
    experimental: {
      componentDetection: ['Modal', 'Dropdown', 'Popover']
    }
  }
})
```

::note
When providing an array of component names, automatic detection is enabled and these components (along with their dependencies) are guaranteed to be included. This is useful for dynamic components like `<component :is="..." />` that can't be statically analyzed.
::

## Continuous releases

Nuxt UI uses [pkg.pr.new](https://github.com/stackblitz-labs/pkg.pr.new) for continuous preview releases, providing developers with instant access to the latest features and bug fixes without waiting for official releases.

Automatic preview releases are created for all commits and PRs to the `v4` branch. Use them by replacing your package version with the specific commit hash or PR number.

```diff [package.json]
{
  "dependencies": {
-   "@nuxt/ui": "^4.0.0",
+   "@nuxt/ui": "https://pkg.pr.new/@nuxt/ui@4c96909",
  }
}
```

::note
**pkg.pr.new** will automatically comment on PRs with the installation URL, making it easy to test changes.
::
