---
title: Content
description: 'Nuxt UI integrates with Nuxt Content to deliver beautiful typography and consistent component styling.'
framework: nuxt
navigation.icon: i-simple-icons-markdown
---

## Installation

To get started, you can follow the official [guide](https://content.nuxt.com/docs/getting-started/installation) or in summary:

::code-group{sync="pm"}

```bash [pnpm]
pnpm add @nuxt/content
```

```bash [yarn]
yarn add @nuxt/content
```

```bash [npm]
npm install @nuxt/content
```

```bash [bun]
bun add @nuxt/content
```

::

Then, add the `@nuxt/content` module in your `nuxt.config.ts`:

```ts [nuxt.config.ts] {4}
export default defineNuxtConfig({
  modules: [
    '@nuxt/ui',
    '@nuxt/content'
  ],
  css: ['~/assets/css/main.css']
})
```

::caution
You need to register `@nuxt/content` after `@nuxt/ui` in the `modules` array, otherwise the prose components will not be available.
::

## Configuration

When using Tailwind CSS classes in your markdown content files, you need to ensure Tailwind can detect and generate the necessary utility classes. By default, Tailwind's automatic content detection might not pick up classes written in markdown files.

To fix this, use the [`@source` directive](https://tailwindcss.com/docs/functions-and-directives#source-directive) in your CSS file to explicitly include your content directory:

```css [app/assets/css/main.css]
@import "tailwindcss";
@import "@nuxt/ui";

@source "../../../content/**/*";
```

This ensures that:
- Tailwind scans all markdown files in your content directory
- Any utility classes used in your markdown (like `text-primary`) are included in the final CSS
- Dynamic classes in MDC components or custom Vue components within your content work properly

::tip
You can also use glob patterns to be more specific about which files to scan:
- `@source "../../../content/docs/**/*.md"` - Only scan markdown in the docs folder
- `@source "../../../content/**/*.{md,yml}"` - Include both markdown and YAML files
::

::note{to="https://tailwindcss.com/docs/detecting-classes-in-source-files" target="_blank"}
Learn more about Tailwind's automatic content detection and best practices for optimizing build performance.
::

## Components

You might be using `@nuxt/content` to build a documentation. To help you with that, we've built some components that you can use in your pages:

- a built-in full-text search command palette with [ContentSearch](/docs/components/content-search), replacing the need for Algolia DocSearch
- a navigation tree with the [ContentNavigation](/docs/components/content-navigation) component
- a sticky Table of Contents with the [ContentToc](/docs/components/content-toc) component
- a prev / next navigation with the [ContentSurround](/docs/components/content-surround) component

## Typography

Nuxt UI provides its own custom implementations of all prose components for seamless integration with `@nuxt/content`. This approach ensures consistent styling, complete control over typography, and perfect alignment with the Nuxt UI design system so your content always looks and feels cohesive out of the box.

::note{to="/docs/typography"}
Discover the full **Typography** system and explore all available prose components for rich, consistent content presentation.
::

## Utils

### `mapContentNavigation`

This util will map the navigation from `queryCollectionNavigation` and transform it recursively into an array of objects that can be used by various components.

`mapContentNavigation(navigation, options?)`
- `navigation`: The navigation tree (array of ContentNavigationItem).
- `options` (optional):
  - `labelAttribute`: (string) Which field to use as label (`title` by default)
  - `deep`: (number or undefined) Controls how many levels of navigation are included (`undefined` by default : includes all levels)

**Example:** As shown in the breadcrumb example below, it's commonly used to transform the navigation data into the correct format.

```vue [app.vue]
<script setup lang="ts">
import { mapContentNavigation } from '@nuxt/ui/utils/content'
import { findPageBreadcrumb } from '@nuxt/content/utils'

const { data: navigation } = await useAsyncData('navigation', () => queryCollectionNavigation('content'))

const breadcrumb = computed(() => mapContentNavigation(findPageBreadcrumb(navigation?.value, page.value?.path, { indexAsChild: true })).map(({ icon, ...link }) => link), { deep: 0 })
</script>

<template>
  <UPage>
    <UPageHeader v-bind="page">
      <template #headline>
        <UBreadcrumb :items="breadcrumb" />
      </template>
    </UPageHeader>
  </UPage>
</template>
```
