## Complex DSL extensions

Complex DSL extensions are not complex to use, but they encapsulate enterprise integration patterns that would be anything
but trivial to write using plain Camel.

### Validation

The `validation` DSL extension implements a validation process that delegates the actual message validation to a
validator, which can be an endpoint or an object that implements validation logic.

If validation succeeds the message is forwarded *in-only* to the next processor defined in the route,
otherwise the message is dropped. In both cases the response generated by the validator is returned to the sender of the
original *in-out* message exchange.

In the following example, the validation process delegates message validation to the `direct:validator` endpoint.
If validation succeeds the message is forwarded as in-only exchange to mock:output, otherwise it is dropped.
The response returned from direct:validator is returned to the direct:input endpoint.

```groovy
    from('direct:input')
        .validation('direct:validator')
        .to('mock:output')
```

The validation process interprets a message exchange as failed if any of the following conditions is true:

* an exception was thrown by the validator
* the message exchange contains an exception
* the message exchange contains a fault message (i.e. an out message marked as fault)

Instead of providing an endpoint, the validator logic can also be provided as Groovy Closure or as Camel Processor:

```groovy

    Processor validator = new MyFamousValidator()

    from('direct:input1')
        .validation(validator)
        .to('mock:output')

    from('direct:input2')
        // throw a validation exception
        .validation { throw new ValidationException('input sucks in any case') }
        .to('mock:output')

```

### Multiplast

The *Multiplast* enterprise integration pattern combines [splitter] and [recipient list] EIPs. It generates a list of n
messages and a list of n endpoint URIs, sends splitted messages in parallel to their corresponding endpoint,
and aggregates their responses.

The DSL element therefore requires three parameters:
* splittingExpression: Camel expression that creates a list of messages from the original one
* recipientListExpression: Camel expression which creates the list of target endpoint URIs
* aggregationStrategy: strategy for aggregating received responses

```groovy
    from('direct:input')
        .multiplast(splitter, recipients, aggregator)
        .to('mock:output')
```


[splitter]: http://camel.apache.org/splitter.html
[recipient list]: http://camel.apache.org/recipient-list.html