/*
 * Copyright 2009 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openehealth.ipf.commons.test.performance.processingtime

import groovy.xml.MarkupBuilder
import org.openehealth.ipf.commons.test.performance.Statistics
import org.openehealth.ipf.commons.test.performance.StatisticsRenderer
import static org.openehealth.ipf.commons.test.performance.utils.NumberUtils.format

/**
 * Prints the measurement data in the HTML reports, generated by the rendeder. Optionally stores 
 * CSV data file on the disk.  
 *  
 * @author Mitko Kolev
 */
class ProcessingTimeDescriptiveStatisticsRenderer implements StatisticsRenderer {
    
    boolean createMeasurementLocationDataFiles = false
    
    /*
     * Render the data as HTML and optionally save a CSV file with the data.  
     */
    String render(Statistics model) {
        if (!model instanceof ProcessingTimeDescriptiveStatistics){
            throw new IllegalArgumentException('Only instances of ProcessingTimeDescriptiveStatistics are allowed!')
        }
        
        StringWriter writer = new StringWriter()
        MarkupBuilder builder = new MarkupBuilder(writer)
        List names = model.getMeasurementNames()
        int locationIndex = 1 
        builder.processingtime(){
            h3('Processing time data')
            table(border:'1', width:'1000'){
                for (name in names){
                    
                    def summary = model.getStatisticalSummaryByName(name)
                    def data = model.getData(name)
                    def csv = createCSV(data)
                    //create a CSV data file for every measurement location
                    if (createMeasurementLocationDataFiles){
                        // Do not escape the characters, that are not allowed for file names.
                        // Use a generated path and include the measurement location as a comment
                        // in the CSV file
                        def locationFileName = "measurement-location-${locationIndex}.csv"
                        locationIndex ++
                        createCSVDataFile(locationFileName, csv, name)
                    }
                    tr(){th(name)}
                    tr(){ td(csv) }
                }
            }
        }
        writer.toString()
    }
    
    String createCSV(Object data){
        StringBuilder csv = new StringBuilder()
        for (double d in data){
            csv.append(format(d, 0)).append(',')
        }
        csv.delete(csv.length() - 1, csv.length()) //delete the last comma 
        return csv.toString()
    }
    
    void createCSVDataFile(String fileName, Object csv, String locationName){
        String header = "# Measurement data ${locationName}\n"
        File f = new File(fileName)
        f.createNewFile()
        f.setWritable(true)
        
        f.write(header)
        f.append(csv)
    }
}
