import * as fs from "fs";
import * as path from "path";

// CONFIGURATION
// We are reading the migrations generated for the Native app (SQLite)
// to use them in the Web app (SQLocal/WASM).
const MIGRATIONS_DIR = path.join(
  process.cwd(),
  "../../apps/native/src-tauri/migrations"
);
const OUTPUT_FILE = path.join(process.cwd(), "drizzle/sqlite/migrations.ts");

interface MigrationEntry {
  id: number;
  name: string;
  sql: string;
}

interface JournalEntry {
  idx: number;
  version: string;
  when: number;
  tag: string;
  breakpoints: boolean;
}

interface Journal {
  entries: JournalEntry[];
}

function bundleMigrations() {
  console.log("Bundling migrations for Web (SQLocal)...");
  console.log(`Reading from: ${MIGRATIONS_DIR}`);

  // 1. Check if migrations directory exists
  const journalPath = path.join(MIGRATIONS_DIR, "meta", "_journal.json");

  if (!fs.existsSync(journalPath)) {
    console.error(`Migration journal not found at: ${journalPath}`);
    console.error(
      'Please run "pnpm generate:sqlite" first to generate the SQL files.'
    );
    process.exit(1);
  }

  // 2. Read the journal to get the correct order
  let journal: Journal;
  try {
    const journalContent = fs.readFileSync(journalPath, "utf-8");
    journal = JSON.parse(journalContent);
  } catch (e) {
    console.error("Failed to parse journal file:", e);
    process.exit(1);
  }

  const migrations: MigrationEntry[] = journal.entries.map((entry, index) => {
    const sqlFileName = `${entry.tag}.sql`;
    const sqlPath = path.join(MIGRATIONS_DIR, sqlFileName);

    if (!fs.existsSync(sqlPath)) {
      console.warn(`Warning: SQL file not found for migration ${entry.tag}`);
      return { id: index + 1, name: entry.tag, sql: "" };
    }

    const sqlContent = fs.readFileSync(sqlPath, "utf-8");
    const safeSql = sqlContent;

    return {
      id: index + 1,
      name: entry.tag,
      sql: safeSql,
    };
  });

  // 3. Generate TypeScript file content
  const fileContent = `// THIS FILE IS AUTOMATICALLY GENERATED. DO NOT EDIT MANUALLY.

export const MIGRATIONS = ${JSON.stringify(migrations, null, 2)};
`;

  // 4. Write the file
  try {
    // Ensure the output directory exists
    const outputDir = path.dirname(OUTPUT_FILE);
    if (!fs.existsSync(outputDir)) {
      fs.mkdirSync(outputDir, { recursive: true });
    }

    fs.writeFileSync(OUTPUT_FILE, fileContent);
    console.log(
      `Success! Bundled ${migrations.length} migrations into '${OUTPUT_FILE}'`
    );
  } catch (e) {
    console.error("Failed to write output file:", e);
    process.exit(1);
  }
}

bundleMigrations();
