/*
 *  Flo's Open libRary (floor)
 *  Copyright (C) 2004 - 2025 Florian Ziesche
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License only.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#pragma once

#include <floor/core/essentials.hpp>

// NOTE: this requires both VR and Vulkan support (no point in anything else)
#if !defined(FLOOR_NO_OPENXR) && !defined(FLOOR_NO_VULKAN)
#include <floor/compute/vulkan/vulkan_common.hpp>

#define XR_USE_GRAPHICS_API_VULKAN 1
#if defined(__linux__)
#define XR_USE_TIMESPEC 1
#endif
#include <openxr/openxr.h>
#include <openxr/openxr_platform.h>

constexpr const char* xr_error_to_string(const int& error_code) {
	switch(error_code) {
		case XR_SUCCESS: return "XR_SUCCESS";
		case XR_TIMEOUT_EXPIRED: return "XR_TIMEOUT_EXPIRED";
		case XR_SESSION_LOSS_PENDING: return "XR_SESSION_LOSS_PENDING";
		case XR_EVENT_UNAVAILABLE: return "XR_EVENT_UNAVAILABLE";
		case XR_SPACE_BOUNDS_UNAVAILABLE: return "XR_SPACE_BOUNDS_UNAVAILABLE";
		case XR_SESSION_NOT_FOCUSED: return "XR_SESSION_NOT_FOCUSED";
		case XR_FRAME_DISCARDED: return "XR_FRAME_DISCARDED";
		case XR_ERROR_VALIDATION_FAILURE: return "XR_ERROR_VALIDATION_FAILURE";
		case XR_ERROR_RUNTIME_FAILURE: return "XR_ERROR_RUNTIME_FAILURE";
		case XR_ERROR_OUT_OF_MEMORY: return "XR_ERROR_OUT_OF_MEMORY";
		case XR_ERROR_API_VERSION_UNSUPPORTED: return "XR_ERROR_API_VERSION_UNSUPPORTED";
		case XR_ERROR_INITIALIZATION_FAILED: return "XR_ERROR_INITIALIZATION_FAILED";
		case XR_ERROR_FUNCTION_UNSUPPORTED: return "XR_ERROR_FUNCTION_UNSUPPORTED";
		case XR_ERROR_FEATURE_UNSUPPORTED: return "XR_ERROR_FEATURE_UNSUPPORTED";
		case XR_ERROR_EXTENSION_NOT_PRESENT: return "XR_ERROR_EXTENSION_NOT_PRESENT";
		case XR_ERROR_LIMIT_REACHED: return "XR_ERROR_LIMIT_REACHED";
		case XR_ERROR_SIZE_INSUFFICIENT: return "XR_ERROR_SIZE_INSUFFICIENT";
		case XR_ERROR_HANDLE_INVALID: return "XR_ERROR_HANDLE_INVALID";
		case XR_ERROR_INSTANCE_LOST: return "XR_ERROR_INSTANCE_LOST";
		case XR_ERROR_SESSION_RUNNING: return "XR_ERROR_SESSION_RUNNING";
		case XR_ERROR_SESSION_NOT_RUNNING: return "XR_ERROR_SESSION_NOT_RUNNING";
		case XR_ERROR_SESSION_LOST: return "XR_ERROR_SESSION_LOST";
		case XR_ERROR_SYSTEM_INVALID: return "XR_ERROR_SYSTEM_INVALID";
		case XR_ERROR_PATH_INVALID: return "XR_ERROR_PATH_INVALID";
		case XR_ERROR_PATH_COUNT_EXCEEDED: return "XR_ERROR_PATH_COUNT_EXCEEDED";
		case XR_ERROR_PATH_FORMAT_INVALID: return "XR_ERROR_PATH_FORMAT_INVALID";
		case XR_ERROR_PATH_UNSUPPORTED: return "XR_ERROR_PATH_UNSUPPORTED";
		case XR_ERROR_LAYER_INVALID: return "XR_ERROR_LAYER_INVALID";
		case XR_ERROR_LAYER_LIMIT_EXCEEDED: return "XR_ERROR_LAYER_LIMIT_EXCEEDED";
		case XR_ERROR_SWAPCHAIN_RECT_INVALID: return "XR_ERROR_SWAPCHAIN_RECT_INVALID";
		case XR_ERROR_SWAPCHAIN_FORMAT_UNSUPPORTED: return "XR_ERROR_SWAPCHAIN_FORMAT_UNSUPPORTED";
		case XR_ERROR_ACTION_TYPE_MISMATCH: return "XR_ERROR_ACTION_TYPE_MISMATCH";
		case XR_ERROR_SESSION_NOT_READY: return "XR_ERROR_SESSION_NOT_READY";
		case XR_ERROR_SESSION_NOT_STOPPING: return "XR_ERROR_SESSION_NOT_STOPPING";
		case XR_ERROR_TIME_INVALID: return "XR_ERROR_TIME_INVALID";
		case XR_ERROR_REFERENCE_SPACE_UNSUPPORTED: return "XR_ERROR_REFERENCE_SPACE_UNSUPPORTED";
		case XR_ERROR_FILE_ACCESS_ERROR: return "XR_ERROR_FILE_ACCESS_ERROR";
		case XR_ERROR_FILE_CONTENTS_INVALID: return "XR_ERROR_FILE_CONTENTS_INVALID";
		case XR_ERROR_FORM_FACTOR_UNSUPPORTED: return "XR_ERROR_FORM_FACTOR_UNSUPPORTED";
		case XR_ERROR_FORM_FACTOR_UNAVAILABLE: return "XR_ERROR_FORM_FACTOR_UNAVAILABLE";
		case XR_ERROR_API_LAYER_NOT_PRESENT: return "XR_ERROR_API_LAYER_NOT_PRESENT";
		case XR_ERROR_CALL_ORDER_INVALID: return "XR_ERROR_CALL_ORDER_INVALID";
		case XR_ERROR_GRAPHICS_DEVICE_INVALID: return "XR_ERROR_GRAPHICS_DEVICE_INVALID";
		case XR_ERROR_POSE_INVALID: return "XR_ERROR_POSE_INVALID";
		case XR_ERROR_INDEX_OUT_OF_RANGE: return "XR_ERROR_INDEX_OUT_OF_RANGE";
		case XR_ERROR_VIEW_CONFIGURATION_TYPE_UNSUPPORTED: return "XR_ERROR_VIEW_CONFIGURATION_TYPE_UNSUPPORTED";
		case XR_ERROR_ENVIRONMENT_BLEND_MODE_UNSUPPORTED: return "XR_ERROR_ENVIRONMENT_BLEND_MODE_UNSUPPORTED";
		case XR_ERROR_NAME_DUPLICATED: return "XR_ERROR_NAME_DUPLICATED";
		case XR_ERROR_NAME_INVALID: return "XR_ERROR_NAME_INVALID";
		case XR_ERROR_ACTIONSET_NOT_ATTACHED: return "XR_ERROR_ACTIONSET_NOT_ATTACHED";
		case XR_ERROR_ACTIONSETS_ALREADY_ATTACHED: return "XR_ERROR_ACTIONSETS_ALREADY_ATTACHED";
		case XR_ERROR_LOCALIZED_NAME_DUPLICATED: return "XR_ERROR_LOCALIZED_NAME_DUPLICATED";
		case XR_ERROR_LOCALIZED_NAME_INVALID: return "XR_ERROR_LOCALIZED_NAME_INVALID";
		case XR_ERROR_GRAPHICS_REQUIREMENTS_CALL_MISSING: return "XR_ERROR_GRAPHICS_REQUIREMENTS_CALL_MISSING";
		case XR_ERROR_RUNTIME_UNAVAILABLE: return "XR_ERROR_RUNTIME_UNAVAILABLE";
		case XR_ERROR_ANDROID_THREAD_SETTINGS_ID_INVALID_KHR: return "XR_ERROR_ANDROID_THREAD_SETTINGS_ID_INVALID_KHR";
		case XR_ERROR_ANDROID_THREAD_SETTINGS_FAILURE_KHR: return "XR_ERROR_ANDROID_THREAD_SETTINGS_FAILURE_KHR";
		case XR_ERROR_CREATE_SPATIAL_ANCHOR_FAILED_MSFT: return "XR_ERROR_CREATE_SPATIAL_ANCHOR_FAILED_MSFT";
		case XR_ERROR_SECONDARY_VIEW_CONFIGURATION_TYPE_NOT_ENABLED_MSFT: return "XR_ERROR_SECONDARY_VIEW_CONFIGURATION_TYPE_NOT_ENABLED_MSFT";
		case XR_ERROR_CONTROLLER_MODEL_KEY_INVALID_MSFT: return "XR_ERROR_CONTROLLER_MODEL_KEY_INVALID_MSFT";
		case XR_ERROR_REPROJECTION_MODE_UNSUPPORTED_MSFT: return "XR_ERROR_REPROJECTION_MODE_UNSUPPORTED_MSFT";
		case XR_ERROR_COMPUTE_NEW_SCENE_NOT_COMPLETED_MSFT: return "XR_ERROR_COMPUTE_NEW_SCENE_NOT_COMPLETED_MSFT";
		case XR_ERROR_SCENE_COMPONENT_ID_INVALID_MSFT: return "XR_ERROR_SCENE_COMPONENT_ID_INVALID_MSFT";
		case XR_ERROR_SCENE_COMPONENT_TYPE_MISMATCH_MSFT: return "XR_ERROR_SCENE_COMPONENT_TYPE_MISMATCH_MSFT";
		case XR_ERROR_SCENE_MESH_BUFFER_ID_INVALID_MSFT: return "XR_ERROR_SCENE_MESH_BUFFER_ID_INVALID_MSFT";
		case XR_ERROR_SCENE_COMPUTE_FEATURE_INCOMPATIBLE_MSFT: return "XR_ERROR_SCENE_COMPUTE_FEATURE_INCOMPATIBLE_MSFT";
		case XR_ERROR_SCENE_COMPUTE_CONSISTENCY_MISMATCH_MSFT: return "XR_ERROR_SCENE_COMPUTE_CONSISTENCY_MISMATCH_MSFT";
		case XR_ERROR_DISPLAY_REFRESH_RATE_UNSUPPORTED_FB: return "XR_ERROR_DISPLAY_REFRESH_RATE_UNSUPPORTED_FB";
		case XR_ERROR_COLOR_SPACE_UNSUPPORTED_FB: return "XR_ERROR_COLOR_SPACE_UNSUPPORTED_FB";
		case XR_ERROR_SPACE_COMPONENT_NOT_SUPPORTED_FB: return "XR_ERROR_SPACE_COMPONENT_NOT_SUPPORTED_FB";
		case XR_ERROR_SPACE_COMPONENT_NOT_ENABLED_FB: return "XR_ERROR_SPACE_COMPONENT_NOT_ENABLED_FB";
		case XR_ERROR_SPACE_COMPONENT_STATUS_PENDING_FB: return "XR_ERROR_SPACE_COMPONENT_STATUS_PENDING_FB";
		case XR_ERROR_SPACE_COMPONENT_STATUS_ALREADY_SET_FB: return "XR_ERROR_SPACE_COMPONENT_STATUS_ALREADY_SET_FB";
		case XR_ERROR_UNEXPECTED_STATE_PASSTHROUGH_FB: return "XR_ERROR_UNEXPECTED_STATE_PASSTHROUGH_FB";
		case XR_ERROR_FEATURE_ALREADY_CREATED_PASSTHROUGH_FB: return "XR_ERROR_FEATURE_ALREADY_CREATED_PASSTHROUGH_FB";
		case XR_ERROR_FEATURE_REQUIRED_PASSTHROUGH_FB: return "XR_ERROR_FEATURE_REQUIRED_PASSTHROUGH_FB";
		case XR_ERROR_NOT_PERMITTED_PASSTHROUGH_FB: return "XR_ERROR_NOT_PERMITTED_PASSTHROUGH_FB";
		case XR_ERROR_INSUFFICIENT_RESOURCES_PASSTHROUGH_FB: return "XR_ERROR_INSUFFICIENT_RESOURCES_PASSTHROUGH_FB";
		case XR_ERROR_UNKNOWN_PASSTHROUGH_FB: return "XR_ERROR_UNKNOWN_PASSTHROUGH_FB";
		case XR_ERROR_RENDER_MODEL_KEY_INVALID_FB: return "XR_ERROR_RENDER_MODEL_KEY_INVALID_FB";
		case XR_RENDER_MODEL_UNAVAILABLE_FB: return "XR_RENDER_MODEL_UNAVAILABLE_FB";
		case XR_ERROR_MARKER_NOT_TRACKED_VARJO: return "XR_ERROR_MARKER_NOT_TRACKED_VARJO";
		case XR_ERROR_MARKER_ID_INVALID_VARJO: return "XR_ERROR_MARKER_ID_INVALID_VARJO";
		case XR_ERROR_SPATIAL_ANCHOR_NAME_NOT_FOUND_MSFT: return "XR_ERROR_SPATIAL_ANCHOR_NAME_NOT_FOUND_MSFT";
		case XR_ERROR_SPATIAL_ANCHOR_NAME_INVALID_MSFT: return "XR_ERROR_SPATIAL_ANCHOR_NAME_INVALID_MSFT";
		case XR_ERROR_SPACE_MAPPING_INSUFFICIENT_FB: return "XR_ERROR_SPACE_MAPPING_INSUFFICIENT_FB";
		case XR_ERROR_SPACE_LOCALIZATION_FAILED_FB: return "XR_ERROR_SPACE_LOCALIZATION_FAILED_FB";
		case XR_ERROR_SPACE_NETWORK_TIMEOUT_FB: return "XR_ERROR_SPACE_NETWORK_TIMEOUT_FB";
		case XR_ERROR_SPACE_NETWORK_REQUEST_FAILED_FB: return "XR_ERROR_SPACE_NETWORK_REQUEST_FAILED_FB";
		case XR_ERROR_SPACE_CLOUD_STORAGE_DISABLED_FB: return "XR_ERROR_SPACE_CLOUD_STORAGE_DISABLED_FB";
		case XR_ERROR_HINT_ALREADY_SET_QCOM: return "XR_ERROR_HINT_ALREADY_SET_QCOM";
		default: break;
	}
	return "<UNKNOWN_ERROR>";
}

FLOOR_PUSH_WARNINGS()
FLOOR_IGNORE_WARNING(unused-macros)

#define XR_CALL_RET(call, error_msg, ...) do { \
	const auto call_err_var = call; \
	if (call_err_var != XR_SUCCESS) { \
		log_error("$: $: $", error_msg, call_err_var, xr_error_to_string(call_err_var)); \
		return __VA_ARGS__; \
	} \
} while(false)
#define XR_CALL_CONT(call, error_msg) { \
	const int32_t call_err_var = call; \
	if (call_err_var != XR_SUCCESS) { \
		log_error("$: $: $", error_msg, call_err_var, xr_error_to_string(call_err_var)); \
		continue; \
	} \
} do {} while(false)
#define XR_CALL_BREAK(call, error_msg) { \
	const int32_t call_err_var = call; \
	if (call_err_var != XR_SUCCESS) { \
		log_error("$: $: $", error_msg, call_err_var, xr_error_to_string(call_err_var)); \
		break; \
	} \
} do {} while(false)
#define XR_CALL_ERR_EXEC(call, error_msg, do_stuff) do { \
	const auto call_err_var = call; \
	if (call_err_var != XR_SUCCESS) { \
		log_error("$: $: $", error_msg, call_err_var, xr_error_to_string(call_err_var)); \
		do_stuff \
	} \
} while(false)
#define XR_CALL_IGNORE(call, error_msg) do { \
	const auto call_err_var = call; \
	if (call_err_var != XR_SUCCESS) { \
		log_error("$: $: $", error_msg, call_err_var, xr_error_to_string(call_err_var)); \
	} \
} while(false)

FLOOR_POP_WARNINGS()

#endif
