// SPDX-License-Identifier: Apache-2.0
// ----------------------------------------------------------------------------
// Copyright 2011-2021 Arm Limited
//
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy
// of the License at:
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations
// under the License.
// ----------------------------------------------------------------------------

/**
 * @brief Percentile data tables for different block encodings.
 *
 * To reduce binary size the tables are stored using a packed differential encoding.
 */

#include "astcenc_internal.h"

#if !defined(ASTCENC_DECOMPRESS_ONLY)
/**
 * @brief Structure containing packed percentile metadata.
 *
 * Note that percentile tables do not exist for 3D textures, so no zdim is stored.
 */
struct packed_percentile_table
{
	/** The block X dimension. */
	uint8_t xdim;
	/** The block Y dimension. */
	uint8_t ydim;
	/** The number of packed items in the 1 and 2 plane data. */
	uint16_t itemcounts[2];
	/** The accumulator divisor for 1 and 2 plane data. */
	uint16_t difscales[2];
	/** The initital accumulator values for 1 and 2 plane data. */
	uint16_t initial_percs[2];
	/** The packed data for the 1 and 2 plane data. */
	const uint16_t *items[2];
};

#if ASTCENC_BLOCK_MAX_TEXELS >= (4 * 4)
static const uint16_t percentile_arr_4x4_0[61] {
	0x0242,0x7243,0x6A51,0x6A52,0x5A41,0x4A53,0x8851,0x3842,
	0x3852,0x3853,0x3043,0xFA33,0x1BDF,0x2022,0x1032,0x29CE,
	0x21DE,0x2823,0x0813,0x0A13,0x0A31,0x0A23,0x09CF,0x0833,
	0x0A32,0x01DF,0x0BDD,0x0BCF,0x0221,0x095F,0x0A01,0x0BDE,
	0x0BCD,0x0A22,0x09AF,0x0B5F,0x0B4D,0x0BCE,0x0BBF,0x0A11,
	0x01BF,0x0202,0x0B5D,0x1203,0x034E,0x0B8E,0x035E,0x0212,
	0x032E,0x0B4F,0x03AF,0x03AD,0x03BD,0x0BBE,0x03AE,0x039F,
	0x039E,0x033E,0x033F,0x038F,0x032F
};

static const uint16_t percentile_arr_4x4_1[84] {
	0x0452,0xFFAE,0x2433,0x1DDF,0x17CD,0x1E21,0x1C43,0x1442,
	0x3FBE,0x1FDD,0x0E31,0x0F4F,0x1423,0x0FBD,0x1451,0x0E03,
	0x05CF,0x0C32,0x0DDE,0x27AD,0x274E,0x0E02,0x0F5E,0x07AF,
	0x0F5F,0x0DCE,0x0C41,0x0422,0x0613,0x0E12,0x0611,0x0F3F,
	0x0601,0x0DBF,0x05DD,0x075D,0x0C02,0x054E,0x0431,0x0413,
	0x079F,0x05BE,0x0F4D,0x0403,0x05AF,0x055F,0x05AE,0x054F,
	0x0421,0x05BD,0x0DCD,0x0411,0x0412,0x055E,0x055D,0x073D,
	0x058E,0x072F,0x072D,0x079D,0x0D2E,0x0453,0x078D,0x053E,
	0x053F,0x059E,0x052F,0x058F,0x072E,0x078F,0x059F,0x078E,
	0x071F,0x073E,0x051F,0x070D,0x079E,0x070E,0x071D,0x0622,
	0x070F,0x071E,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_4x4 {
	4, 4,
	{ 61, 84 },
	{ 184, 141 },
	{ 0, 53 },
	{ percentile_arr_4x4_0, percentile_arr_4x4_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (5 * 4)
static const uint16_t percentile_arr_5x4_0[91] {
	0x02C1,0xFAD1,0xE8D3,0xDAC2,0xA8D2,0x70D1,0x50C2,0x80C3,
	0xD2C3,0x4AA2,0x2AD2,0x2242,0x2251,0x42A3,0x1A43,0x4A52,
	0x32B3,0x2A41,0x1042,0x1851,0x5892,0x10A2,0x2253,0x10B2,
	0x10B3,0x13DF,0x3083,0x08B1,0x1043,0x12B1,0x0AB2,0x1A93,
	0x1852,0x1A33,0x09CE,0x08A3,0x1022,0x1283,0x0853,0x1AA1,
	0x1093,0x11DE,0x135F,0x1832,0x195F,0x0A81,0x11CF,0x0A31,
	0x09DF,0x0B4D,0x09AF,0x03CF,0x0813,0x03DD,0x0A92,0x0A82,
	0x03CD,0x0023,0x0BDE,0x0BBF,0x1232,0x0221,0x0291,0x0A23,
	0x0833,0x035D,0x0BCE,0x01BF,0x0222,0x134E,0x0213,0x0A01,
	0x0B4F,0x0B5E,0x038E,0x032E,0x03AF,0x0A11,0x03AD,0x0203,
	0x0202,0x0BBD,0x033E,0x03AE,0x03BE,0x0212,0x033F,0x039E,
	0x039F,0x032F,0x038F
};

static const uint16_t percentile_arr_5x4_1[104] {
	0x0433,0xB621,0x5452,0x4443,0x7FAE,0xFCA3,0x7CC2,0x24B2,
	0x45DF,0x44B3,0x7631,0x27CD,0x1CD1,0x1E03,0x4FBE,0x774F,
	0x1C42,0x7691,0x24A2,0x2681,0x3C23,0x3C93,0x0FBD,0x1C32,
	0x1E82,0x1E12,0x0F4E,0x1602,0x0FAD,0x0C51,0x1FDD,0x0E13,
	0x0DCF,0x175E,0x0C22,0x175F,0x15DE,0x0CB1,0x17AF,0x1CC1,
	0x1F3F,0x1483,0x0441,0x0C91,0x04D2,0x0DCE,0x154E,0x079F,
	0x0CA1,0x0F5D,0x0431,0x15DD,0x05BF,0x0C92,0x0611,0x0C82,
	0x0402,0x074D,0x0DBD,0x055E,0x05BE,0x0DCD,0x0421,0x05AF,
	0x0403,0x0D4F,0x055F,0x05AE,0x0413,0x0E01,0x055D,0x073D,
	0x0C12,0x0692,0x0411,0x072D,0x078D,0x079D,0x058E,0x0D2E,
	0x0453,0x072F,0x059E,0x052F,0x071F,0x053F,0x053E,0x078F,
	0x058F,0x051F,0x0F2E,0x059F,0x078E,0x073E,0x071D,0x070D,
	0x070E,0x079E,0x0622,0x0683,0x070F,0x071E,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_5x4 {
	5, 4,
	{ 91, 104 },
	{ 322, 464 },
	{ 0, 202 },
	{ percentile_arr_5x4_0, percentile_arr_5x4_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (5 * 5)
static const uint16_t percentile_arr_5x5_0[129] {
	0x00F3,0xF8F2,0x70E3,0x62E1,0x60E1,0x4AC1,0x3261,0x38D3,
	0x3271,0x5AF1,0x5873,0x2AD1,0x28E2,0x28F1,0x2262,0x9AC2,
	0x18D2,0x1072,0x1071,0x22A2,0x2062,0x1A51,0x10C2,0x0892,
	0x08D1,0x1AA3,0x23EE,0x08C3,0x0BEF,0x2242,0x0863,0x0AB3,
	0x0BFF,0x0A93,0x08A2,0x0A41,0x1083,0x0842,0x10B3,0x21EE,
	0x10B2,0x00B1,0x1263,0x12C3,0x0A83,0x0851,0x11FE,0x0253,
	0x09FD,0x0A72,0x09FF,0x1AB2,0x0BDF,0x0A33,0x0243,0x0B7F,
	0x0AB1,0x12D2,0x0252,0x096F,0x00A3,0x0893,0x0822,0x0843,
	0x097E,0x097F,0x01EF,0x09CE,0x03FE,0x0A81,0x036F,0x0052,
	0x13FD,0x0AA1,0x1853,0x036D,0x0A92,0x0832,0x01DE,0x0A82,
	0x0BED,0x0231,0x0BBF,0x03DD,0x0B6E,0x01AF,0x0813,0x0023,
	0x0A91,0x015F,0x037E,0x01CF,0x0232,0x0BCD,0x0221,0x0BDE,
	0x0213,0x035F,0x0B7D,0x0223,0x01BF,0x0BCF,0x01DF,0x0033,
	0x0222,0x03CE,0x0A01,0x03AF,0x034D,0x0B8E,0x032E,0x0203,
	0x0211,0x0202,0x0B5D,0x03AD,0x034E,0x03AE,0x034F,0x033F,
	0x039F,0x03BD,0x03BE,0x035E,0x0212,0x033E,0x039E,0x032F,
	0x038F
};

static const uint16_t percentile_arr_5x5_1[126] {
	0x0443,0x6452,0xFE21,0x27AE,0x2433,0x1FCD,0x25DF,0x6CC2,
	0x2C62,0x1F4F,0x4C42,0x1FBE,0x0DEF,0x34A3,0x0E03,0x54B2,
	0x1F7D,0x17DD,0x0DFF,0x0CD1,0x0E31,0x0C71,0x1CF1,0x15FE,
	0x1691,0x1681,0x24B3,0x174E,0x0F6E,0x0493,0x175E,0x1C51,
	0x17BD,0x076D,0x2CA2,0x05EE,0x1472,0x2423,0x0DCF,0x0432,
	0x15DE,0x0612,0x0CD2,0x0682,0x0F5F,0x07AD,0x0602,0x0CE1,
	0x0C91,0x0FAF,0x073F,0x0E13,0x0D7F,0x0DCE,0x0422,0x0D7D,
	0x0441,0x05FD,0x0CB1,0x0C83,0x04C1,0x0461,0x0F9F,0x0DDD,
	0x056E,0x0C92,0x0482,0x0431,0x05ED,0x0D6F,0x075D,0x0402,
	0x057E,0x0DBF,0x04A1,0x054E,0x0F4D,0x0403,0x05CD,0x0453,
	0x05AE,0x0421,0x0F1F,0x05BE,0x0601,0x0611,0x05BD,0x05AF,
	0x078D,0x072D,0x073D,0x055E,0x0F9D,0x0411,0x0413,0x0412,
	0x055F,0x077E,0x055D,0x052E,0x054F,0x053E,0x058E,0x078F,
	0x059E,0x071D,0x0E92,0x053F,0x059F,0x051F,0x072F,0x052F,
	0x070D,0x079E,0x058F,0x072E,0x070E,0x078E,0x070F,0x073E,
	0x0622,0x0683,0x071E,0x076F,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_5x5 {
	5, 5,
	{ 129, 126 },
	{ 258, 291 },
	{ 0, 116 },
	{ percentile_arr_5x5_0, percentile_arr_5x5_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (6 * 5)
static const uint16_t percentile_arr_6x5_0[165] {
	0x0163,0xF8F3,0x9962,0x8972,0x7961,0x7173,0x6953,0x5943,
	0x4B41,0x3AE1,0x38E3,0x6971,0x32C1,0x28D3,0x2A61,0xC8F2,
	0x2271,0x4873,0x5B21,0x3AD1,0x1B13,0x1952,0x1B51,0x12F1,
	0x1A62,0x1322,0x1951,0x10E2,0x1B31,0x20F1,0x2102,0x2072,
	0x10D2,0x1142,0x2912,0x3871,0x2BEE,0x0862,0x1123,0x0AC2,
	0x12A2,0x0A51,0x1922,0x0941,0x1BEF,0x0B42,0x08D1,0x13FF,
	0x1933,0x08C3,0x08C2,0x1131,0x08E1,0x2903,0x0863,0x0B32,
	0x1132,0x1AC3,0x0A42,0x1A41,0x0042,0x21EE,0x09FF,0x03DF,
	0x0AA3,0x11FE,0x02B3,0x0B11,0x10B3,0x0B03,0x11FD,0x0913,
	0x0A53,0x037F,0x1263,0x0051,0x0A33,0x0B01,0x016F,0x0A72,
	0x1312,0x08A2,0x10B1,0x0BFE,0x11EF,0x0B02,0x0A52,0x0043,
	0x0822,0x01CE,0x0A43,0x097F,0x036F,0x08B2,0x03FD,0x0A83,
	0x0B33,0x0AB1,0x017E,0x0B23,0x0852,0x02D2,0x0BBF,0x0BDD,
	0x03ED,0x0AB2,0x02A1,0x0853,0x036D,0x0892,0x0032,0x0A31,
	0x0083,0x09DE,0x0A93,0x08A3,0x1213,0x0BDE,0x03CD,0x036E,
	0x037E,0x0A21,0x0023,0x0BCF,0x01CF,0x0013,0x01AF,0x0A92,
	0x0232,0x035F,0x0093,0x0B7D,0x015F,0x0282,0x01BF,0x09DF,
	0x03CE,0x0223,0x0833,0x0222,0x03AF,0x0A01,0x0291,0x0B4D,
	0x032E,0x038E,0x0203,0x0281,0x035D,0x03AD,0x0B9F,0x0202,
	0x034F,0x03BE,0x0211,0x03AE,0x03BD,0x0212,0x034E,0x033F,
	0x033E,0x035E,0x039E,0x032F,0x038F
};

static const uint16_t percentile_arr_6x5_1[145] {
	0x0443,0xEFAE,0x2CC2,0x2E21,0x2C52,0x7C33,0x47CD,0x25DF,
	0x3CA3,0xFFBE,0x2551,0x24B3,0x474F,0x1513,0x2691,0x1603,
	0x1462,0x1D32,0x14B2,0x5442,0x2CD2,0x35EF,0x0CD1,0x3D22,
	0x17BD,0x0FDD,0x0DFF,0x2631,0x177D,0x0CF1,0x1E81,0x0E82,
	0x1DFE,0x0F5E,0x0701,0x2CA2,0x1D03,0x0F4E,0x1471,0x0C51,
	0x1F6E,0x2FAF,0x0561,0x0C72,0x176D,0x0FAD,0x0DEE,0x05CF,
	0x0E13,0x0F5F,0x0E12,0x0C23,0x1E02,0x1D12,0x0CB1,0x0C32,
	0x0C93,0x15DE,0x0F9F,0x0F3F,0x0D41,0x0C41,0x0CC1,0x0D31,
	0x0C22,0x05FD,0x057F,0x0D01,0x0461,0x04E1,0x0D7D,0x05CE,
	0x0502,0x0C31,0x05ED,0x05DD,0x0511,0x0F11,0x0491,0x0D6F,
	0x0521,0x056E,0x0C83,0x0D23,0x04A1,0x0C02,0x075D,0x05BF,
	0x0C21,0x079D,0x0482,0x05BD,0x0DBE,0x05CD,0x054E,0x057E,
	0x0DAE,0x074D,0x078D,0x0542,0x0492,0x05AF,0x0611,0x0F3D,
	0x0601,0x071F,0x055E,0x059E,0x0571,0x054F,0x0412,0x0453,
	0x058E,0x0413,0x0D3E,0x077E,0x072D,0x052E,0x059F,0x055D,
	0x072F,0x0403,0x0411,0x058F,0x055F,0x0692,0x078E,0x053F,
	0x0D2F,0x078F,0x070D,0x071D,0x051F,0x072E,0x079E,0x070E,
	0x070F,0x073E,0x0622,0x0683,0x0702,0x071E,0x076F,0x07BF,
	0x07CE
};

static const packed_percentile_table block_pcd_6x5 {
	6, 5,
	{ 165, 145 },
	{ 388, 405 },
	{ 0, 156 },
	{ percentile_arr_6x5_0, percentile_arr_6x5_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (6 * 6)
static const uint16_t percentile_arr_6x6_0[206] {
	0x006F,0xF908,0xF104,0xE918,0xE963,0xD114,0xB0F3,0xA07E,
	0x7972,0x705F,0x687F,0x6162,0x5953,0x586E,0x610C,0x524D,
	0x5973,0x9943,0x98E3,0x904F,0x8341,0x7AC1,0x3A61,0x70D3,
	0xA073,0x6AE1,0x30F2,0x3313,0x2B21,0x9A2E,0x4322,0x225D,
	0x2331,0x2271,0x22D1,0x1A2D,0x221F,0x22F1,0x1971,0x6952,
	0x1951,0x187D,0x18F1,0x1902,0x185E,0x1B51,0x105D,0x1A3D,
	0x30E2,0x10D2,0x1961,0x12A2,0x6072,0x3942,0x386D,0x33EE,
	0x104E,0x4923,0x101E,0x2122,0x1251,0x1141,0x182F,0x3133,
	0x080E,0x1262,0x123E,0x1B32,0x102E,0x1931,0x10D1,0x1912,
	0x0871,0x12C2,0x08C2,0x1103,0x0B03,0x1062,0x083D,0x08E1,
	0x1132,0x184D,0x0863,0x08C3,0x303F,0x083E,0x10B3,0x12A3,
	0x0BEF,0x0B11,0x1A42,0x2233,0x13FF,0x080F,0x0A41,0x0AC3,
	0x0842,0x1A63,0x0BDF,0x09FF,0x12B3,0x124E,0x0B12,0x0B42,
	0x0A2F,0x1253,0x0913,0x1051,0x0B01,0x120F,0x0B02,0x08A2,
	0x0BBF,0x00B1,0x22B1,0x01EE,0x1B33,0x0B23,0x0283,0x13FD,
	0x0AB2,0x11FD,0x09FE,0x0A43,0x08B2,0x0A1D,0x0A52,0x023F,
	0x101F,0x01CE,0x0A31,0x0BDD,0x0293,0x1822,0x12A1,0x03FE,
	0x121E,0x0843,0x0272,0x0B6F,0x0052,0x0A0D,0x0BED,0x12D2,
	0x1B7F,0x1053,0x0032,0x01DE,0x08A3,0x020E,0x0883,0x09EF,
	0x0892,0x0A21,0x03CD,0x0B5F,0x0213,0x0A32,0x016F,0x1292,
	0x03DE,0x017E,0x0BAF,0x0223,0x1093,0x0BCF,0x037E,0x01DF,
	0x09CF,0x015F,0x09AF,0x0023,0x01BF,0x0222,0x0282,0x03CE,
	0x1013,0x036E,0x097F,0x0033,0x0A01,0x0B6D,0x03BE,0x037D,
	0x0281,0x0BAE,0x0203,0x032E,0x034D,0x034F,0x0291,0x0211,
	0x038E,0x03BD,0x039E,0x0BAD,0x033E,0x034E,0x039F,0x0202,
	0x035D,0x0212,0x033F,0x035E,0x038F,0x032F
};

static const uint16_t percentile_arr_6x6_1[164] {
	0x07AE,0x8443,0x7E21,0x77CD,0x6C62,0x9433,0x6452,0x34C2,
	0x5DDF,0xC7BE,0x25EF,0x24A3,0x3CF1,0xFDFF,0x177D,0x1F4F,
	0xC551,0x5CB3,0x1532,0x1513,0x143E,0x245D,0x14B2,0x2472,
	0x14D2,0x1FBD,0x1631,0x2DFE,0x1691,0x17DD,0x2E03,0x376E,
	0x2442,0x0F6D,0x3C71,0x2CD1,0x2522,0x6C51,0x260D,0x17AF,
	0x0DEE,0x1C1F,0x2F01,0x142E,0x0CA2,0x0FAD,0x3D03,0x275E,
	0x1681,0x274E,0x1682,0x1C23,0x273F,0x0F5F,0x05DE,0x15FD,
	0x0DCF,0x1E02,0x04B1,0x144D,0x0E12,0x0D12,0x1CC1,0x0E13,
	0x1C6D,0x0C32,0x043D,0x0C61,0x0F9F,0x04E1,0x0DCE,0x0D41,
	0x1C93,0x0C22,0x061D,0x0D7F,0x0C41,0x0561,0x0531,0x0D21,
	0x0711,0x0C91,0x0501,0x0C1E,0x040F,0x15DD,0x0431,0x0C2F,
	0x057D,0x0C2D,0x0DBE,0x040E,0x0D02,0x0D11,0x054E,0x040D,
	0x0D23,0x0DBF,0x04A1,0x05ED,0x0C1D,0x05BD,0x072D,0x056E,
	0x0483,0x0F3D,0x0482,0x078D,0x0F5D,0x0453,0x0D9E,0x0C4E,
	0x05CD,0x079D,0x0402,0x05AE,0x0F1F,0x0542,0x074D,0x056F,
	0x0421,0x0D4F,0x0601,0x0571,0x0492,0x059F,0x053F,0x05AF,
	0x0611,0x055E,0x0D8E,0x053E,0x055D,0x047D,0x0411,0x052E,
	0x058F,0x051F,0x055F,0x0D7E,0x072F,0x052F,0x0412,0x078F,
	0x0403,0x077E,0x070D,0x070E,0x078E,0x0F1D,0x072E,0x0413,
	0x070F,0x0692,0x079E,0x060E,0x0622,0x0683,0x0702,0x071E,
	0x073E,0x076F,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_6x6 {
	6, 6,
	{ 206, 164 },
	{ 769, 644 },
	{ 0, 256 },
	{ percentile_arr_6x6_0, percentile_arr_6x6_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (8 * 5)
static const uint16_t percentile_arr_8x5_0[226] {
	0x0066,0xF865,0xE963,0xA856,0xA1F2,0x9875,0x91C3,0x91E2,
	0x80F3,0x8076,0x61E3,0x6153,0x5172,0x59D2,0x51D3,0x5047,
	0xA943,0x49B3,0x4846,0x4962,0xC037,0x4173,0x39F1,0x7027,
	0xA2C1,0x3AE1,0x9341,0x30D3,0x5225,0x2A61,0x33C1,0x28E3,
	0x53A1,0x49C2,0x2A06,0x4055,0x2006,0x21D1,0x2271,0x4321,
	0x3873,0x18F2,0x2015,0x1A15,0x1857,0x52D1,0x3045,0x4835,
	0x1952,0x29E1,0x3207,0x1036,0x1816,0x2A16,0x2971,0x13B1,
	0x2A17,0x2351,0x1025,0x1826,0x30E2,0x1262,0x20F1,0x1007,
	0x1072,0x1151,0x10D2,0x1235,0x1205,0x1062,0x4AF1,0x1251,
	0x0B31,0x1381,0x13EE,0x1B92,0x13EF,0x0942,0x1AA2,0x13FF,
	0x1161,0x0B93,0x19A2,0x11B1,0x08D1,0x12C2,0x0B13,0x1B22,
	0x2123,0x09A3,0x2071,0x1B7F,0x1817,0x0A42,0x10C2,0x1233,
	0x08C3,0x0A41,0x0B42,0x09C1,0x0933,0x1AB3,0x1382,0x1BDF,
	0x2122,0x0A53,0x0AC3,0x20E1,0x0941,0x0931,0x0042,0x0BA2,
	0x0AA3,0x0992,0x0863,0x08B3,0x11B2,0x0902,0x1283,0x09FF,
	0x0B83,0x0982,0x0932,0x0BFE,0x0B32,0x0BBF,0x11FE,0x036F,
	0x0851,0x08B1,0x18A2,0x11EE,0x0A52,0x0BB2,0x01FD,0x0A43,
	0x1A63,0x1193,0x0B91,0x0043,0x1231,0x0A26,0x0AB1,0x03FD,
	0x096F,0x00B2,0x0983,0x0A72,0x01CE,0x0BDD,0x0022,0x0B11,
	0x1213,0x0B6D,0x017E,0x1333,0x0112,0x0852,0x02D2,0x097F,
	0x01EF,0x0AB2,0x0293,0x0853,0x0BED,0x0B12,0x1303,0x02A1,
	0x0892,0x0032,0x0883,0x0B6E,0x0292,0x0A32,0x037E,0x0B23,
	0x0103,0x0A21,0x0B01,0x0302,0x0BCD,0x00A3,0x0BCF,0x0BDE,
	0x0113,0x01DE,0x0B5F,0x0013,0x0BAF,0x0223,0x0222,0x0A82,
	0x0833,0x0023,0x09CF,0x037D,0x01AF,0x095F,0x03CE,0x09DF,
	0x01BF,0x0893,0x0203,0x0201,0x0B4D,0x03BE,0x032E,0x03AE,
	0x0291,0x0A02,0x0211,0x039F,0x0281,0x038E,0x03AD,0x033F,
	0x035D,0x033E,0x034E,0x034F,0x0212,0x03BD,0x032F,0x035E,
	0x038F,0x039E
};

static const uint16_t percentile_arr_8x5_1[167] {
	0x0621,0xFCC2,0x3443,0xA433,0x5532,0x2551,0x6CA3,0x27AE,
	0x6452,0x8E03,0x3CB3,0x4DA2,0x6DDF,0x37CD,0x6F01,0x1691,
	0x2E82,0x27BE,0x1513,0x34D2,0x1D22,0x3E31,0x2593,0x2CB2,
	0x1C16,0x374F,0x0DD1,0x2583,0x6613,0x0CD1,0x0C35,0x1462,
	0x3E81,0x2612,0x2C42,0x3407,0x14A2,0x0E02,0x1CF1,0x0C06,
	0x17BD,0x0F7D,0x1D23,0x35B1,0x179F,0x0D92,0x0F5E,0x1451,
	0x04B1,0x1F6E,0x0DEF,0x0D31,0x374E,0x15C1,0x0541,0x2405,
	0x17AD,0x0471,0x1472,0x0DFE,0x0711,0x0FDD,0x0DFF,0x0432,
	0x1D82,0x0423,0x0F6D,0x07AF,0x0F5F,0x04C1,0x1542,0x0561,
	0x0DCF,0x1D03,0x1493,0x0422,0x0445,0x0D12,0x0C25,0x0415,
	0x0DA1,0x1591,0x0DEE,0x05DE,0x0C31,0x0491,0x0441,0x0D21,
	0x078D,0x057D,0x0C61,0x0F3F,0x0581,0x0D6E,0x0501,0x0CA1,
	0x04E1,0x0DFD,0x057F,0x0502,0x0511,0x0C82,0x0483,0x0C03,
	0x079D,0x0402,0x0DDD,0x0611,0x05AE,0x0DCE,0x056F,0x0421,
	0x057E,0x071F,0x0DBF,0x05BE,0x0412,0x059F,0x054E,0x077E,
	0x0C26,0x05ED,0x073D,0x0601,0x0492,0x0453,0x075D,0x058E,
	0x0F2D,0x05CD,0x0571,0x053E,0x0692,0x05BD,0x054F,0x055E,
	0x0411,0x0F1D,0x074D,0x059E,0x05AF,0x070D,0x053F,0x058F,
	0x0413,0x070F,0x055D,0x070E,0x078F,0x052E,0x072F,0x055F,
	0x078E,0x0F2E,0x052F,0x051F,0x0417,0x071E,0x0781,0x0622,
	0x0683,0x0702,0x073E,0x076F,0x079E,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_8x5 {
	8, 5,
	{ 226, 167 },
	{ 763, 517 },
	{ 0, 178 },
	{ percentile_arr_8x5_0, percentile_arr_8x5_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (8 * 6)
static const uint16_t percentile_arr_8x6_0[273] {
	0x0154,0xF944,0xE066,0xA128,0x9963,0x8118,0x806F,0x79F2,
	0x79E2,0x7108,0xD934,0x6056,0x69C3,0x60F3,0x5972,0x59E3,
	0x5075,0x91B3,0xC9D2,0x807E,0x385F,0x4153,0x3943,0x4162,
	0x3837,0x3847,0x7173,0x31D3,0x6948,0x3046,0x307F,0x5827,
	0x3114,0x32C1,0x3076,0x2A4D,0x58E3,0x306E,0x2924,0x2A61,
	0x29F1,0x50D3,0x704F,0x210C,0x2BA1,0x2225,0x2873,0x4865,
	0x2206,0x8341,0x2006,0x3B21,0x18F2,0x21C2,0x1A1F,0x23C1,
	0x3AE1,0x1855,0x19D1,0x1A15,0x3815,0x1207,0x1835,0x2A2E,
	0x1A16,0x1836,0x2271,0x2845,0x1A2D,0x11E1,0x1816,0x1171,
	0x2217,0x1952,0x12D1,0x3904,0x125D,0x4BB1,0x207D,0x10E2,
	0x1026,0x2025,0x12F1,0x28F1,0x105D,0x1235,0x12A2,0x1007,
	0x123D,0x1A05,0x1072,0x1331,0x101E,0x0951,0x10D2,0x1057,
	0x1B92,0x185E,0x1251,0x19A2,0x186D,0x0B81,0x2BEE,0x080E,
	0x1A33,0x1942,0x0B13,0x0B51,0x11A3,0x0923,0x2322,0x09B1,
	0x184E,0x1161,0x18D1,0x0933,0x0B93,0x4A62,0x1017,0x082F,
	0x0A42,0x0B82,0x0AA3,0x0A41,0x08C2,0x08B3,0x0A3E,0x22B3,
	0x0871,0x1BBF,0x09C1,0x0AC2,0x09B2,0x0BEF,0x082E,0x1062,
	0x0922,0x08C3,0x1063,0x0A53,0x0BDF,0x080F,0x0B42,0x0A83,
	0x084D,0x103F,0x0931,0x08E1,0x0A0F,0x1BA2,0x09FF,0x1332,
	0x03FF,0x0941,0x12C3,0x0A63,0x003D,0x0842,0x083E,0x0B83,
	0x0BB2,0x0A31,0x0932,0x1102,0x0992,0x0982,0x1051,0x08B1,
	0x0A2F,0x121E,0x02B1,0x0A4E,0x11EE,0x00A2,0x1022,0x0043,
	0x0A52,0x0A1D,0x0226,0x1193,0x03DD,0x08B2,0x0BFD,0x0A43,
	0x0A13,0x0AB2,0x01FD,0x09FE,0x020D,0x081F,0x0B33,0x0053,
	0x0B91,0x0293,0x0B11,0x0B7F,0x0AA1,0x0B03,0x0A0E,0x03FE,
	0x01CE,0x0B6F,0x0183,0x0912,0x023F,0x0852,0x0A21,0x0323,
	0x03ED,0x0A32,0x13AF,0x0272,0x08A3,0x0B12,0x0083,0x0832,
	0x13CD,0x0223,0x0A92,0x0092,0x0AD2,0x0301,0x0302,0x0BDE,
	0x0A22,0x01EF,0x0B5F,0x0103,0x0BCF,0x096F,0x017E,0x0113,
	0x01DE,0x0823,0x0282,0x0B6E,0x015F,0x0813,0x01AF,0x01CF,
	0x0B7E,0x0033,0x01DF,0x0BCE,0x01BF,0x036D,0x0A03,0x017F,
	0x03BE,0x0201,0x0893,0x038E,0x034D,0x03AE,0x0202,0x039F,
	0x0291,0x0A11,0x032E,0x033F,0x034F,0x0281,0x037D,0x03BD,
	0x0212,0x033E,0x035E,0x034E,0x035D,0x03AD,0x032F,0x038F,
	0x039E
};

static const uint16_t percentile_arr_8x6_1[186] {
	0x0621,0xFC33,0x37AE,0x1CC2,0x2C43,0xAD32,0x34A3,0x4551,
	0x6452,0x5C62,0x1FCD,0x14F1,0x4CB3,0x24D2,0x15DF,0x0FBE,
	0x2603,0x3DA2,0x2E31,0x25D1,0x25EF,0x0D22,0x2E91,0x1E82,
	0x0FBD,0x1513,0x0CB2,0x0CD1,0x0F4F,0x1F7D,0x1701,0x0C16,
	0x2593,0x2C42,0x0C72,0x14A2,0x0F6E,0x0C35,0x0C71,0x0D83,
	0x0C07,0x1DFF,0x043E,0x1613,0x07DD,0x0FAD,0x1451,0x076D,
	0x0E81,0x05FE,0x0406,0x0E0D,0x045D,0x2612,0x0E02,0x07AF,
	0x0DB1,0x0F5E,0x15C1,0x0C23,0x1523,0x0C1F,0x0D92,0x04B1,
	0x0D31,0x0432,0x0D61,0x0F4E,0x0D41,0x0DEE,0x0D42,0x04C1,
	0x0CE1,0x079F,0x0C2E,0x0405,0x0C22,0x0461,0x0E1D,0x0582,
	0x073F,0x0571,0x0C4D,0x0DFD,0x05CE,0x0C6D,0x05DE,0x0415,
	0x0C45,0x075F,0x0C41,0x0D03,0x05A1,0x0711,0x05CF,0x0425,
	0x0C93,0x0D21,0x0591,0x043D,0x0D12,0x0501,0x040F,0x0511,
	0x0431,0x0C03,0x04A1,0x078D,0x0581,0x041E,0x040D,0x0C02,
	0x040E,0x05DD,0x057F,0x079D,0x042D,0x0D9F,0x0502,0x056E,
	0x0412,0x071F,0x044E,0x05BF,0x0C1D,0x0482,0x05AE,0x042F,
	0x057D,0x0491,0x054E,0x047D,0x0DBE,0x0611,0x0492,0x0601,
	0x05BD,0x05CD,0x0426,0x05ED,0x072D,0x073D,0x0483,0x0F5D,
	0x0421,0x056F,0x053F,0x058E,0x054F,0x078F,0x053E,0x059E,
	0x057E,0x051F,0x055D,0x0413,0x070D,0x05AF,0x0411,0x0453,
	0x0D5E,0x077E,0x052F,0x070F,0x074D,0x0692,0x070E,0x072F,
	0x072E,0x058F,0x071D,0x052E,0x0417,0x073E,0x0781,0x078E,
	0x055F,0x060E,0x0622,0x0683,0x0702,0x071E,0x076F,0x079E,
	0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_8x6 {
	8, 6,
	{ 273, 186 },
	{ 880, 300 },
	{ 0, 64 },
	{ percentile_arr_8x6_0, percentile_arr_8x6_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (8 * 8)
static const uint16_t percentile_arr_8x8_0[347] {
	0x0334,0xFD44,0xDD14,0x9154,0x9B08,0x906A,0x8928,0x8108,
	0xE866,0xC918,0x606F,0xC0FE,0x5963,0x58EE,0x6534,0x505A,
	0x51E2,0xA8CF,0x5354,0x5314,0x5134,0x5524,0x48F3,0x504B,
	0x487E,0x5344,0x49C3,0x4972,0x49F2,0x4856,0xD0EF,0x81D2,
	0x78DE,0x4261,0x3AC1,0x71E3,0x6879,0x390C,0x3143,0x31B3,
	0x385F,0x3153,0x306E,0x3037,0x30DF,0x3162,0x304F,0x3075,
	0xB03B,0x2847,0x28E3,0x2914,0x507F,0x28BF,0x5173,0x5073,
	0x20D3,0x2A06,0x2827,0x2508,0x2229,0x29D3,0x204A,0x207A,
	0x2046,0x4148,0x20FD,0x4225,0x23A1,0x3944,0x2065,0x1924,
	0x2324,0x1806,0x19F1,0x2215,0x1876,0x22AD,0x502B,0x1B04,
	0x18F2,0x3A4D,0x3216,0x3504,0x18DD,0x1B21,0x10CE,0x1869,
	0x1B41,0x1855,0x1207,0x1AE1,0x2845,0x19D1,0x2A0A,0x1A2D,
	0x2A1A,0x11C2,0x1A0B,0x1217,0x2816,0x121B,0x1271,0x2AD1,
	0x1035,0x1015,0x287D,0x12F1,0x43C1,0x1171,0x1A05,0x08E2,
	0x11E1,0x3251,0x2049,0x20F1,0x12CD,0x0A39,0x1219,0x1059,
	0x1104,0x1036,0x1872,0x3007,0x08ED,0x205E,0x1026,0x0952,
	0x1392,0x1019,0x0951,0x100A,0x13EE,0x08D2,0x1242,0x0ABD,
	0x22A2,0x0BDF,0x2B81,0x0A35,0x13B1,0x0839,0x13BF,0x0A33,
	0x1B31,0x205D,0x1241,0x183A,0x2025,0x0B93,0x0A3D,0x1017,
	0x1313,0x1253,0x082A,0x204E,0x09A2,0x080B,0x0A1F,0x125D,
	0x0A2E,0x081A,0x08D1,0x082F,0x086D,0x1B82,0x0A09,0x0B22,
	0x1062,0x11A3,0x2161,0x0923,0x129F,0x1A62,0x0871,0x0942,
	0x081B,0x1133,0x18AE,0x0A9E,0x0863,0x09FF,0x18C2,0x0B51,
	0x08BD,0x0AA3,0x09B1,0x1AC2,0x08B3,0x0829,0x0BEF,0x0B83,
	0x0AAE,0x0A8D,0x1857,0x185B,0x08AF,0x103F,0x08C3,0x09B2,
	0x0A4E,0x11C1,0x0A31,0x0B42,0x0A83,0x0BFF,0x13DD,0x00CD,
	0x0AB3,0x0842,0x08BE,0x0922,0x1A8E,0x08E1,0x002E,0x0BA2,
	0x0A8F,0x2263,0x0252,0x0B32,0x0AC3,0x0941,0x0A43,0x083D,
	0x083E,0x0A3E,0x084D,0x1131,0x136F,0x0AB1,0x0193,0x0BFD,
	0x0391,0x0851,0x13AF,0x0843,0x0213,0x1226,0x0932,0x03B2,
	0x0902,0x0BCD,0x0221,0x089E,0x00B1,0x0BDE,0x03FE,0x02A1,
	0x0982,0x009F,0x080E,0x0B5F,0x02BE,0x0A32,0x0A2A,0x01EE,
	0x0053,0x0AB2,0x0192,0x09FD,0x0052,0x0B03,0x0293,0x00A2,
	0x0B7F,0x0BED,0x0311,0x08B2,0x0A72,0x088E,0x0333,0x0B12,
	0x0A23,0x0822,0x0083,0x11CE,0x021D,0x08A3,0x088F,0x029D,
	0x0A22,0x0A3F,0x01FE,0x020F,0x0983,0x02D2,0x0292,0x0B23,
	0x001E,0x0BCF,0x03CE,0x09AF,0x0B02,0x0301,0x022F,0x137E,
	0x021E,0x09EF,0x016F,0x0112,0x097E,0x080F,0x020D,0x0092,
	0x01DE,0x09DF,0x0032,0x0033,0x0A82,0x03BE,0x0B6E,0x001F,
	0x020E,0x0023,0x09CF,0x0113,0x0103,0x0013,0x0BAE,0x0203,
	0x0BAD,0x01BF,0x034F,0x095F,0x036D,0x0202,0x017F,0x0093,
	0x0201,0x034D,0x0212,0x035D,0x03BD,0x0B3F,0x035E,0x0211,
	0x0281,0x0291,0x032E,0x037D,0x034E,0x038E,0x039F,0x032F,
	0x033E,0x038F,0x039E
};

static const uint16_t percentile_arr_8x8_1[208] {
	0x0621,0x3443,0x47CD,0x97AE,0xFC62,0x14F1,0x24C2,0x25DF,
	0x3C33,0x1C52,0x9C72,0x0FBE,0x0C5D,0x343E,0x24A3,0x1551,
	0x5D32,0x1CD2,0x15EF,0x4E31,0x04DD,0x1FDD,0x174F,0x0DD1,
	0x3E0D,0x15FF,0x0DA2,0x1E03,0x17BD,0x177D,0x14B3,0x0471,
	0x0CAE,0x1C1F,0x04D1,0x0F6E,0x0DFE,0x1C42,0x0C16,0x0D22,
	0x0C9F,0x2C2E,0x0FAD,0x0571,0x147D,0x0C07,0x04B2,0x0F6D,
	0x0F5E,0x07AF,0x146D,0x0C51,0x0593,0x2583,0x0C4E,0x040B,
	0x0C35,0x0513,0x0E91,0x0406,0x073F,0x144D,0x0561,0x048F,
	0x0F01,0x0F4E,0x0CA2,0x075F,0x1682,0x04E1,0x0C1A,0x04BD,
	0x0542,0x0D41,0x0DEE,0x04CD,0x0DCF,0x04B1,0x0C15,0x0C3D,
	0x0423,0x0592,0x0DDE,0x0422,0x0432,0x05FD,0x0DC1,0x05B1,
	0x0DCE,0x0612,0x0C2F,0x0445,0x0602,0x0531,0x0439,0x0E81,
	0x0582,0x0C61,0x061D,0x049E,0x0405,0x0409,0x0DBE,0x079F,
	0x0D21,0x04C1,0x0C0A,0x0E13,0x04AD,0x040E,0x0581,0x0419,
	0x05DD,0x0D03,0x049D,0x0449,0x0429,0x048E,0x0DA1,0x0425,
	0x0512,0x0501,0x0431,0x0523,0x0441,0x042D,0x040F,0x0D7D,
	0x0511,0x0502,0x05BF,0x04A1,0x0C03,0x0402,0x079D,0x05AE,
	0x075D,0x057F,0x041D,0x048D,0x042A,0x0453,0x05AF,0x078D,
	0x0C0D,0x073D,0x0491,0x0591,0x05BD,0x072D,0x057E,0x051F,
	0x0482,0x0492,0x041E,0x0412,0x0D9F,0x0421,0x0493,0x0711,
	0x056E,0x059E,0x054E,0x0611,0x05ED,0x074D,0x070F,0x056F,
	0x052F,0x053F,0x071F,0x054F,0x05CD,0x0483,0x055E,0x072F,
	0x0E01,0x0426,0x058F,0x0413,0x078F,0x071D,0x055F,0x058E,
	0x0411,0x053E,0x071E,0x055D,0x077E,0x052E,0x0692,0x0417,
	0x070D,0x078E,0x070E,0x072E,0x041B,0x060E,0x0622,0x0683,
	0x068D,0x0702,0x073E,0x076F,0x0781,0x079E,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_8x8 {
	8, 8,
	{ 347, 208 },
	{ 1144, 267 },
	{ 0, 38 },
	{ percentile_arr_8x8_0, percentile_arr_8x8_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 5)
static const uint16_t percentile_arr_10x5_0[274] {
	0x0165,0xF975,0xD866,0xC056,0xA946,0x90C6,0x90F5,0x8963,
	0x80D6,0x80E6,0x60F3,0x61C3,0x59F2,0xA927,0x5075,0x4847,
	0x5153,0x4955,0x49E2,0x48B6,0x41D2,0x4943,0x8305,0x8172,
	0x4046,0x4037,0x40A7,0x70B7,0x7AC1,0x31E3,0x7027,0x30E5,
	0x69D3,0x99B3,0x3315,0x6115,0x3136,0x3076,0x3173,0x30D5,
	0x3106,0x8962,0x2916,0x30C7,0x5126,0x30D3,0x2956,0x5117,
	0x2B41,0x2AE1,0x2A61,0x29F1,0x2306,0x2145,0x4A85,0x2057,
	0x40E3,0x4137,0x3B21,0x23C1,0x2065,0x1925,0x51C2,0x5225,
	0x4935,0x1AD1,0x23A1,0x19D1,0x1A71,0x4055,0x1873,0x1A86,
	0x1295,0x18F2,0x28A6,0x1952,0x4AA5,0x20B5,0x10C5,0x2AA2,
	0x11E1,0x1107,0x10D2,0x2171,0x1351,0x3036,0x1331,0x1BEE,
	0x2035,0x1045,0x1313,0x0A15,0x1087,0x1296,0x13EF,0x18E2,
	0x1151,0x1086,0x10F1,0x08A5,0x12C2,0x1BFF,0x1095,0x1A62,
	0x1322,0x0942,0x1026,0x1872,0x1062,0x0897,0x1123,0x08D1,
	0x1A06,0x0806,0x137F,0x13B1,0x13DF,0x1A51,0x09B1,0x0A83,
	0x1015,0x22F1,0x0961,0x0B81,0x12B3,0x0A35,0x0AA3,0x20B3,
	0x08C3,0x2342,0x0933,0x0A33,0x09A2,0x10C2,0x0896,0x2205,
	0x0825,0x20E1,0x0922,0x1242,0x0B16,0x0B32,0x09A3,0x0AC3,
	0x0BBF,0x0B93,0x0071,0x0931,0x0A41,0x2392,0x13FE,0x09C1,
	0x0B07,0x0016,0x1182,0x09B2,0x0A26,0x0132,0x0941,0x0A93,
	0x0992,0x1063,0x1217,0x01FF,0x11EE,0x1216,0x0B23,0x0B82,
	0x0042,0x1102,0x0213,0x0B6F,0x09FE,0x1207,0x0807,0x18B1,
	0x0253,0x0AB1,0x08A2,0x13FD,0x01FD,0x1983,0x0AB2,0x0A31,
	0x016F,0x0B11,0x00B2,0x0851,0x0AD2,0x0993,0x0BDD,0x12A1,
	0x017F,0x0A97,0x1022,0x0383,0x0843,0x0A52,0x03A2,0x097E,
	0x0817,0x03B2,0x0A43,0x09EF,0x0A63,0x0B33,0x0B03,0x0292,
	0x0272,0x09CE,0x0287,0x136D,0x0053,0x0B12,0x0083,0x0892,
	0x0112,0x1282,0x03ED,0x0852,0x0301,0x1391,0x0232,0x0B7E,
	0x0221,0x08A3,0x0BCD,0x0BCF,0x036E,0x09DE,0x0103,0x03DE,
	0x0832,0x0BAF,0x0302,0x13CE,0x035F,0x0093,0x0A23,0x01DF,
	0x0013,0x0A22,0x0023,0x0113,0x09AF,0x01BF,0x0033,0x095F,
	0x0203,0x0281,0x09CF,0x037D,0x0201,0x0B4D,0x03AE,0x03BE,
	0x0291,0x035E,0x038E,0x0B9F,0x03AD,0x0202,0x034F,0x0211,
	0x035D,0x0212,0x032E,0x039E,0x033F,0x034E,0x03BD,0x032F,
	0x033E,0x038F
};

static const uint16_t percentile_arr_10x5_1[180] {
	0x0532,0xFCA3,0x3621,0x6E82,0x2CC2,0x3D51,0x3F01,0x2691,
	0x17AE,0x35A2,0x74B3,0x1603,0x4433,0x3C43,0x6C35,0x25D1,
	0x1D13,0x15DF,0x37CD,0x0D93,0x1D22,0x0E81,0x1452,0x0CD2,
	0x37BE,0x0CB2,0x3407,0x1523,0x0C16,0x0CB5,0x0C96,0x1486,
	0x2631,0x1506,0x0F4F,0x1583,0x0CD1,0x2CA2,0x2612,0x1613,
	0x1602,0x1F11,0x179F,0x17BD,0x15B1,0x0406,0x1D41,0x0CF1,
	0x0D31,0x0442,0x1C62,0x0F6E,0x077D,0x0C51,0x0445,0x0D15,
	0x2592,0x0CB1,0x05EF,0x0542,0x17AF,0x1425,0x075E,0x0FAD,
	0x0CC1,0x0503,0x0512,0x15C1,0x0C95,0x0415,0x0505,0x0F4E,
	0x04A5,0x0493,0x0C32,0x0F5F,0x04E1,0x0521,0x0C85,0x07DD,
	0x0582,0x15FF,0x05CF,0x0405,0x0D91,0x05A1,0x05FE,0x0C23,
	0x0561,0x0472,0x0471,0x0C22,0x0DEE,0x076D,0x0502,0x0426,
	0x0C61,0x0D7D,0x0525,0x05DE,0x0DCE,0x079D,0x0692,0x0441,
	0x0C91,0x05DD,0x0511,0x057F,0x0611,0x0DFD,0x078D,0x056E,
	0x0492,0x04A1,0x073F,0x0C31,0x05BE,0x0483,0x0571,0x056F,
	0x0D9F,0x0581,0x0501,0x057E,0x05BF,0x078F,0x0516,0x05ED,
	0x0402,0x0F7E,0x0482,0x054E,0x075D,0x071F,0x05CD,0x0535,
	0x05AE,0x0C11,0x058F,0x05AF,0x0421,0x0413,0x0601,0x054F,
	0x073D,0x059E,0x0487,0x070F,0x078E,0x0781,0x053E,0x0403,
	0x072D,0x055D,0x05BD,0x079E,0x0D8E,0x0412,0x052E,0x074D,
	0x053F,0x051F,0x070E,0x055F,0x072F,0x052F,0x070D,0x055E,
	0x0417,0x0453,0x072E,0x0622,0x0683,0x0702,0x071D,0x071E,
	0x073E,0x076F,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_10x5 {
	10, 5,
	{ 274, 180 },
	{ 954, 324 },
	{ 0, 79 },
	{ percentile_arr_10x5_0, percentile_arr_10x5_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 6)
static const uint16_t percentile_arr_10x6_0[325] {
	0x01A4,0xF954,0xA066,0x9975,0x80F5,0x7056,0x6918,0x6963,
	0x58C6,0x5946,0x5928,0x5174,0x586F,0xA0E6,0x5108,0x48D6,
	0x49E2,0x40F3,0x9172,0x41F2,0xB875,0x3927,0x39C3,0xA953,
	0x3934,0x3305,0x30B6,0x6943,0x31D2,0x3876,0x3037,0x2955,
	0x30A7,0x32C1,0x29B3,0x3027,0x287E,0x30B7,0x29E3,0x5846,
	0x2B15,0x2847,0x3162,0x5173,0x4936,0x285F,0x48D3,0x2164,
	0x4906,0x20E5,0x2915,0x2116,0x407F,0x20D5,0x2A61,0x4117,
	0x20E3,0x2126,0x4148,0x206E,0x39D3,0x2145,0x41B4,0x1B06,
	0x2114,0x2165,0x5321,0x5A85,0x1A4D,0x1A1F,0x19F1,0x3341,
	0x184F,0x1956,0x3125,0x30C7,0x28F2,0x1937,0x1AE1,0x1073,
	0x1BA1,0x1935,0x110C,0x1BC1,0x3A25,0x19C2,0x1295,0x122E,
	0x1944,0x11D1,0x1124,0x1857,0x22D1,0x2286,0x1A2D,0x12A2,
	0x2107,0x1055,0x2065,0x0A71,0x2152,0x10C5,0x10D2,0x1331,
	0x08B5,0x1171,0x2836,0x10A6,0x0904,0x123D,0x20F1,0x12A5,
	0x10E2,0x107D,0x1AF1,0x1313,0x0951,0x11E1,0x1B22,0x1B51,
	0x0835,0x101E,0x0A5D,0x0A15,0x3045,0x0A96,0x08A5,0x1142,
	0x12A3,0x1872,0x085D,0x09B1,0x100E,0x0887,0x0886,0x086D,
	0x0933,0x12B3,0x0897,0x08B3,0x0A33,0x0923,0x1095,0x0BEE,
	0x2BB1,0x085E,0x1283,0x0A51,0x1026,0x0A06,0x12C2,0x08D1,
	0x11A2,0x13BF,0x08C3,0x10C2,0x0A3E,0x0BDF,0x0B81,0x13EF,
	0x0A35,0x0B16,0x082F,0x2161,0x1B32,0x0806,0x084E,0x11A3,
	0x1015,0x1122,0x2931,0x0342,0x0825,0x0A0F,0x0896,0x0A05,
	0x0241,0x09C1,0x083F,0x0A42,0x0071,0x0B07,0x082E,0x0393,
	0x12B1,0x0A62,0x0226,0x0A2F,0x0B92,0x0063,0x0932,0x0862,
	0x09FF,0x0A31,0x00E1,0x12B2,0x09B2,0x0AC3,0x0941,0x0293,
	0x1323,0x104D,0x003E,0x083D,0x0992,0x1382,0x03FF,0x0A13,
	0x1016,0x0A53,0x0182,0x1007,0x0AA1,0x080F,0x0A16,0x0A1E,
	0x0042,0x0902,0x13DD,0x0BB2,0x0A63,0x00A2,0x08B1,0x03FE,
	0x1207,0x08B2,0x0B83,0x09EE,0x0311,0x0A87,0x0BAF,0x03A2,
	0x09FD,0x0051,0x0B33,0x020D,0x09CE,0x0217,0x021D,0x0817,
	0x020E,0x0A4E,0x001F,0x0BFD,0x0297,0x0983,0x0A92,0x0252,
	0x0243,0x0B03,0x0193,0x036F,0x0B12,0x0043,0x0822,0x0A21,
	0x01FE,0x0853,0x037F,0x023F,0x0BED,0x02D2,0x0B91,0x0232,
	0x0282,0x0912,0x08A3,0x0852,0x0223,0x0BCD,0x0083,0x0301,
	0x0832,0x01EF,0x0892,0x0302,0x0A72,0x03DE,0x0893,0x0BCF,
	0x09DE,0x03CE,0x035F,0x0833,0x0023,0x0103,0x017E,0x0813,
	0x01CF,0x01BF,0x016F,0x0A22,0x037E,0x0113,0x01AF,0x0B6E,
	0x03BE,0x0201,0x0A03,0x01DF,0x036D,0x03AE,0x015F,0x0281,
	0x033E,0x0A02,0x038E,0x017F,0x0291,0x034D,0x03BD,0x0B7D,
	0x03AD,0x0211,0x0212,0x034F,0x032E,0x039F,0x034E,0x035D,
	0x035E,0x033F,0x039E,0x032F,0x038F
};

static const uint16_t percentile_arr_10x6_1[199] {
	0x0621,0xBD32,0x5CA3,0x1FAE,0x64C2,0x1D51,0x6C33,0xFC43,
	0x5CB3,0x25A2,0x2E82,0x35D1,0x4F01,0x3FBE,0x3691,0x2DDF,
	0x2E03,0x3FCD,0x14D2,0x1CF1,0x0C52,0x3C35,0x2D22,0x1513,
	0x1462,0x54B2,0x0E31,0x4E81,0x1593,0x1D23,0x1CD1,0x14B5,
	0x2FBD,0x0C07,0x1D06,0x0DEF,0x14A2,0x1612,0x1F4F,0x0C16,
	0x1F7D,0x0C96,0x0486,0x1F9F,0x0D42,0x4583,0x0E02,0x0472,
	0x0DB1,0x1613,0x0FAD,0x0D41,0x0F11,0x0E0D,0x1C42,0x143E,
	0x076E,0x04B1,0x0FAF,0x0D61,0x0531,0x0C71,0x0DFF,0x0DFE,
	0x0406,0x0C45,0x0451,0x0D15,0x05C1,0x2CC1,0x141F,0x0CE1,
	0x0FDD,0x0C22,0x0582,0x0D92,0x0571,0x0F6D,0x0C93,0x045D,
	0x0F5E,0x044D,0x0423,0x0D05,0x0425,0x0C95,0x04A5,0x0DCE,
	0x075F,0x0E1D,0x0503,0x042E,0x0D91,0x0512,0x0DDE,0x05A1,
	0x074E,0x0C32,0x0431,0x0415,0x0D21,0x05EE,0x040E,0x0DDD,
	0x0485,0x1525,0x0491,0x0C26,0x046D,0x0C05,0x05CF,0x05FD,
	0x0E92,0x073F,0x0C0D,0x043D,0x0502,0x0C1E,0x041D,0x0461,
	0x04A1,0x0511,0x0581,0x05BD,0x0C41,0x059F,0x05BF,0x040F,
	0x0C7D,0x0402,0x054E,0x057D,0x0403,0x078D,0x05AE,0x042D,
	0x0483,0x079D,0x0D7F,0x0482,0x0611,0x056E,0x0516,0x05BE,
	0x0535,0x044E,0x05AF,0x0DED,0x042F,0x0492,0x058E,0x078F,
	0x0412,0x057E,0x053E,0x0F1F,0x073D,0x0601,0x0501,0x075D,
	0x059E,0x05CD,0x053F,0x054F,0x055E,0x055D,0x0421,0x074D,
	0x051F,0x072F,0x0781,0x0411,0x0D6F,0x077E,0x0487,0x070E,
	0x070F,0x072D,0x058F,0x078E,0x079E,0x052E,0x0413,0x072E,
	0x071D,0x052F,0x055F,0x073E,0x0417,0x0453,0x060E,0x0622,
	0x0683,0x0702,0x070D,0x071E,0x076F,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_10x6 {
	10, 6,
	{ 325, 199 },
	{ 922, 381 },
	{ 0, 78 },
	{ percentile_arr_10x6_0, percentile_arr_10x6_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 8)
static const uint16_t percentile_arr_10x8_0[400] {
	0x0154,0xAB34,0xAD44,0x8308,0x7866,0x7B64,0x79A4,0x7975,
	0x686A,0x6908,0xC514,0x6174,0x6128,0x6118,0x5B54,0x5163,
	0xF856,0x50F5,0x986F,0xDD34,0x48FE,0x4972,0x48E6,0x4146,
	0x48EE,0x40F3,0x4AC1,0x38C6,0x41E2,0xBB05,0x707E,0x38D6,
	0x3927,0x6B14,0x384B,0x3948,0x3153,0x385A,0x3134,0x6B15,
	0x39F2,0x30CF,0x3143,0x91D2,0x31C3,0x60EF,0x5973,0x3076,
	0x28D3,0x3261,0x2875,0x28DE,0x290C,0x51E3,0x28A7,0x20E3,
	0x2962,0x2B06,0x2917,0x483B,0x20B6,0x2D24,0x206E,0x285F,
	0x20B7,0x2936,0x4047,0x2037,0x20DF,0x28BF,0x21B4,0x21B3,
	0x1D08,0x2027,0x404F,0x3846,0x2116,0x187F,0x1879,0x2285,
	0x1A29,0x3915,0x4873,0x1955,0x3114,0x1B44,0x2165,0x107A,
	0x1956,0x6137,0x1106,0x3145,0x1B21,0x19D3,0x12AD,0x1B41,
	0x1AD1,0x1126,0x18F2,0x282B,0x40E5,0x20D5,0x2A0A,0x284A,
	0x1286,0x1295,0x121A,0x2A0B,0x321B,0x122D,0x10FD,0x13A1,
	0x32A2,0x12E1,0x1164,0x13C1,0x124D,0x1239,0x4504,0x10C7,
	0x22F1,0x11F1,0x0AC2,0x2125,0x1225,0x0B04,0x1107,0x1069,
	0x1A19,0x13BF,0x2A96,0x08D2,0x1271,0x0952,0x2BDF,0x0B31,
	0x1251,0x2124,0x0B13,0x12BD,0x1233,0x13EE,0x2144,0x0B16,
	0x0A15,0x18E2,0x08DD,0x1097,0x0857,0x0B24,0x0AA5,0x12A3,
	0x11C2,0x11D1,0x10CE,0x0865,0x123D,0x08B3,0x0B51,0x1971,
	0x0A41,0x0A06,0x1039,0x080A,0x0B22,0x0923,0x0836,0x08C3,
	0x0A1F,0x1072,0x080B,0x0935,0x0855,0x18A6,0x0A42,0x1133,
	0x0A83,0x0A09,0x0ACD,0x0A2E,0x0887,0x083A,0x10C5,0x085E,
	0x13B1,0x087D,0x0819,0x0A9F,0x0049,0x08F1,0x0BEF,0x1161,
	0x0B42,0x09E1,0x0A05,0x0904,0x12AE,0x029E,0x0A31,0x09FF,
	0x0951,0x0859,0x001A,0x082F,0x0B81,0x08B5,0x0A35,0x082A,
	0x08ED,0x1142,0x1262,0x0B32,0x08A5,0x12D2,0x03DD,0x0B07,
	0x18AE,0x083F,0x00AF,0x0AB3,0x086D,0x0287,0x0A93,0x025D,
	0x0816,0x13FF,0x0A8D,0x005D,0x08D1,0x0392,0x0845,0x0AC3,
	0x08C2,0x01A3,0x0AB1,0x09A2,0x005B,0x0B93,0x02B2,0x1086,
	0x001B,0x0863,0x0216,0x0AA1,0x0896,0x0A8F,0x084E,0x0A8E,
	0x0A53,0x0026,0x0A26,0x0382,0x0807,0x0862,0x0029,0x0871,
	0x00BD,0x0835,0x024E,0x0806,0x0941,0x0895,0x03AF,0x0A13,
	0x0932,0x03ED,0x0BFD,0x0207,0x0B83,0x0993,0x09B1,0x03CD,
	0x0A3E,0x03FE,0x0A21,0x0015,0x0B11,0x0A43,0x00E1,0x136F,
	0x00BE,0x00A2,0x0842,0x0043,0x0825,0x082E,0x0A2A,0x03DE,
	0x0BA2,0x0122,0x0BCF,0x004D,0x0323,0x09C1,0x0292,0x083E,
	0x0252,0x0017,0x0A72,0x00CD,0x0182,0x0A63,0x0131,0x09B2,
	0x0303,0x0902,0x0053,0x035F,0x0A32,0x003D,0x0992,0x0A2F,
	0x03B2,0x0ABE,0x009F,0x0183,0x0312,0x08B1,0x0B02,0x0A17,
	0x0B7F,0x0333,0x0297,0x0A23,0x020F,0x0282,0x0851,0x0822,
	0x03CE,0x01EE,0x000E,0x08B2,0x0083,0x0A1D,0x00A3,0x0222,
	0x088F,0x0112,0x029D,0x0092,0x0A3F,0x0391,0x089E,0x0301,
	0x01FD,0x09BF,0x01CE,0x0852,0x01FE,0x0013,0x0903,0x088E,
	0x037E,0x021E,0x01EF,0x095F,0x016F,0x09DE,0x03BE,0x020E,
	0x0113,0x01DF,0x080F,0x020D,0x0833,0x03AE,0x0032,0x03BD,
	0x0823,0x001E,0x01AF,0x0203,0x034F,0x0093,0x0A81,0x036E,
	0x0291,0x038E,0x0A01,0x001F,0x017F,0x01CF,0x017E,0x0202,
	0x0BAD,0x0211,0x035D,0x035E,0x039F,0x0212,0x032E,0x033F,
	0x034D,0x034E,0x036D,0x032F,0x033E,0x037D,0x038F,0x039E
};

static const uint16_t percentile_arr_10x8_1[221] {
	0x0621,0xDFAE,0x2443,0x54C2,0x37CD,0x1CF1,0xFCA3,0x14D2,
	0x2D32,0x5551,0x7DDF,0x5C33,0x15D1,0x3462,0x24B3,0x7452,
	0x5FBE,0x6472,0x65A2,0x1D06,0x445D,0x15EF,0x0E31,0x1D71,
	0x343E,0x0D42,0x0CDD,0x1F01,0x4691,0x1435,0x0E82,0x0DFF,
	0x17DD,0x0D22,0x24B2,0x1603,0x04B5,0x24AE,0x060D,0x2D13,
	0x0C7D,0x0496,0x17BD,0x1F4F,0x1F7D,0x1486,0x0593,0x1C16,
	0x0C07,0x15FE,0x041F,0x14D1,0x0C9F,0x0E81,0x0D15,0x27AF,
	0x0C2E,0x0D23,0x176E,0x0FAD,0x1C06,0x1561,0x0DB1,0x040B,
	0x1C4E,0x0D83,0x1711,0x0C42,0x0C71,0x1C1A,0x0D25,0x04A2,
	0x0C45,0x076D,0x0F9F,0x075F,0x0E12,0x046D,0x048F,0x1D92,
	0x0602,0x0C39,0x174E,0x0C51,0x0CA1,0x075E,0x05C1,0x14BD,
	0x0D31,0x0423,0x0F3F,0x0495,0x0C93,0x049E,0x0D05,0x04E1,
	0x0DEE,0x0415,0x04B1,0x0503,0x0CCD,0x042F,0x0DCF,0x044D,
	0x0541,0x1582,0x05DE,0x0D01,0x0487,0x040A,0x0516,0x0CA5,
	0x05FD,0x05BF,0x057D,0x0DA1,0x0426,0x040F,0x071F,0x0613,
	0x0432,0x0D12,0x043D,0x0425,0x0461,0x061D,0x0D21,0x0591,
	0x079D,0x048D,0x0429,0x0C49,0x04C1,0x042A,0x040E,0x0485,
	0x0511,0x0405,0x0502,0x0441,0x0C19,0x0692,0x0535,0x058F,
	0x041D,0x059F,0x072D,0x04AD,0x049D,0x05CE,0x048E,0x0C31,
	0x057F,0x078D,0x0409,0x041E,0x05AE,0x0611,0x058E,0x05DD,
	0x05CD,0x056E,0x0483,0x073D,0x054E,0x0D9E,0x0402,0x0491,
	0x040D,0x056F,0x042D,0x0581,0x0421,0x057E,0x0781,0x053E,
	0x0482,0x078F,0x0413,0x052E,0x0601,0x0422,0x0492,0x055E,
	0x05BE,0x0F9E,0x072F,0x074D,0x0412,0x070F,0x075D,0x05BD,
	0x051F,0x071D,0x073E,0x077E,0x0403,0x0411,0x078E,0x055D,
	0x05AF,0x05ED,0x052F,0x053F,0x070D,0x070E,0x072E,0x054F,
	0x0417,0x041B,0x0453,0x055F,0x060E,0x0622,0x0683,0x068D,
	0x0702,0x071E,0x076F,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_10x8 =
{
	10, 8,
	{ 400, 221 },
	{ 1119, 376 },
	{ 0, 52 },
	{ percentile_arr_10x8_0, percentile_arr_10x8_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 10)
static const uint16_t percentile_arr_10x10_0[453] {
	0x0334,0x9514,0x8954,0x806A,0x6F14,0x6724,0x6108,0x6364,
	0x5175,0x5D44,0x5866,0x5118,0x5308,0xA179,0x5128,0xF534,
	0x49A4,0x5354,0x9174,0x486F,0x48EA,0x40F3,0x4963,0x414A,
	0xF8F9,0x3984,0x4172,0x387E,0x405A,0x38DA,0x38F5,0x9B05,
	0x30EE,0x32C1,0x3261,0x3D08,0x31E2,0x3056,0x292B,0x3146,
	0x3127,0x3315,0x58CA,0x58E6,0x290C,0x3314,0x8134,0x28E3,
	0x28FE,0x2948,0x28C6,0x78DE,0x28BB,0x68D6,0x286E,0x2173,
	0x2962,0x21D2,0x205F,0x49F2,0x2917,0x2306,0x207F,0x404F,
	0x2153,0x2943,0x20CF,0x21C3,0x2073,0x20D3,0x2136,0x183B,
	0x430A,0x40A7,0x18B6,0x2079,0x2309,0x2075,0x184B,0x20EF,
	0x187A,0x7837,0x1B19,0x20AB,0x18BA,0x20B7,0x1994,0x19E3,
	0x21B4,0x49B3,0x38BF,0x193B,0x1876,0x182B,0x30F2,0x193A,
	0x1827,0x1965,0x1914,0x184A,0x4047,0x1916,0x1285,0x1937,
	0x122D,0x1915,0x1321,0x1955,0x1046,0x191B,0x2106,0x2919,
	0x1344,0x1524,0x12E1,0x3926,0x10E5,0x2295,0x1159,0x1145,
	0x10DF,0x124D,0x1271,0x092A,0x2169,0x1704,0x22A2,0x1164,
	0x13EE,0x12F1,0x0AD1,0x128A,0x110A,0x11D3,0x1286,0x115A,
	0x2BA1,0x0BBF,0x3956,0x2A89,0x12AD,0x10E9,0x0B41,0x1A29,
	0x2225,0x08FD,0x1107,0x08D5,0x191A,0x1125,0x1A96,0x0B04,
	0x18D9,0x2B16,0x11F1,0x0A33,0x0924,0x131A,0x1149,0x1324,
	0x0BEF,0x0A99,0x08CB,0x123D,0x1331,0x0BDF,0x0872,0x22A3,
	0x0AC2,0x1144,0x0D04,0x08D2,0x08CE,0x0AA9,0x0A9A,0x0B13,
	0x1251,0x0865,0x1069,0x0897,0x1215,0x18B3,0x1A62,0x08C7,
	0x185E,0x10E2,0x0AA5,0x21FF,0x090B,0x0952,0x09E1,0x0A42,
	0x08F1,0x0A06,0x0B22,0x087D,0x1139,0x021F,0x122E,0x082F,
	0x09C2,0x0887,0x0A0A,0x03C1,0x0929,0x0A5D,0x0A83,0x0BFF,
	0x0935,0x085B,0x0104,0x08DD,0x0923,0x083F,0x0241,0x09D1,
	0x0A39,0x0863,0x0A8B,0x08A6,0x008B,0x1133,0x13B1,0x089B,
	0x0AB3,0x0036,0x0BDD,0x08ED,0x0857,0x0971,0x0219,0x1235,
	0x0AB1,0x0ACD,0x036F,0x0A31,0x08AA,0x003A,0x08C3,0x0A05,
	0x02BD,0x0B92,0x0B07,0x12B2,0x08C5,0x0B51,0x0381,0x0A8D,
	0x01A3,0x0896,0x0855,0x0BFD,0x005D,0x0BFE,0x023E,0x08AF,
	0x00B9,0x0A93,0x00B5,0x0862,0x0A0B,0x0A09,0x0A72,0x0332,
	0x0AA1,0x08C9,0x024E,0x1382,0x0951,0x00A5,0x0A2A,0x0059,
	0x0A9E,0x0B42,0x004E,0x0942,0x03ED,0x09B2,0x02D2,0x0849,
	0x0035,0x0216,0x0961,0x0BAF,0x00AE,0x0826,0x0287,0x0A1A,
	0x0393,0x0221,0x09A2,0x086D,0x0226,0x0871,0x0039,0x082A,
	0x08C2,0x08E1,0x0845,0x0207,0x0B23,0x0015,0x00D1,0x0B83,
	0x037F,0x0252,0x08A9,0x0099,0x0A13,0x0053,0x0807,0x03CD,
	0x0BDE,0x0016,0x089A,0x0232,0x035F,0x0A8E,0x0AC3,0x022F,
	0x0263,0x0829,0x004D,0x0132,0x0806,0x0311,0x01B1,0x0941,
	0x0086,0x000B,0x1122,0x0025,0x0842,0x00BD,0x0BCF,0x03A2,
	0x0043,0x0B03,0x0895,0x0A8F,0x008A,0x09EF,0x0253,0x0A1B,
	0x0182,0x0243,0x0A92,0x00CD,0x083E,0x030B,0x0223,0x081A,
	0x0A9F,0x0193,0x00BE,0x0017,0x0931,0x0391,0x037E,0x09C1,
	0x0312,0x0333,0x03B2,0x083D,0x08B1,0x00B2,0x002E,0x021D,
	0x0A9D,0x0192,0x02AE,0x0102,0x0022,0x081B,0x0222,0x009E,
	0x021E,0x000A,0x089F,0x0217,0x0BCE,0x0052,0x020F,0x0A97,
	0x0282,0x008E,0x0A3F,0x01FD,0x00A3,0x0019,0x08A2,0x0301,
	0x036E,0x01FE,0x03BE,0x0ABE,0x01CE,0x0302,0x029B,0x0051,
	0x0883,0x008F,0x0BAE,0x01DF,0x0183,0x0912,0x000E,0x020D,
	0x01EE,0x0B4F,0x0033,0x0103,0x020E,0x0832,0x01AF,0x0913,
	0x01DE,0x0203,0x001E,0x0092,0x0093,0x000F,0x015F,0x0291,
	0x0281,0x0813,0x001F,0x01CF,0x033F,0x0023,0x01BF,0x0202,
	0x016F,0x017E,0x03AD,0x0201,0x034E,0x0BBD,0x036D,0x017F,
	0x0211,0x038E,0x0212,0x032E,0x034D,0x035E,0x037D,0x039E,
	0x032F,0x033E,0x035D,0x038F,0x039F
};

static const uint16_t percentile_arr_10x10_1[234] {
	0x07CD,0x6E21,0x24F1,0x8443,0xD7AE,0x24C2,0x1C62,0xCCA3,
	0x1C33,0xFDEF,0x2532,0x55DF,0x1472,0x6C3E,0x14D2,0x34DD,
	0x1452,0x745D,0x4D51,0x8DD1,0x247D,0x75FF,0x0CB3,0x17BE,
	0x6CAE,0x17DD,0x1571,0x3D06,0x4E31,0x0DA2,0x67BD,0x160D,
	0x2C4E,0x0D22,0x176E,0x3CB2,0x142E,0x4DFE,0x0F4F,0x1435,
	0x0F01,0x0D42,0x0F7D,0x0CB5,0x1E03,0x149F,0x1C96,0x141F,
	0x14B9,0x0FAF,0x0439,0x0E91,0x2682,0x1D13,0x1FAD,0x0407,
	0x3471,0x0C86,0x0F6D,0x0D15,0x0D61,0x040B,0x0C6D,0x0C16,
	0x0C9A,0x0D0A,0x0593,0x0CD1,0x248F,0x0C2F,0x3C42,0x1523,
	0x0445,0x0E81,0x0CA2,0x1525,0x0406,0x1C8A,0x0C1A,0x04BD,
	0x0F5E,0x0F3F,0x1F4E,0x0E1D,0x0423,0x0DCF,0x044D,0x0D92,
	0x0583,0x0DB1,0x1449,0x15EE,0x0F5F,0x079F,0x0D19,0x0409,
	0x04CD,0x05FD,0x143D,0x0612,0x0D03,0x0D82,0x04B1,0x0C95,
	0x0C2A,0x049E,0x05AF,0x0D31,0x05BE,0x04E1,0x0D05,0x0516,
	0x0711,0x05C1,0x0509,0x0D41,0x0493,0x048E,0x0602,0x05BF,
	0x0CA5,0x0529,0x0535,0x0D12,0x0539,0x0451,0x0C29,0x071F,
	0x040A,0x0F3D,0x0432,0x059F,0x0425,0x0C99,0x05DE,0x05CE,
	0x0C0F,0x0489,0x051A,0x0501,0x0415,0x057F,0x0431,0x0E13,
	0x040D,0x041D,0x075D,0x0C53,0x0502,0x04C1,0x049D,0x0426,
	0x040E,0x05A1,0x055F,0x0781,0x0591,0x04A9,0x048B,0x0D8E,
	0x052E,0x0412,0x0521,0x0405,0x04AD,0x074D,0x0611,0x077E,
	0x078F,0x078D,0x048D,0x041E,0x0487,0x0461,0x0C85,0x05ED,
	0x0402,0x0483,0x0419,0x0511,0x0491,0x0482,0x059E,0x068D,
	0x055D,0x072E,0x05DD,0x054E,0x0441,0x0422,0x052F,0x057D,
	0x072D,0x079D,0x0CA1,0x072F,0x079E,0x0581,0x042D,0x055E,
	0x0601,0x0413,0x0692,0x0403,0x051F,0x053F,0x054F,0x05CD,
	0x070F,0x071D,0x05AE,0x05BD,0x0492,0x056E,0x0411,0x0417,
	0x041B,0x0421,0x053E,0x056F,0x057E,0x058F,0x060E,0x0622,
	0x0683,0x0702,0x070D,0x070E,0x071E,0x073E,0x076F,0x078E,
	0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_10x10 {
	10, 10,
	{ 453, 234 },
	{ 1095, 472 },
	{ 0, 70 },
	{ percentile_arr_10x10_0, percentile_arr_10x10_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (12 * 10)
static const uint16_t percentile_arr_12x10_0[491] {
	0x0334,0x9954,0x8514,0x7128,0x6364,0xC174,0x5D34,0x5866,
	0x5975,0x5354,0xAF14,0x506A,0x5108,0x5724,0x5308,0x4544,
	0x4918,0x4064,0x49E2,0x4179,0x8163,0x4054,0xF81C,0x394A,
	0x38F3,0x4172,0x38F5,0xA06F,0x68EA,0x69F2,0x3134,0x31A4,
	0x305A,0x68DA,0x3056,0x3146,0x31F5,0x3148,0x5A61,0x32C1,
	0x31D2,0x307E,0x29E3,0x30E6,0x59C3,0x2984,0x29B6,0x28F9,
	0x5204,0x28EE,0x50CA,0x2997,0x48C6,0x4838,0x2953,0x200C,
	0x2943,0x2173,0x2D08,0x4162,0x29B4,0x2314,0x21B3,0x212B,
	0x210C,0x48E3,0x60DE,0x205F,0x20FE,0x2028,0x21A6,0x404F,
	0x20D6,0x2214,0x2127,0x1873,0x40CF,0x206E,0x1B09,0x21C6,
	0x2075,0x19D5,0x2305,0x18D3,0x2076,0x1804,0x230A,0x304B,
	0x20BB,0x18B6,0x1936,0x1B19,0x3037,0x187F,0x18A7,0x1B85,
	0x30BA,0x183B,0x1027,0x18EF,0x1B21,0x1879,0x10AB,0x1917,
	0x1114,0x18BF,0x1074,0x1994,0x2847,0x111B,0x28F2,0x11E5,
	0x19A7,0x113A,0x1046,0x28B7,0x207A,0x182B,0x1155,0x104A,
	0x1344,0x293B,0x11D3,0x2014,0x1044,0x1018,0x13A1,0x1315,
	0x2524,0x20DF,0x10E5,0x1126,0x12A2,0x1824,0x2271,0x11F1,
	0x2964,0x12D1,0x115A,0x092A,0x2341,0x1A2D,0x12E1,0x090A,
	0x13BF,0x0A4D,0x2119,0x0BC1,0x1233,0x1A8A,0x2008,0x1159,
	0x1A89,0x08D5,0x1156,0x0834,0x13EE,0x1169,0x1187,0x1AA3,
	0x1229,0x1331,0x0A85,0x0937,0x1704,0x08FD,0x2124,0x0B13,
	0x1251,0x0AAD,0x082C,0x091A,0x18D9,0x0A99,0x1848,0x18E9,
	0x0B95,0x1144,0x0AF1,0x1A25,0x131A,0x09C5,0x0986,0x1BDF,
	0x0B24,0x0965,0x1262,0x0949,0x0872,0x09C2,0x12C2,0x0916,
	0x085E,0x0B06,0x08CB,0x08C7,0x1242,0x1BEF,0x0A9A,0x1152,
	0x08B3,0x0AA9,0x090B,0x08D2,0x1B22,0x0B04,0x0865,0x0A15,
	0x1286,0x0A83,0x0A95,0x09D1,0x0A06,0x0196,0x1139,0x0A3D,
	0x0933,0x13B1,0x0123,0x0D04,0x08E2,0x122E,0x08A6,0x00CE,
	0x0A31,0x1241,0x0B51,0x1057,0x1171,0x007D,0x1145,0x0A0A,
	0x0129,0x09FF,0x089B,0x085B,0x0063,0x0AB1,0x0A1F,0x0A5D,
	0x0AA5,0x0036,0x0904,0x0B86,0x0A8B,0x0897,0x11E1,0x0332,
	0x083F,0x0A19,0x02B3,0x0859,0x08C3,0x0855,0x11B5,0x01A5,
	0x0AB2,0x0392,0x10DD,0x09A3,0x00ED,0x0907,0x1161,0x002F,
	0x0887,0x0216,0x0ABD,0x0B81,0x0A93,0x0A21,0x003A,0x0ACD,
	0x0AA1,0x0A35,0x0272,0x0BDD,0x03FE,0x0BAF,0x0869,0x0213,
	0x088B,0x020B,0x00B5,0x1035,0x08F1,0x0151,0x0A4E,0x0239,
	0x0BA2,0x00AA,0x0896,0x0382,0x0A08,0x0A05,0x0A09,0x0142,
	0x086D,0x004E,0x0B23,0x0106,0x0807,0x036F,0x0995,0x03FD,
	0x08AF,0x08C5,0x0062,0x0053,0x0B42,0x0826,0x021A,0x01A2,
	0x09B1,0x00C9,0x09B2,0x0045,0x0207,0x08B9,0x00A5,0x0AD2,
	0x0095,0x003E,0x0A32,0x0383,0x0849,0x0135,0x029E,0x0A26,
	0x023E,0x0BFF,0x0A52,0x0311,0x001B,0x0915,0x0A8D,0x0223,
	0x022A,0x0BED,0x0086,0x0A96,0x0222,0x035F,0x0A43,0x085D,
	0x0303,0x0393,0x0A63,0x082A,0x037F,0x0932,0x0043,0x0292,
	0x03CD,0x0BDE,0x009F,0x0125,0x08A9,0x0253,0x0015,0x0192,
	0x0A17,0x08C2,0x0316,0x00D1,0x0282,0x0871,0x0312,0x0122,
	0x0A9F,0x02AE,0x0006,0x0A8E,0x08E1,0x0016,0x0B0B,0x00AE,
	0x0025,0x0193,0x0AC3,0x0017,0x0307,0x00BD,0x08BE,0x0039,
	0x0BB2,0x021B,0x01FD,0x084D,0x03CE,0x00A3,0x0302,0x0BCF,
	0x0033,0x0391,0x028F,0x0852,0x0287,0x008A,0x0333,0x080B,
	0x0131,0x01C1,0x037E,0x0A0F,0x00B1,0x002E,0x0099,0x0902,
	0x009A,0x003D,0x0982,0x0301,0x00CD,0x0941,0x0042,0x0183,
	0x029D,0x08A2,0x021D,0x001A,0x0A97,0x01EF,0x01CE,0x0051,
	0x0BAE,0x022F,0x03BE,0x021E,0x000A,0x09DF,0x0029,0x020D,
	0x02BE,0x029B,0x09EE,0x00B2,0x0912,0x036E,0x009E,0x0022,
	0x0019,0x0892,0x0032,0x01FE,0x0083,0x023F,0x0B96,0x000E,
	0x008F,0x0113,0x0103,0x001E,0x0A0E,0x0013,0x008E,0x0281,
	0x09AF,0x017E,0x0203,0x016F,0x0291,0x0023,0x0093,0x03BD,
	0x001F,0x01CF,0x01DE,0x0201,0x01BF,0x0B4F,0x000F,0x0202,
	0x037D,0x038E,0x0211,0x0212,0x034E,0x039F,0x03AD,0x015F,
	0x017F,0x032E,0x033F,0x034D,0x035E,0x036D,0x032F,0x033E,
	0x035D,0x038F,0x039E
};

static const uint16_t percentile_arr_12x10_1[240] {
	0x0621,0xA443,0xFCC2,0x3CA3,0x1D32,0x14F1,0x7462,0x1433,
	0x27CD,0x2571,0x57AE,0x5DD1,0x64B3,0x44D2,0x2C72,0x25A2,
	0x1E31,0x55DF,0x4C52,0x1DEF,0x0D51,0x3C5D,0x3C3E,0x74DD,
	0x347D,0x27BE,0x5CB5,0x17DD,0x2C14,0x0CAE,0x24B2,0x15FF,
	0x2701,0x0D42,0x1FBD,0x0C35,0x1603,0x060D,0x1D93,0x0C96,
	0x1C07,0x1522,0x0D06,0x0F4F,0x0C9F,0x1F6E,0x0D86,0x0C2E,
	0x1DFE,0x0682,0x1E91,0x0F7D,0x0C86,0x040B,0x1513,0x044E,
	0x14D1,0x0C39,0x14B9,0x1C71,0x05B1,0x0C1F,0x0681,0x1445,
	0x0C16,0x0D95,0x1583,0x0D61,0x0FAD,0x1442,0x048F,0x0D0A,
	0x049A,0x0F6D,0x146D,0x0C2F,0x0D25,0x0406,0x0C1A,0x0D23,
	0x0612,0x0FAF,0x0F11,0x0592,0x0515,0x14E1,0x0602,0x048A,
	0x0E1D,0x0CBD,0x0F9F,0x0423,0x075E,0x174E,0x0426,0x0404,
	0x0C22,0x0CA2,0x0DEE,0x0CA5,0x0F3F,0x05C1,0x0CCD,0x0503,
	0x044D,0x0D16,0x0449,0x0D82,0x0613,0x0585,0x0519,0x0C95,
	0x075F,0x0D35,0x04B1,0x0509,0x0531,0x0DA1,0x049E,0x040A,
	0x05CF,0x0D41,0x0415,0x0692,0x05FD,0x0C25,0x04A1,0x0529,
	0x0591,0x0C93,0x057F,0x04C1,0x0512,0x051A,0x078D,0x0451,
	0x0C0F,0x0487,0x0611,0x0432,0x042A,0x05AF,0x0461,0x072D,
	0x0409,0x0405,0x0D39,0x05DE,0x048E,0x0499,0x0483,0x04A9,
	0x0491,0x042D,0x049D,0x0429,0x040E,0x05AE,0x0521,0x043D,
	0x0581,0x05DD,0x0492,0x0CAD,0x041E,0x058F,0x071F,0x072F,
	0x0419,0x073D,0x057D,0x0511,0x05CE,0x041D,0x0485,0x056E,
	0x0412,0x0431,0x05BF,0x0441,0x054E,0x0489,0x0421,0x0502,
	0x0408,0x040D,0x051F,0x059F,0x073E,0x078F,0x0482,0x079D,
	0x0C02,0x05BE,0x048B,0x0411,0x0505,0x057E,0x052E,0x074D,
	0x077E,0x054F,0x0601,0x055F,0x068D,0x070D,0x070F,0x071E,
	0x072E,0x05CD,0x0403,0x0501,0x055D,0x059E,0x0781,0x0413,
	0x0417,0x041B,0x0453,0x048D,0x052F,0x053E,0x053F,0x055E,
	0x056F,0x058E,0x05BD,0x05ED,0x060E,0x0622,0x0683,0x0702,
	0x070E,0x071D,0x075D,0x076F,0x078E,0x079E,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_12x10 =
{
	12, 10,
	{ 491, 240 },
	{ 1099, 341 },
	{ 0, 23 },
	{ percentile_arr_12x10_0, percentile_arr_12x10_1 }
};
#endif

#if ASTCENC_BLOCK_MAX_TEXELS >= (12 * 12)
static const uint16_t percentile_arr_12x12_0[529] {
	0x0334,0xF534,0x8514,0x8954,0x7F14,0xFB54,0x7B08,0x7128,
	0x7974,0x6179,0x6B64,0x6908,0x606A,0x6724,0xB544,0xB066,
	0xA14A,0x5118,0x9975,0x51F9,0x981C,0x49CA,0x4854,0x886F,
	0x88D4,0x48EE,0x41E2,0x4163,0x40F3,0x4261,0x4064,0x407E,
	0x385A,0x42C1,0x4172,0x38EA,0x3946,0x78CF,0xA056,0x38DE,
	0x3D08,0x38F9,0x3B14,0x38FE,0xA134,0x38B8,0x31A4,0x71D2,
	0x60DA,0x39C3,0x99BA,0x60CA,0x39F2,0x30F5,0x304F,0x31B6,
	0x31F5,0x3204,0x3148,0x305F,0x2953,0x3194,0x3184,0x310C,
	0x889C,0x300C,0x2943,0x30EF,0x28C6,0x2997,0x2838,0x58E6,
	0x20E4,0x28E3,0x2873,0x29E3,0x2A84,0x28D3,0x492B,0x2962,
	0x286E,0x20BF,0x21AA,0x29A6,0x6A14,0x2828,0x89C6,0x21B3,
	0x2305,0x29B4,0x2173,0x2127,0x20D6,0x407F,0x2294,0x21D9,
	0x21D5,0x2004,0x404B,0x18DF,0x2079,0x219B,0x18A8,0x2385,
	0x1936,0x21AB,0x188C,0x1B09,0x18BA,0x203B,0x187A,0x1875,
	0x2344,0x18BB,0x18B6,0x193A,0x1837,0x1914,0x1846,0x1876,
	0x1884,0x1D24,0x182B,0x284A,0x18A7,0x18AB,0x1917,0x322D,
	0x1047,0x1874,0x1818,0x18F2,0x1164,0x1B89,0x2959,0x1B21,
	0x39E5,0x1827,0x10F4,0x18B7,0x11D3,0x1A4D,0x1315,0x12AD,
	0x1AD1,0x3A71,0x1319,0x11A7,0x2044,0x2F04,0x2341,0x10E5,
	0x1155,0x195A,0x1024,0x111B,0x1251,0x1233,0x12E1,0x13A1,
	0x13BF,0x212A,0x22A2,0x113B,0x23DF,0x10D5,0x2399,0x0814,
	0x1126,0x13EE,0x1285,0x10C4,0x18FD,0x20D9,0x0987,0x1242,
	0x29C5,0x2313,0x0898,0x13C1,0x08C8,0x11F1,0x1034,0x1B24,
	0x0B0A,0x11E9,0x0808,0x125D,0x18E9,0x0848,0x1395,0x0965,
	0x123D,0x2186,0x1295,0x18CE,0x098B,0x0BEF,0x1504,0x082C,
	0x0A41,0x1144,0x0A89,0x0956,0x1331,0x085E,0x0B04,0x128A,
	0x12A3,0x1937,0x19C2,0x0952,0x0872,0x08B4,0x1262,0x1124,
	0x1969,0x1063,0x0AF1,0x1225,0x0894,0x11C9,0x18D2,0x0ACD,
	0x0A29,0x0B06,0x09B5,0x18C7,0x0916,0x1088,0x09FF,0x2206,
	0x0A15,0x08B3,0x0B51,0x0A1F,0x18CB,0x0AC2,0x0A2E,0x1865,
	0x08AC,0x0A31,0x08A4,0x138A,0x0A99,0x09D1,0x0A86,0x189B,
	0x0283,0x0BDD,0x0ABD,0x1933,0x083F,0x1386,0x0923,0x0322,
	0x0869,0x10DD,0x13B1,0x082F,0x087D,0x11B9,0x085B,0x08ED,
	0x00C3,0x08E2,0x084E,0x0887,0x0855,0x0A0A,0x0857,0x0B92,
	0x1036,0x12A5,0x0293,0x0945,0x08A6,0x0196,0x19A3,0x036F,
	0x0904,0x1205,0x09E1,0x0381,0x0971,0x1219,0x0BAF,0x0949,
	0x00AF,0x0AA9,0x018A,0x0907,0x0BFD,0x003A,0x0BCD,0x0AB2,
	0x088B,0x0252,0x0A4E,0x03FF,0x0845,0x0897,0x0059,0x090B,
	0x0B42,0x0807,0x0A16,0x0853,0x0A8D,0x01B2,0x0AB1,0x091A,
	0x0195,0x0A35,0x00B5,0x10AA,0x0115,0x0A21,0x0096,0x0A08,
	0x03FE,0x0B7F,0x08B9,0x12B3,0x023E,0x0A23,0x029E,0x08F1,
	0x01A9,0x0BDE,0x0843,0x02D2,0x0A1A,0x08C5,0x0151,0x0A43,
	0x0332,0x0383,0x0826,0x0BED,0x10C2,0x00AE,0x0B82,0x0213,
	0x0232,0x085D,0x02A1,0x101B,0x035F,0x0303,0x0A39,0x0207,
	0x0A53,0x0142,0x01A5,0x082A,0x0099,0x0A17,0x03CF,0x0906,
	0x0125,0x0A96,0x0A9A,0x0209,0x0393,0x0961,0x0131,0x0A88,
	0x0139,0x099A,0x0292,0x0272,0x0862,0x08BE,0x0141,0x02C3,
	0x0886,0x0039,0x08A9,0x01A2,0x01B1,0x0851,0x020B,0x086D,
	0x0312,0x08CD,0x020F,0x0311,0x0BCE,0x0135,0x0006,0x0849,
	0x0132,0x0A8F,0x022F,0x022A,0x0AAE,0x0A8E,0x0263,0x03A2,
	0x083E,0x009A,0x021B,0x0835,0x0323,0x0871,0x0993,0x0226,
	0x0302,0x0922,0x0119,0x0222,0x021D,0x0B07,0x08C9,0x037E,
	0x08BD,0x0042,0x00D1,0x0B33,0x01C1,0x0B9A,0x0282,0x088A,
	0x0182,0x083D,0x004D,0x010A,0x0A1E,0x0019,0x00B2,0x0999,
	0x00A5,0x0095,0x0817,0x0022,0x031A,0x0902,0x00A3,0x01BF,
	0x029F,0x0816,0x03B2,0x0015,0x0391,0x0BBE,0x01FE,0x1129,
	0x002E,0x01DF,0x0301,0x0033,0x0B6E,0x00E1,0x0297,0x00B1,
	0x009F,0x0B16,0x000A,0x001A,0x0052,0x080B,0x030B,0x029D,
	0x0BAE,0x01FD,0x020E,0x00A2,0x0A3F,0x0192,0x0ABE,0x020D,
	0x008F,0x028B,0x0083,0x0025,0x09EE,0x01EF,0x0029,0x0291,
	0x0B4F,0x0396,0x0287,0x008E,0x0092,0x0B4E,0x017E,0x001E,
	0x009E,0x0103,0x080F,0x000E,0x0113,0x0203,0x01CF,0x0183,
	0x01CE,0x001F,0x0112,0x01DE,0x038E,0x0832,0x033E,0x0212,
	0x029B,0x0023,0x016F,0x0201,0x09AF,0x0202,0x0281,0x035E,
	0x034D,0x037D,0x03AD,0x0013,0x0093,0x015F,0x0211,0x033F,
	0x036D,0x039F,0x03BD,0x017F,0x032E,0x032F,0x035D,0x038F,
	0x039E
};

static const uint16_t percentile_arr_12x12_1[246] {
	0x0443,0xFFCD,0x2C62,0x2E21,0x3CF1,0x34C2,0x4CDD,0x2452,
	0xD5DF,0x1DD1,0x0FAE,0x64A3,0x0C7D,0x3433,0x1CD2,0x2DEF,
	0x0C3E,0x1D71,0xA472,0x0D32,0x54B3,0x4D51,0x445D,0x0E31,
	0x1FDD,0x0DFF,0x0CAE,0x45A2,0x2FBE,0xA4B9,0x1C4E,0x2C9F,
	0x160D,0x0D42,0x342E,0x074F,0x1414,0x0F6E,0x0CB2,0x34B5,
	0x0DFE,0x0D86,0x1496,0x1D22,0x0691,0x140B,0x041F,0x0C35,
	0x1D93,0x1506,0x1439,0x0C9A,0x0F01,0x2442,0x0C8F,0x04D1,
	0x1486,0x0C6D,0x0513,0x0C71,0x0E82,0x177D,0x0E03,0x07BD,
	0x0C2F,0x0D83,0x07AF,0x0D61,0x1407,0x0DB1,0x050A,0x0C94,
	0x07AD,0x0D8A,0x0C04,0x0416,0x0C49,0x0445,0x15C1,0x0C1A,
	0x0525,0x0595,0x0C8A,0x075E,0x0CBD,0x0681,0x0F4E,0x075F,
	0x061D,0x1541,0x0CB1,0x0F3F,0x0406,0x076D,0x0DCF,0x05EE,
	0x0D23,0x0599,0x0CCD,0x0711,0x0C23,0x079F,0x0D15,0x0585,
	0x04A2,0x042A,0x0D31,0x05BF,0x0D92,0x0C26,0x043D,0x0C93,
	0x0502,0x0C15,0x048B,0x0D03,0x0613,0x0516,0x0495,0x0C29,
	0x04A5,0x040F,0x0425,0x0539,0x0D19,0x04E1,0x05BE,0x0422,
	0x0432,0x0C0A,0x0431,0x041E,0x0492,0x04A9,0x0582,0x0529,
	0x0487,0x0C4D,0x0512,0x049E,0x0505,0x0451,0x0D7F,0x0489,
	0x0602,0x05DE,0x0591,0x0535,0x074D,0x055E,0x04C1,0x0612,
	0x05DD,0x05FD,0x0C61,0x0521,0x0484,0x05CE,0x0581,0x0491,
	0x051A,0x04A1,0x048E,0x040D,0x0499,0x071F,0x072E,0x075D,
	0x0441,0x0589,0x057E,0x0CAD,0x0501,0x054F,0x0692,0x0511,
	0x049D,0x0509,0x056E,0x040E,0x0409,0x0601,0x048D,0x0413,
	0x053E,0x0419,0x072D,0x0408,0x0485,0x042D,0x041D,0x05A1,
	0x0781,0x0402,0x05ED,0x0C82,0x0403,0x057D,0x05CD,0x0611,
	0x0488,0x0411,0x054E,0x051F,0x053F,0x056F,0x059F,0x070F,
	0x071D,0x073D,0x073E,0x077E,0x078F,0x0405,0x079D,0x079E,
	0x058E,0x0412,0x055D,0x05AE,0x041B,0x0421,0x0453,0x0417,
	0x0483,0x052E,0x052F,0x055F,0x058F,0x059E,0x05AF,0x05BD,
	0x060E,0x0622,0x0683,0x068D,0x0702,0x070D,0x070E,0x071E,
	0x072F,0x076F,0x078D,0x078E,0x07BF,0x07CE
};

static const packed_percentile_table block_pcd_12x12 {
	12, 12,
	{ 529, 246 },
	{ 1435, 335 },
	{ 0, 22 },
	{ percentile_arr_12x12_0, percentile_arr_12x12_1 }
};
#endif

/**
 * @brief Fetch the packed percentile table for the given 2D block size.
 *
 * @param xdim The block x size.
 * @param ydim The block y size.
 *
 * @return The packed table.
 */
static const packed_percentile_table *get_packed_table(
	int xdim,
	int ydim
) {
	int idx = (ydim << 8) | xdim;
	switch (idx)
	{
#if ASTCENC_BLOCK_MAX_TEXELS >= (4 * 4)
		case 0x0404: return &block_pcd_4x4;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (5 * 4)
		case 0x0405: return &block_pcd_5x4;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (5 * 5)
		case 0x0505: return &block_pcd_5x5;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (6 * 5)
		case 0x0506: return &block_pcd_6x5;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (6 * 6)
		case 0x0606: return &block_pcd_6x6;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (8 * 5)
		case 0x0508: return &block_pcd_8x5;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (8 * 6)
		case 0x0608: return &block_pcd_8x6;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (8 * 8)
		case 0x0808: return &block_pcd_8x8;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 5)
		case 0x050A: return &block_pcd_10x5;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 6)
		case 0x060A: return &block_pcd_10x6;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 8)
		case 0x080A: return &block_pcd_10x8;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (10 * 10)
		case 0x0A0A: return &block_pcd_10x10;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (12 * 10)
		case 0x0A0C: return &block_pcd_12x10;
#endif
#if ASTCENC_BLOCK_MAX_TEXELS >= (12 * 12)
		case 0x0C0C: return &block_pcd_12x12;
#endif
	}

	// Should never hit this with a valid 2D block size
	return nullptr;
}

/* See header for documentation. */
const float *get_2d_percentile_table(
	unsigned int xdim,
	unsigned int ydim
) {
	float* unpacked_table = new float[2048];
	const packed_percentile_table *apt = get_packed_table(xdim, ydim);

	// Set the default percentile
	for (unsigned int i = 0; i < 2048; i++)
	{
		unpacked_table[i] = 1.0f;
	}

	// Populate the unpacked percentile values
	for (int i = 0; i < 2; i++)
	{
		unsigned int itemcount = apt->itemcounts[i];
		unsigned int difscale = apt->difscales[i];
		unsigned int accum = apt->initial_percs[i];
		const uint16_t *item_ptr = apt->items[i];

		for (unsigned int j = 0; j < itemcount; j++)
		{
			uint16_t item = item_ptr[j];
			unsigned int idx = item & 0x7FF;
			unsigned int weight = (item >> 11) & 0x1F;
			accum += weight;
			unpacked_table[idx] = static_cast<float>(accum) / static_cast<float>(difscale);
		}
	}

	return unpacked_table;
}
#endif

/* See header for documentation. */
bool is_legal_2d_block_size(
	unsigned int xdim,
	unsigned int ydim
) {
	unsigned int idx = (xdim << 8) | ydim;
	switch (idx)
	{
		case 0x0404:
		case 0x0504:
		case 0x0505:
		case 0x0605:
		case 0x0606:
		case 0x0805:
		case 0x0806:
		case 0x0808:
		case 0x0A05:
		case 0x0A06:
		case 0x0A08:
		case 0x0A0A:
		case 0x0C0A:
		case 0x0C0C:
			return true;
	}

	return false;
}

/* See header for documentation. */
bool is_legal_3d_block_size(
	unsigned int xdim,
	unsigned int ydim,
	unsigned int zdim
) {
	unsigned int idx = (xdim << 16) | (ydim << 8) | zdim;
	switch (idx)
	{
		case 0x030303:
		case 0x040303:
		case 0x040403:
		case 0x040404:
		case 0x050404:
		case 0x050504:
		case 0x050505:
		case 0x060505:
		case 0x060605:
		case 0x060606:
			return true;
	}

	return false;
}
