use std::{net::SocketAddr, sync::Arc, time::Duration};

use anyhow::anyhow;
use bytes::Bytes;
use tokio::{
    net::UdpSocket,
    sync::{mpsc, oneshot},
};
use tracing::debug;

use crate::{platform::UdpSocketExt, tunnel::TunnelEvent};

// Both packets are IKE SA requests that do some magic of unblocking NATT port for some users.
const NMAP_KNOCK: &[&[u8]] = &[
    &[
        0x0, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x10, 0x2, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xc0, 0x0, 0x0, 0x0, 0xa4, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
        0x0, 0x98, 0x1, 0x1, 0x0, 0x4, 0x3, 0x0, 0x0, 0x24, 0x1, 0x1, 0x0, 0x0, 0x80, 0x1, 0x0, 0x5, 0x80, 0x2, 0x0,
        0x2, 0x80, 0x3, 0x0, 0x1, 0x80, 0x4, 0x0, 0x2, 0x80, 0xb, 0x0, 0x1, 0x0, 0xc, 0x0, 0x4, 0x0, 0x0, 0x0, 0x1,
        0x3, 0x0, 0x0, 0x24, 0x2, 0x1, 0x0, 0x0, 0x80, 0x1, 0x0, 0x5, 0x80, 0x2, 0x0, 0x1, 0x80, 0x3, 0x0, 0x1, 0x80,
        0x4, 0x0, 0x2, 0x80, 0xb, 0x0, 0x1, 0x0, 0xc, 0x0, 0x4, 0x0, 0x0, 0x0, 0x1, 0x3, 0x0, 0x0, 0x24, 0x3, 0x1, 0x0,
        0x0, 0x80, 0x1, 0x0, 0x1, 0x80, 0x2, 0x0, 0x2, 0x80, 0x3, 0x0, 0x1, 0x80, 0x4, 0x0, 0x2, 0x80, 0xb, 0x0, 0x1,
        0x0, 0xc, 0x0, 0x4, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x24, 0x4, 0x1, 0x0, 0x0, 0x80, 0x1, 0x0, 0x1, 0x80,
        0x2, 0x0, 0x1, 0x80, 0x3, 0x0, 0x1, 0x80, 0x4, 0x0, 0x2, 0x80, 0xb, 0x0, 0x1, 0x0, 0xc, 0x0, 0x4, 0x0, 0x0,
        0x0, 0x1,
    ],
    &[
        0x31, 0x27, 0xfc, 0xb0, 0x38, 0x10, 0x9e, 0x89, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x1, 0x10, 0x2, 0x0,
        0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xcc, 0xd, 0x0, 0x0, 0x5c, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
        0x0, 0x50, 0x1, 0x1, 0x0, 0x2, 0x3, 0x0, 0x0, 0x24, 0x1, 0x1, 0x0, 0x0, 0x80, 0x1, 0x0, 0x5, 0x80, 0x2, 0x0,
        0x2, 0x80, 0x4, 0x0, 0x2, 0x80, 0x3, 0x0, 0x3, 0x80, 0xb, 0x0, 0x1, 0x0, 0xc, 0x0, 0x4, 0x0, 0x0, 0xe, 0x10,
        0x0, 0x0, 0x0, 0x24, 0x2, 0x1, 0x0, 0x0, 0x80, 0x1, 0x0, 0x5, 0x80, 0x2, 0x0, 0x1, 0x80, 0x4, 0x0, 0x2, 0x80,
        0x3, 0x0, 0x3, 0x80, 0xb, 0x0, 0x1, 0x0, 0xc, 0x0, 0x4, 0x0, 0x0, 0xe, 0x10, 0xd, 0x0, 0x0, 0x18, 0x1e, 0x2b,
        0x51, 0x69, 0x5, 0x99, 0x1c, 0x7d, 0x7c, 0x96, 0xfc, 0xbf, 0xb5, 0x87, 0xe4, 0x61, 0x0, 0x0, 0x0, 0x4, 0xd,
        0x0, 0x0, 0x14, 0x40, 0x48, 0xb7, 0xd5, 0x6e, 0xbc, 0xe8, 0x85, 0x25, 0xe7, 0xde, 0x7f, 0x0, 0xd6, 0xc2, 0xd3,
        0xd, 0x0, 0x0, 0x14, 0x90, 0xcb, 0x80, 0x91, 0x3e, 0xbb, 0x69, 0x6e, 0x8, 0x63, 0x81, 0xb5, 0xec, 0x42, 0x7b,
        0x1f, 0x0, 0x0, 0x0, 0x14, 0x26, 0x24, 0x4d, 0x38, 0xed, 0xdb, 0x61, 0xb3, 0x17, 0x2a, 0x36, 0xe3, 0xd0, 0xcf,
        0xb8, 0x19,
    ],
];

pub struct NattProber {
    address: SocketAddr,
    port_knock: bool,
}

impl NattProber {
    pub fn new(address: SocketAddr, port_knock: bool) -> Self {
        Self { address, port_knock }
    }

    pub async fn probe(&self) -> anyhow::Result<()> {
        if self.send_probe().await.is_err() {
            if self.port_knock {
                // attempt to unblock NATT port by sending some magic packets to port 500
                self.send_nmap_knock().await?;
                self.send_nmap_knock().await?;
            }

            self.send_probe()
                .await
                .map_err(|_| anyhow!(i18n::tr!("error-probing-failed")))
        } else {
            Ok(())
        }
    }

    async fn send_probe(&self) -> anyhow::Result<()> {
        debug!("Sending NAT-T probe to {}", self.address);

        let udp = UdpSocket::bind("0.0.0.0:0").await?;

        let data = vec![0u8; 32];

        let result = udp.send_receive(&data, Duration::from_secs(2), self.address).await;

        match result {
            Ok(reply) if reply.len() == 32 => {
                let source_port: [u8; 4] = reply[8..12].try_into()?;
                let dest_port: [u8; 4] = reply[12..16].try_into()?;
                debug!(
                    "Received NAT-T reply from {}: source port: {}, dest port: {}, hash: {}",
                    self.address,
                    u32::from_be_bytes(source_port),
                    u32::from_be_bytes(dest_port),
                    hex::encode(&reply[reply.len() - 16..reply.len()])
                );
                Ok(())
            }
            _ => Err(anyhow!(i18n::tr!("error-no-natt-reply"))),
        }
    }

    async fn send_nmap_knock(&self) -> anyhow::Result<()> {
        debug!("Sending magic knock IKE SAs to {}", self.address);

        let udp = UdpSocket::bind("0.0.0.0:0").await?;
        udp.connect(format!("{}:500", self.address.ip())).await?;

        for probe in NMAP_KNOCK {
            let _ = udp.send(probe).await;
        }
        tokio::time::sleep(Duration::from_millis(100)).await;

        Ok(())
    }
}

// start a fake UDP listener with the UDP_ENCAP option.
// this is necessary to perform automatic decapsulation of incoming ESP packets
pub async fn start_natt_listener(
    socket: Arc<UdpSocket>,
    sender: mpsc::Sender<TunnelEvent>,
) -> anyhow::Result<oneshot::Sender<()>> {
    let (tx, mut rx) = oneshot::channel();

    debug!("Listening for NAT-T packets on port {}", socket.local_addr()?);

    tokio::spawn(async move {
        let mut buf = [0u8; 1024];

        loop {
            tokio::select! {
                result = socket.recv_from(&mut buf) => {
                    if let Ok((size, _)) = result {
                        let data = Bytes::copy_from_slice(&buf[0..size]);
                        let _ = sender.send(TunnelEvent::RemoteControlData(data)).await;
                    }
                }
                _ = &mut rx => {
                    break;
                }
            }
        }
        debug!("NAT-T listener stopped");
    });

    Ok(tx)
}
