use std::{sync::Arc, time::Duration};

use gtk4::{ApplicationWindow, glib, glib::clone, prelude::*};
use i18n::tr;
use snxcore::{
    browser::{BrowserController, SystemBrowser},
    model::params::TunnelParams,
};
use webkit6::{LoadEvent, NetworkSession, TLSErrorsPolicy, WebView, prelude::*};

const PASSWORD_TIMEOUT: Duration = Duration::from_secs(120);

const JS_PASSWORD_SCRIPT: &str = r#"
(function() {
  const regex1 = /sPropertyName = "password";\n\s*SNXParams\.addProperty\(sPropertyName, Function\.READ_WRITE, "([^"]+)"\);/;
  const regex2 = /Extender\.password\s*=\s*"([^"]+)"/;

  const scripts = document.querySelectorAll("script:not([src])");
  for (const s of scripts) {
    match1 = s.textContent.match(regex1);
    if (match1) return match1[1];
    match2 = s.textContent.match(regex2);
    if (match2) return match2[1];
  }

  return "";
})();
"#;

pub struct WebKitBrowser {
    params: Arc<TunnelParams>,
}

impl WebKitBrowser {
    pub fn new(params: Arc<TunnelParams>) -> Self {
        Self { params }
    }
}

#[async_trait::async_trait]
impl BrowserController for WebKitBrowser {
    fn open(&self, url: &str) -> anyhow::Result<()> {
        SystemBrowser.open(url)
    }

    fn close(&self) {}

    async fn acquire_tunnel_password(&self, url: &str) -> anyhow::Result<String> {
        let url = url.to_owned();
        let params = self.params.clone();

        let (tx, mut rx) = tokio::sync::mpsc::channel(1);

        glib::idle_add(move || {
            let window = ApplicationWindow::builder()
                .title("Mobile Access")
                .width_request(720)
                .height_request(500)
                .build();

            let session = NetworkSession::new_ephemeral();
            if params.ignore_server_cert {
                session.set_tls_errors_policy(TLSErrorsPolicy::Ignore);
            }
            let webview = WebView::builder().network_session(&session).build();

            let tx = tx.clone();
            webview.connect_load_changed(clone!(
                #[weak]
                window,
                move |webview, event| {
                    if event == LoadEvent::Finished {
                        let tx = tx.clone();
                        webview.evaluate_javascript(
                            JS_PASSWORD_SCRIPT,
                            None,
                            None,
                            gtk4::gio::Cancellable::NONE,
                            move |result| {
                                if let Ok(value) = result
                                    && value.is_string()
                                {
                                    let password = value.to_str();
                                    if !password.is_empty() {
                                        let tx = tx.clone();
                                        tokio::spawn(async move { tx.send(password.to_string()).await });
                                        window.close();
                                    }
                                }
                            },
                        );
                    }
                }
            ));

            window.set_child(Some(&webview));
            window.present();
            webview.load_uri(&url);

            glib::ControlFlow::Break
        });

        match tokio::time::timeout(PASSWORD_TIMEOUT, rx.recv()).await {
            Ok(Some(password)) => Ok(password),
            _ => anyhow::bail!(tr!("error-cannot-acquire-access-cookie")),
        }
    }
}
