# pylint:disable=arguments-renamed,global-statement
from __future__ import annotations
import copy
import os
import logging
import json
import inspect
from collections import defaultdict
from typing import Any, TYPE_CHECKING

import msgspec
import pydemumble
import archinfo

from angr.errors import AngrMissingTypeError
from angr.sim_type import parse_cpp_file, parse_file, SimTypeFunction, SimTypeBottom, SimType
from angr.calling_conventions import DEFAULT_CC, CC_NAMES
from angr.misc import autoimport
from angr.misc.ux import once
from angr.procedures.stubs.ReturnUnconstrained import ReturnUnconstrained
from angr.procedures.stubs.syscall_stub import syscall as stub_syscall

if TYPE_CHECKING:
    from angr.calling_conventions import SimCCSyscall


l = logging.getLogger(name=__name__)
SIM_LIBRARIES: dict[str, list[SimLibrary]] = {}
SIM_TYPE_COLLECTIONS: dict[str, SimTypeCollection] = {}


class SimTypeCollection:
    """
    A type collection is the mechanism for describing types. Types in a type collection can be referenced using
    """

    def __init__(self):
        self.names: list[str] | None = None
        self.types: dict[str, SimType] = {}
        self.types_json: dict[str, Any] = {}

    def __contains__(self, name: str) -> bool:
        return name in self.types or name in self.types_json

    def set_names(self, *names: str):
        self.names = list(names)
        for name in names:
            SIM_TYPE_COLLECTIONS[name] = self

    def add(self, name: str, t: SimType) -> None:
        """
        Add a type to the collection.

        :param name:    Name of the type to add.
        :param t:       The SimType object to add to the collection.
        """

        self.types[name] = t

    def get(self, name: str, bottom_on_missing: bool = False) -> SimType:
        """
        Get a SimType object from the collection as identified by the name.

        :param name:    Name of the type to get.
        :param bottom_on_missing:    Return a SimTypeBottom object if the required type does not exist.
        :return:        The SimType object.
        """
        if bottom_on_missing and name not in self:
            return SimTypeBottom(label=name)
        if name not in self:
            raise AngrMissingTypeError(name)
        if name not in self.types and name in self.types_json:
            d = self.types_json[name]
            if isinstance(d, str):
                d = msgspec.json.decode(d.replace("'", '"').encode("utf-8"))
            try:
                t = SimType.from_json(d)
            except (TypeError, ValueError) as ex:
                l.warning("Failed to load type %s from JSON", name, exc_info=True)
                # the type is missing
                if bottom_on_missing:
                    return SimTypeBottom(label=name)
                raise AngrMissingTypeError(name) from ex
            self.types[name] = t
        return self.types[name]

    def init_str(self) -> str:
        lines = [
            "typelib = SimTypeCollection()",
            "" if not self.names else f"typelib.set_names(*{self.names})",
            "typelib.types = {",
        ]
        for name in sorted(self.types):
            t = self.types[name]
            lines.append(f'    "{name}": {t._init_str()},')
        lines.append("}")

        return "\n".join(lines)

    def to_json(self, types_as_string: bool = False) -> dict[str, Any]:
        d = {"_t": "types", "names": [*self.names] if self.names else [], "types": {}}
        for name in sorted(self.types):
            t = self.types[name]
            d["types"][name] = json.dumps(t.to_json()).replace('"', "'") if types_as_string else t.to_json()
        return d

    @classmethod
    def from_json(cls, d: dict[str, Any]) -> SimTypeCollection:
        typelib = SimTypeCollection()
        if d.get("_t", "") != "types":
            raise TypeError("Not a SimTypeCollection JSON object")
        if "names" in d:
            typelib.set_names(*d["names"])
        if "types" in d:
            for name, t_value in d["types"].items():
                typelib.types_json[name] = t_value
        return typelib

    def __repr__(self):
        keys = set(self.types) | set(self.types_json)
        return f"<SimTypeCollection with {len(keys)} types>"


_ARCH_NAME_CACHE: dict[str, str] = {}


class SimLibrary:
    """
    A SimLibrary is the mechanism for describing a dynamic library's API, its functions and metadata.

    Any instance of this class (or its subclasses) found in the ``angr.procedures.definitions`` package will be
    automatically picked up and added to ``angr.SIM_LIBRARIES`` via all its names.

    :ivar fallback_cc:      A mapping from architecture to the default calling convention that should be used if no
                            other information is present. Contains some sane defaults for linux.
    :ivar fallback_proc:    A SimProcedure class that should be used to provide stub procedures. By default,
                            ``ReturnUnconstrained``.
    """

    def __init__(self):
        self.type_collection_names: list[str] = []
        self.procedures = {}
        self.non_returning = set()
        self.prototypes: dict[str, SimTypeFunction] = {}
        self.prototypes_json: dict[str, Any] = {}
        self.default_ccs = {}
        self.names = []
        self.fallback_cc = dict(DEFAULT_CC)
        self.fallback_proc = ReturnUnconstrained

    @staticmethod
    def from_json(d: dict[str, Any]) -> SimLibrary:
        lib = SimLibrary()
        if d.get("_t", "") != "lib":
            raise TypeError("Not a SimLibrary JSON object")
        if "type_collection_names" in d:
            lib.type_collection_names = d["type_collection_names"]
        if "default_cc" in d:
            if not isinstance(d["default_cc"], dict):
                raise TypeError("default_cc must be a dict")
            for arch_name, cc_name in d["default_cc"].items():
                cc = CC_NAMES[cc_name]
                lib.set_default_cc(arch_name, cc)
        if "library_names" in d:
            lib.set_library_names(*d["library_names"])
        else:
            raise KeyError("library_names is required")
        if "non_returning" in d:
            lib.set_non_returning(*d["non_returning"])
        if "functions" in d:
            lib.prototypes_json = {k: v["proto"] for k, v in d["functions"].items() if "proto" in v}
        return lib

    def copy(self):
        """
        Make a copy of this SimLibrary, allowing it to be mutated without affecting the global version.

        :return:    A new SimLibrary object with the same library references but different dict/list references
        """
        o = SimLibrary()
        o.procedures = dict(self.procedures)
        o.non_returning = set(self.non_returning)
        o.prototypes = dict(self.prototypes)
        o.prototypes_json = self.prototypes_json
        o.default_ccs = dict(self.default_ccs)
        o.names = list(self.names)
        return o

    def update(self, other: SimLibrary):
        """
        Augment this SimLibrary with the information from another SimLibrary

        :param other:   The other SimLibrary
        """
        self.procedures.update(other.procedures)
        self.non_returning.update(other.non_returning)
        self.prototypes.update(other.prototypes)
        self.default_ccs.update(other.default_ccs)

    @property
    def name(self):
        """
        The first common name of this library, e.g. libc.so.6, or '??????' if none are known.
        """
        return self.names[0] if self.names else "??????"

    def set_library_names(self, *names):
        """
        Set some common names of this library by which it may be referred during linking

        :param names:   Any number of string library names may be passed as varargs.
        """
        for name in names:
            self.names.append(name)
            if name in SIM_LIBRARIES:
                SIM_LIBRARIES[name].append(self)
            else:
                SIM_LIBRARIES[name] = [self]

    def set_default_cc(self, arch_name, cc_cls):
        """
        Set the default calling convention used for this library under a given architecture

        :param arch_name:   The string name of the architecture, i.e. the ``.name`` field from archinfo.
        :parm cc_cls:       The SimCC class (not an instance!) to use
        """
        if arch_name not in _ARCH_NAME_CACHE:
            _ARCH_NAME_CACHE[arch_name] = archinfo.arch_from_id(arch_name).name
        arch_name = _ARCH_NAME_CACHE[arch_name]
        self.default_ccs[arch_name] = cc_cls

    def set_non_returning(self, *names):
        """
        Mark some functions in this class as never returning, i.e. loops forever or terminates execution

        :param names:   Any number of string function names may be passed as varargs
        """
        for name in names:
            self.non_returning.add(name)

    def set_prototype(self, name, proto: SimTypeFunction) -> None:
        """
        Set the prototype of a function in the form of a SimTypeFunction containing argument and return types

        :param name:    The name of the function as a string
        :param proto:   The prototype of the function as a SimTypeFunction
        """
        self.prototypes[name] = proto

    def set_prototypes(self, protos: dict[str, SimTypeFunction]) -> None:
        """
        Set the prototypes of many functions

        :param protos:   Dictionary mapping function names to SimTypeFunction objects
        """
        self.prototypes.update(protos)

    def set_c_prototype(self, c_decl: str) -> tuple[str, SimTypeFunction]:
        """
        Set the prototype of a function in the form of a C-style function declaration.

        :param str c_decl: The C-style declaration of the function.
        :return:           A tuple of (function name, function prototype)
        """

        parsed = parse_file(c_decl)
        parsed_decl = parsed[0]
        if not parsed_decl:
            raise ValueError("Cannot parse the function prototype.")
        func_name, func_proto = next(iter(parsed_decl.items()))

        self.set_prototype(func_name, func_proto)

        return func_name, func_proto

    def add(self, name, proc_cls, **kwargs):
        """
        Add a function implementation to the library.

        :param name:        The name of the function as a string
        :param proc_cls:    The implementation of the function as a SimProcedure _class_, not instance
        :param kwargs:      Any additional parameters to the procedure class constructor may be passed as kwargs
        """
        self.procedures[name] = proc_cls(display_name=name, **kwargs)

    def add_all_from_dict(self, dictionary, **kwargs):
        """
        Batch-add function implementations to the library.

        :param dictionary:  A mapping from name to procedure class, i.e. the first two arguments to add()
        :param kwargs:      Any additional kwargs will be passed to the constructors of _each_ procedure class
        """
        for name, procedure in dictionary.items():
            self.add(name, procedure, **kwargs)

    def add_alias(self, name, *alt_names):
        """
        Add some duplicate names for a given function. The original function's implementation must already be
        registered.

        :param name:        The name of the function for which an implementation is already present
        :param alt_names:   Any number of alternate names may be passed as varargs
        """
        old_procedure = self.procedures[name]
        for alt in alt_names:
            new_procedure = copy.deepcopy(old_procedure)
            new_procedure.display_name = alt
            self.procedures[alt] = new_procedure
            if self.has_prototype(name):
                self.prototypes[alt] = self.get_prototype(name)  # type:ignore
            if name in self.non_returning:
                self.non_returning.add(alt)

    def _apply_metadata(self, proc, arch):
        if proc.cc is None and arch.name in self.default_ccs:
            proc.cc = self.default_ccs[arch.name](arch)
        if proc.cc is None and arch.name in self.fallback_cc:
            proc.cc = self.fallback_cc[arch.name]["Linux"](arch)
        if self.has_prototype(proc.display_name):
            proc.prototype = self.get_prototype(proc.display_name, deref=True).with_arch(arch)  # type:ignore
            proc.guessed_prototype = False
            if proc.prototype.arg_names is None:
                # Use inspect to extract the parameters from the run python function
                proc.prototype.arg_names = tuple(inspect.getfullargspec(proc.run).args[1:])
            if not proc.ARGS_MISMATCH:
                proc.num_args = len(proc.prototype.args)
        if proc.display_name in self.non_returning:
            proc.returns = False
        proc.library_name = self.name

    def get(self, name, arch):
        """
        Get an implementation of the given function specialized for the given arch, or a stub procedure if none exists.

        :param name:    The name of the function as a string
        :param arch:    The architecure to use, as either a string or an archinfo.Arch instance
        :return:        A SimProcedure instance representing the function as found in the library
        """
        if type(arch) is str:
            arch = archinfo.arch_from_id(arch)
        if name in self.procedures:
            proc = copy.deepcopy(self.procedures[name])
            self._apply_metadata(proc, arch)
            return proc
        return self.get_stub(name, arch)

    def get_stub(self, name, arch):
        """
        Get a stub procedure for the given function, regardless of if a real implementation is available. This will
        apply any metadata, such as a default calling convention or a function prototype.

        By stub, we pretty much always mean a ``ReturnUnconstrained`` SimProcedure with the appropriate display name
        and metadata set. This will appear in ``state.history.descriptions`` as ``<SimProcedure display_name (stub)>``

        :param name:    The name of the function as a string
        :param arch:    The architecture to use, as either a string or an archinfo.Arch instance
        :return:        A SimProcedure instance representing a plausable stub as could be found in the library.
        """
        proc = self.fallback_proc(display_name=name, is_stub=True)
        self._apply_metadata(proc, arch)
        return proc

    def get_prototype(self, name: str, arch=None, deref: bool = False) -> SimTypeFunction | None:
        """
        Get a prototype of the given function name, optionally specialize the prototype to a given architecture.

        :param name:    Name of the function.
        :param arch:    The architecture to specialize to.
        :param deref:   True if any SimTypeRefs in the prototype should be dereferenced using library information.
        :return:        Prototype of the function, or None if the prototype does not exist.
        """
        if name not in self.prototypes and name in self.prototypes_json:
            d = self.prototypes_json[name]
            if isinstance(d, str):
                d = msgspec.json.decode(d.replace("'", '"').encode("utf-8"))
            if not isinstance(d, dict):
                l.warning("Failed to load prototype %s from JSON", name)
                proto = None
            else:
                try:
                    proto = SimTypeFunction.from_json(d)
                except (TypeError, ValueError):
                    l.warning("Failed to load prototype %s from JSON", name, exc_info=True)
                    proto = None
            if proto is not None:
                assert isinstance(proto, SimTypeFunction)
                self.prototypes[name] = proto
        else:
            proto = self.prototypes.get(name, None)
        if proto is None:
            return None
        if deref:
            from angr.utils.types import dereference_simtype_by_lib  # pylint:disable=import-outside-toplevel

            proto = dereference_simtype_by_lib(proto, self.name)
            assert isinstance(proto, SimTypeFunction)
        if arch is not None:
            return proto.with_arch(arch)
        return proto

    def has_metadata(self, name):
        """
        Check if a function has either an implementation or any metadata associated with it

        :param name:    The name of the function as a string
        :return:        A bool indicating if anything is known about the function
        """
        return self.has_implementation(name) or name in self.non_returning or self.has_prototype(name)

    def has_implementation(self, name):
        """
        Check if a function has an implementation associated with it

        :param name:    The name of the function as a string
        :return:        A bool indicating if an implementation of the function is available
        """
        return name in self.procedures

    def has_prototype(self, func_name):
        """
        Check if a function has a prototype associated with it.

        :param str func_name: The name of the function.
        :return:              A bool indicating if a prototype of the function is available.
        :rtype:               bool
        """

        return func_name in self.prototypes or func_name in self.prototypes_json

    def is_returning(self, name: str) -> bool:
        """
        Check if a function is known to return.

        :param name:    The name of the function.
        :return:        A bool indicating if the function is known to return or not.
        """
        return name not in self.non_returning


class SimCppLibrary(SimLibrary):
    """
    SimCppLibrary is a specialized version of SimLibrary that will demangle C++ function names before looking for an
    implementation or prototype for it.
    """

    @staticmethod
    def _try_demangle(name):
        ast = pydemumble.demangle(name)
        return ast if ast else name

    @staticmethod
    def _proto_from_demangled_name(name: str) -> SimTypeFunction | None:
        """
        Attempt to extract arguments and calling convention information for a C++ function whose name was mangled
        according to the Itanium C++ ABI symbol mangling language.

        :param name:    The demangled function name.
        :return:        A prototype or None if a prototype cannot be found.
        """

        try:
            parsed, _ = parse_cpp_file(name, with_param_names=False)
        except ValueError:
            return None
        if not parsed:
            return None
        _, func_proto = next(iter(parsed.items()))
        return func_proto

    def get(self, name, arch):
        """
        Get an implementation of the given function specialized for the given arch, or a stub procedure if none exists.
        Demangle the function name if it is a mangled C++ name.

        :param str name:    The name of the function as a string
        :param arch:    The architecure to use, as either a string or an archinfo.Arch instance
        :return:        A SimProcedure instance representing the function as found in the library
        """
        demangled_name = self._try_demangle(name)
        if demangled_name not in self.procedures:
            return self.get_stub(name, arch)  # get_stub() might use the mangled name to derive the function prototype
        return super().get(demangled_name, arch)

    def get_stub(self, name, arch):
        """
        Get a stub procedure for the given function, regardless of if a real implementation is available. This will
        apply any metadata, such as a default calling convention or a function prototype. Demangle the function name
        if it is a mangled C++ name.

        :param str name:    The name of the function as a string
        :param arch:        The architecture to use, as either a string or an archinfo.Arch instance
        :return:            A SimProcedure instance representing a plausable stub as could be found in the library.
        """
        demangled_name = self._try_demangle(name)
        stub = super().get_stub(demangled_name, arch)
        # try to determine a prototype from the function name if possible
        if demangled_name != name:
            # mangled function name
            stub.prototype = self._proto_from_demangled_name(demangled_name)
            if stub.prototype is not None:
                stub.prototype = stub.prototype.with_arch(arch)
                stub.guessed_prototype = False
                if not stub.ARGS_MISMATCH:
                    stub.num_args = len(stub.prototype.args)
        return stub

    def get_prototype(self, name: str, arch=None, deref: bool = False) -> SimTypeFunction | None:
        """
        Get a prototype of the given function name, optionally specialize the prototype to a given architecture. The
        function name will be demangled first.

        :param name:    Name of the function.
        :param arch:    The architecture to specialize to.
        :param deref:   True if any SimTypeRefs in the prototype should be dereferenced using library information.
        :return:        Prototype of the function, or None if the prototype does not exist.
        """
        demangled_name = self._try_demangle(name)
        return super().get_prototype(demangled_name, arch=arch, deref=deref)

    def has_metadata(self, name):
        """
        Check if a function has either an implementation or any metadata associated with it. Demangle the function name
        if it is a mangled C++ name.

        :param name:    The name of the function as a string
        :return:        A bool indicating if anything is known about the function
        """
        name = self._try_demangle(name)
        return super().has_metadata(name)

    def has_implementation(self, name):
        """
        Check if a function has an implementation associated with it. Demangle the function name if it is a mangled C++
        name.

        :param str name:    A mangled function name.
        :return:            bool
        """
        return super().has_implementation(self._try_demangle(name))

    def has_prototype(self, func_name):
        """
        Check if a function has a prototype associated with it. Demangle the function name if it is a mangled C++ name.

        :param str name:    A mangled function name.
        :return:            bool
        """
        return super().has_prototype(self._try_demangle(func_name))


class SimSyscallLibrary(SimLibrary):
    """
    SimSyscallLibrary is a specialized version of SimLibrary for dealing not with a dynamic library's API but rather
    an operating system's syscall API. Because this interface is inherently lower-level than a dynamic library, many
    parts of this class has been changed to store data based on an "ABI name" (ABI = application binary interface,
    like an API but for when there's no programming language) instead of an architecture. An ABI name is just an
    arbitrary string with which a calling convention and a syscall numbering is associated.

    All the SimLibrary methods for adding functions still work, but now there's an additional layer on top that
    associates them with numbers.
    """

    def __init__(self):
        super().__init__()
        self.syscall_number_mapping: dict[str, dict[int, str]] = defaultdict(dict)  # keyed by abi
        self.syscall_name_mapping: dict[str, dict[str, int]] = defaultdict(dict)  # keyed by abi
        self.default_cc_mapping: dict[str, type[SimCCSyscall]] = {}  # keyed by abi
        self.syscall_prototypes: dict[str, dict[str, SimTypeFunction]] = defaultdict(dict)  # keyed by abi
        self.fallback_proc = stub_syscall

    def copy(self):
        o = SimSyscallLibrary()
        o.procedures = dict(self.procedures)
        o.non_returning = set(self.non_returning)
        o.prototypes = dict(self.prototypes)
        o.default_ccs = dict(self.default_ccs)
        o.names = list(self.names)
        o.syscall_number_mapping = defaultdict(dict, self.syscall_number_mapping)  # {abi: {number: name}}
        o.syscall_name_mapping = defaultdict(dict, self.syscall_name_mapping)  # {abi: {name: number}}
        o.syscall_prototypes = defaultdict(dict, self.syscall_prototypes)  # as above
        o.default_cc_mapping = dict(self.default_cc_mapping)  # {abi: cc}
        return o

    def update(self, other):
        super().update(other)
        if isinstance(other, SimSyscallLibrary):
            self.syscall_number_mapping.update(other.syscall_number_mapping)
            self.syscall_name_mapping.update(other.syscall_name_mapping)
            self.default_cc_mapping.update(other.default_cc_mapping)

    def minimum_syscall_number(self, abi):
        """
        :param abi: The abi to evaluate
        :return:    The smallest syscall number known for the given abi
        """
        if abi not in self.syscall_number_mapping or not self.syscall_number_mapping[abi]:
            return 0
        return min(self.syscall_number_mapping[abi])

    def maximum_syscall_number(self, abi):
        """
        :param abi: The abi to evaluate
        :return:    The largest syscall number known for the given abi
        """
        if abi not in self.syscall_number_mapping or not self.syscall_number_mapping[abi]:
            return 0
        return max(self.syscall_number_mapping[abi])

    def add_number_mapping(self, abi, number, name):
        """
        Associate a syscall number with the name of a function present in the underlying SimLibrary

        :param abi:     The abi for which this mapping applies
        :param number:  The syscall number
        :param name:    The name of the function
        """
        self.syscall_number_mapping[abi][number] = name
        self.syscall_name_mapping[abi][name] = number

    def add_number_mapping_from_dict(self, abi, mapping):
        """
        Batch-associate syscall numbers with names of functions present in the underlying SimLibrary

        :param abi:     The abi for which this mapping applies
        :param mapping: A dict mapping syscall numbers to function names
        """
        self.syscall_number_mapping[abi].update(mapping)
        self.syscall_name_mapping[abi].update({b: a for a, b in mapping.items()})

    def set_abi_cc(self, abi, cc_cls):
        """
        Set the default calling convention for an abi

        :param abi:     The name of the abi
        :param cc_cls:  A SimCC _class_, not an instance, that should be used for syscalls using the abi
        """
        self.default_cc_mapping[abi] = cc_cls

    # pylint: disable=arguments-differ
    def set_prototype(self, abi: str, name: str, proto: SimTypeFunction) -> None:  # type:ignore
        """
        Set the prototype of a function in the form of a SimTypeFunction containing argument and return types

        :param abi:     ABI of the syscall.
        :param name:    The name of the syscall as a string
        :param proto:   The prototype of the syscall as a SimTypeFunction
        """
        self.syscall_prototypes[abi][name] = proto

    # pylint: disable=arguments-differ
    def set_prototypes(self, abi: str, protos: dict[str, SimTypeFunction]) -> None:  # type:ignore
        """
        Set the prototypes of many syscalls.

        :param abi:     ABI of the syscalls.
        :param protos:  Dictionary mapping syscall names to SimTypeFunction objects
        """
        self.syscall_prototypes[abi].update(protos)

    def _canonicalize(self, number, arch, abi_list):
        if type(arch) is str:
            arch = archinfo.arch_from_id(arch)
        if type(number) is str:
            return number, arch, None
        for abi in abi_list:
            mapping = self.syscall_number_mapping[abi]
            if number in mapping:
                return mapping[number], arch, abi
        return f"sys_{number}", arch, None

    def _apply_numerical_metadata(self, proc, number, arch, abi):
        proc.syscall_number = number
        proc.abi = abi
        if abi in self.default_cc_mapping:
            cc = self.default_cc_mapping[abi](arch)
            proc.cc = cc
        elif arch.name in self.default_ccs:
            proc.cc = self.default_ccs[arch.name](arch)
        # a bit of a hack.
        name = proc.display_name
        if self.has_prototype(abi, name):
            proc.guessed_prototype = False
            proto = self.get_prototype(abi, name, deref=True)
            assert proto is not None
            proc.prototype = proto.with_arch(arch)

    def add_alias(self, name, *alt_names):
        """
        Add some duplicate names for a given function. The original function's implementation must already be
        registered.

        :param name:        The name of the function for which an implementation is already present
        :param alt_names:   Any number of alternate names may be passed as varargs
        """
        old_procedure = self.procedures[name]
        for alt in alt_names:
            new_procedure = copy.deepcopy(old_procedure)
            new_procedure.display_name = alt
            self.procedures[alt] = new_procedure
            for abi in self.syscall_prototypes:
                if self.has_prototype(abi, name):
                    self.syscall_prototypes[abi][alt] = self.get_prototype(abi, name)  # type:ignore
            if name in self.non_returning:
                self.non_returning.add(alt)

    def _apply_metadata(self, proc, arch):
        # this function is a no-op in SimSyscallLibrary; users are supposed to explicitly call
        # _apply_numerical_metadata instead.
        pass

    # pylint: disable=arguments-differ
    def get(self, number, arch, abi_list=()):  # type:ignore
        """
        The get() function for SimSyscallLibrary looks a little different from its original version.

        Instead of providing a name, you provide a number, and you additionally provide a list of abi names that are
        applicable. The first abi for which the number is present in the mapping will be chosen. This allows for the
        easy abstractions of architectures like ARM or MIPS linux for which there are many ABIs that can be used at any
        time by using syscall numbers from various ranges. If no abi knows about the number, the stub procedure with
        the name "sys_%d" will be used.

        :param number:      The syscall number
        :param arch:        The architecture being worked with, as either a string name or an archinfo.Arch
        :param abi_list:    A list of ABI names that could be used
        :return:            A SimProcedure representing the implementation of the given syscall, or a stub if no
                            implementation is available
        """
        name, arch, abi = self._canonicalize(number, arch, abi_list)
        proc = super().get(name, arch)
        proc.is_syscall = True
        self._apply_numerical_metadata(proc, number, arch, abi)
        return proc

    def get_stub(self, number, arch, abi_list=()):  # type:ignore
        """
        Pretty much the intersection of SimLibrary.get_stub() and SimSyscallLibrary.get().

        :param number:      The syscall number
        :param arch:        The architecture being worked with, as either a string name or an archinfo.Arch
        :param abi_list:    A list of ABI names that could be used
        :return:            A SimProcedure representing a plausable stub that could model the syscall
        """
        name, arch, abi = self._canonicalize(number, arch, abi_list)
        proc = super().get_stub(name, arch)
        self._apply_numerical_metadata(proc, number, arch, abi)
        l.debug("unsupported syscall: %s", number)
        return proc

    def get_prototype(  # type:ignore
        self, abi: str, name: str, arch=None, deref: bool = False
    ) -> SimTypeFunction | None:
        """
        Get a prototype of the given syscall name and its ABI, optionally specialize the prototype to a given
        architecture.

        :param abi:     ABI of the prototype to get.
        :param name:    Name of the syscall.
        :param arch:    The architecture to specialize to.
        :param deref:   True if any SimTypeRefs in the prototype should be dereferenced using library information.
        :return:        Prototype of the syscall, or None if the prototype does not exist.
        """
        if abi not in self.syscall_prototypes:
            return None
        proto = self.syscall_prototypes[abi].get(name, None)
        if proto is None:
            return None
        if deref:
            from angr.utils.types import dereference_simtype_by_lib  # pylint:disable=import-outside-toplevel

            proto = dereference_simtype_by_lib(proto, self.name)
            assert isinstance(proto, SimTypeFunction)
        return proto.with_arch(arch=arch)

    def has_metadata(self, number, arch, abi_list=()):  # type:ignore
        """
        Pretty much the intersection of SimLibrary.has_metadata() and SimSyscallLibrary.get().

        :param number:      The syscall number
        :param arch:        The architecture being worked with, as either a string name or an archinfo.Arch
        :param abi_list:    A list of ABI names that could be used
        :return:            A bool of whether or not any implementation or metadata is known about the given syscall
        """
        name, _, abi = self._canonicalize(number, arch, abi_list)
        return (
            name in self.procedures or name in self.non_returning or (abi is not None and self.has_prototype(abi, name))
        )

    def has_implementation(self, number, arch, abi_list=()):  # type:ignore
        """
        Pretty much the intersection of SimLibrary.has_implementation() and SimSyscallLibrary.get().

        :param number:      The syscall number
        :param arch:        The architecture being worked with, as either a string name or an archinfo.Arch
        :param abi_list:    A list of ABI names that could be used
        :return:            A bool of whether or not an implementation of the syscall is available
        """
        name, _, _ = self._canonicalize(number, arch, abi_list)
        return super().has_implementation(name)

    def has_prototype(self, abi: str, name: str) -> bool:  # type:ignore
        """
        Check if a function has a prototype associated with it. Demangle the function name if it is a mangled C++ name.

        :param abi:         Name of the ABI.
        :param name:        The syscall name.
        :return:            bool
        """
        if abi not in self.syscall_prototypes:
            return False
        return name in self.syscall_prototypes[abi]


#
# Autoloading
#

# By default we only load common API definitions (as defined in COMMON_LIBRARIES). For loading more definitions, the
# following logic is followed:
# - We will load all Windows APIs them if the loaded binary is a Windows binary, or when load_win32api_definitions() is
#   called.
# - We will load all APIs when load_all_definitions() is called.

_DEFINITIONS_BASEDIR = os.path.dirname(os.path.realpath(__file__))
_EXTERNAL_DEFINITIONS_DIRS: list[str] | None = None


def load_type_collections(only=None, skip=None) -> None:
    if skip is None:
        skip = set()

    # recursively list and load all _types.json files
    types_json_files = []
    for root, _, files in os.walk(_DEFINITIONS_BASEDIR):
        for filename in files:
            if filename.endswith(".json") and filename.startswith("_types_"):
                module_name = filename[7:-5]
                if only is not None and module_name not in only:
                    continue
                if module_name in skip:
                    continue
                types_json_files.append(os.path.join(root, filename))

    for f in types_json_files:
        with open(f, "rb") as fp:
            data = fp.read()
            d = msgspec.json.decode(data)
            if not isinstance(d, dict) or d.get("_t", "") != "types":
                l.warning("Invalid type collection JSON file: %s", f)
                continue
            if (
                "names" in d
                and isinstance(d["names"], list)
                and any(libname in SIM_TYPE_COLLECTIONS for libname in d["names"])
            ):
                # the type collection is already loaded
                continue
            try:
                SimTypeCollection.from_json(d)
            except TypeError:
                l.warning("Failed to load type collection from %s", f, exc_info=True)

    # supporting legacy type collections defined as Python files
    for _ in autoimport.auto_import_modules(
        "angr.procedures.definitions",
        _DEFINITIONS_BASEDIR,
        filter_func=lambda module_name: module_name.startswith("types_")
        and (only is None or (only is not None and module_name[6:] in only))
        and module_name[6:] not in skip,
    ):
        pass


def load_win32_type_collections() -> None:
    if once("load_win32_type_collections"):
        load_type_collections(only={"win32"})


def _load_definitions(base_dir: str, only: set[str] | None = None, skip: set[str] | None = None):
    if skip is None:
        skip = set()

    for f in os.listdir(base_dir):
        if f.endswith(".json") and not f.startswith("_types_"):
            module_name = f[:-5]
            if only is not None and module_name not in only:
                continue
            if module_name in skip:
                continue
            with open(os.path.join(base_dir, f), "rb") as f:
                d = msgspec.json.decode(f.read())
                if not (isinstance(d, dict) and d.get("_t", "") == "lib"):
                    l.warning("Invalid SimLibrary JSON file: %s", f)
                    continue
                try:
                    SimLibrary.from_json(d)
                except (TypeError, KeyError):
                    l.warning("Failed to load SimLibrary from %s", f, exc_info=True)

    # support for loading legacy prototype definitions defined as Python modules
    for _ in autoimport.auto_import_modules(
        "angr.procedures.definitions",
        base_dir,
        filter_func=lambda module_name: (only is None or (only is not None and module_name in only))
        and module_name not in skip,
    ):
        pass


def load_external_definitions():
    """
    Load library definitions from specific directories. By default it parses ANGR_EXTERNAL_DEFINITIONS_DIRS as a
    semicolon separated list of directory paths. Then it loads all .py files in each directory. These .py files should
    declare SimLibrary() objects and call .set_library_names() to register themselves in angr.SIM_LIBRARIES.
    """

    global _EXTERNAL_DEFINITIONS_DIRS

    if _EXTERNAL_DEFINITIONS_DIRS is None and "ANGR_EXTERNAL_DEFINITIONS_DIRS" in os.environ:
        _EXTERNAL_DEFINITIONS_DIRS = os.environ["ANGR_EXTERNAL_DEFINITIONS_DIRS"].strip('"').split(";")
        l.debug("Using external library definitions from %s", _EXTERNAL_DEFINITIONS_DIRS)
        for d in _EXTERNAL_DEFINITIONS_DIRS:
            if not os.path.isdir(d):
                l.warning("External library definitions directory %s does not exist or is not a directory.", d)

    if _EXTERNAL_DEFINITIONS_DIRS:
        # we must load all definitions prior to any external definitions are loaded. otherwise external definitions may
        # be overwritten by embedded definitions in angr, which is undesirable
        load_all_definitions()

        for d in _EXTERNAL_DEFINITIONS_DIRS:
            _load_definitions(d)


def _update_libkernel32(lib: SimLibrary):
    from angr.procedures.procedure_dict import SIM_PROCEDURES as P  # pylint:disable=import-outside-toplevel

    lib.add_all_from_dict(P["win32"])
    lib.add_alias("EncodePointer", "DecodePointer")
    lib.add_alias("GlobalAlloc", "LocalAlloc")

    lib.add("lstrcatA", P["libc"]["strcat"])
    lib.add("lstrcmpA", P["libc"]["strcmp"])
    lib.add("lstrcpyA", P["libc"]["strcpy"])
    lib.add("lstrcpynA", P["libc"]["strncpy"])
    lib.add("lstrlenA", P["libc"]["strlen"])
    lib.add("lstrcmpW", P["libc"]["wcscmp"])
    lib.add("lstrcmpiW", P["libc"]["wcscasecmp"])


def _update_libntdll(lib: SimLibrary):
    from angr.procedures.procedure_dict import SIM_PROCEDURES as P  # pylint:disable=import-outside-toplevel

    lib.add("RtlEncodePointer", P["win32"]["EncodePointer"])
    lib.add("RtlDecodePointer", P["win32"]["EncodePointer"])
    lib.add("RtlAllocateHeap", P["win32"]["HeapAlloc"])


def _update_libuser32(lib: SimLibrary):
    from angr.procedures.procedure_dict import SIM_PROCEDURES as P  # pylint:disable=import-outside-toplevel
    from angr.calling_conventions import SimCCCdecl  # pylint:disable=import-outside-toplevel

    lib.add_all_from_dict(P["win_user32"])
    lib.add("wsprintfA", P["libc"]["sprintf"], cc=SimCCCdecl(archinfo.ArchX86()))


def _update_libntoskrnl(lib: SimLibrary):
    from angr.procedures.procedure_dict import SIM_PROCEDURES as P  # pylint:disable=import-outside-toplevel

    lib.add_all_from_dict(P["win32_kernel"])


def _update_glibc(libc: SimLibrary):
    from angr.procedures.procedure_dict import SIM_PROCEDURES as P  # pylint:disable=import-outside-toplevel

    libc.add_all_from_dict(P["libc"])
    libc.add_all_from_dict(P["posix"])
    libc.add_all_from_dict(P["glibc"])
    # gotta do this since there's no distinguishing different libcs without analysis. there should be no naming
    # conflicts in the functions.
    libc.add_all_from_dict(P["uclibc"])

    # aliases for SimProcedures
    libc.add_alias("abort", "__assert_fail", "__stack_chk_fail")
    libc.add_alias("memcpy", "memmove", "bcopy")
    libc.add_alias("getc", "_IO_getc")
    libc.add_alias("putc", "_IO_putc")
    libc.add_alias("gets", "_IO_gets")
    libc.add_alias("puts", "_IO_puts")
    libc.add_alias("exit", "_exit", "_Exit")
    libc.add_alias("sprintf", "siprintf")
    libc.add_alias("snprintf", "sniprintf")


def load_win32api_definitions():
    load_win32_type_collections()
    if once("load_win32api_definitions"):
        api_base_dirs = ["win32", "wdk"]
        for api_base_dir in api_base_dirs:
            base_dir = os.path.join(_DEFINITIONS_BASEDIR, api_base_dir)
            if not os.path.isdir(base_dir):
                continue
            _load_definitions(base_dir)

        if "kernel32.dll" in SIM_LIBRARIES:
            _update_libkernel32(SIM_LIBRARIES["kernel32.dll"][0])
        if "ntdll.dll" in SIM_LIBRARIES:
            _update_libntdll(SIM_LIBRARIES["ntdll.dll"][0])
        if "user32.dll" in SIM_LIBRARIES:
            _update_libuser32(SIM_LIBRARIES["user32.dll"][0])
        if "ntoskrnl.exe" in SIM_LIBRARIES:
            _update_libntoskrnl(SIM_LIBRARIES["ntoskrnl.exe"][0])


def load_all_definitions():
    load_type_collections(skip=set())
    if once("load_all_definitions"):
        _load_definitions(_DEFINITIONS_BASEDIR)


COMMON_LIBRARIES = {
    # CGC
    "cgc",
    # (mostly) Linux
    "glibc",
    "gnulib",  # really just for .o files in coreutils
    "libstdcpp",
    "linux_kernel",
    "linux_loader",
    # Windows
    "msvcr",
}


# Load common types
load_type_collections(skip={"win32"})


# Load common definitions
_load_definitions(os.path.join(_DEFINITIONS_BASEDIR, "common"), only=COMMON_LIBRARIES)
_load_definitions(_DEFINITIONS_BASEDIR, only=COMMON_LIBRARIES)
_update_glibc(SIM_LIBRARIES["libc.so"][0])
