# Copyright (c) 2015, Frappe Technologies Pvt. Ltd. and Contributors
# License: GNU General Public License v3. See license.txt


import frappe
from frappe import _, msgprint
from frappe.model.document import Document
from frappe.utils import cint, comma_and, cstr, flt


class LeaveControlPanel(Document):
	def get_employees(self):
		conditions, values = [], []
		for field in [
			"company",
			"employment_type",
			"branch",
			"designation",
			"department",
			"employee",
			"employee_grade",
		]:
			if self.get(field):
				if field == "employee":
					conditions.append("name=%s")
				elif field == "employee_grade":
					conditions.append("grade=%s")
				else:
					conditions.append("{0}=%s".format(field))

				values.append(self.get(field))

		condition_str = " and " + " and ".join(conditions) if len(conditions) else ""

		e = frappe.db.sql(
			"select name from tabEmployee where status='Active' {condition}".format(
				condition=condition_str
			),
			tuple(values),
		)

		return e

	def validate_values(self):
		for f in ["from_date", "to_date", "leave_type", "no_of_days"]:
			if not self.get(f):
				frappe.throw(_("{0} is required").format(self.meta.get_label(f)))
		self.validate_from_to_dates("from_date", "to_date")

	@frappe.whitelist()
	def allocate_leave(self):
		self.validate_values()
		leave_allocated_for = []
		employees = self.get_employees()
		if not employees:
			frappe.throw(_("No employee found"))

		for d in self.get_employees():
			try:
				la = frappe.new_doc("Leave Allocation")
				la.set("__islocal", 1)
				la.employee = cstr(d[0])
				la.employee_name = frappe.db.get_value("Employee", cstr(d[0]), "employee_name")
				la.leave_type = self.leave_type
				la.from_date = self.from_date
				la.to_date = self.to_date
				la.carry_forward = cint(self.carry_forward)
				la.new_leaves_allocated = flt(self.no_of_days)
				la.docstatus = 1
				la.save()
				leave_allocated_for.append(d[0])
			except Exception:
				pass
		if leave_allocated_for:
			msgprint(_("Leaves Allocated Successfully for {0}").format(comma_and(leave_allocated_for)))
