use ark_ec::{
    hashing::curve_maps::{swu::SWUConfig, wb::IsogenyMap},
    models::{
        short_weierstrass::{Affine, SWCurveConfig},
        CurveConfig,
    },
};

use ark_ff::MontFp;

use crate::{g1, Fq, Fr};

type G1Affine = Affine<SwuIsoConfig>;

#[derive(Clone, Default, PartialEq, Eq)]
pub struct SwuIsoConfig;

impl CurveConfig for SwuIsoConfig {
    type BaseField = Fq;
    type ScalarField = Fr;

    /// COFACTOR = (x - 1)^2 / 3  = iso_G1.domain().order() /
    /// 8444461749428370424248824938781546531375899335154063827935233455917409239041
    //  30631250834960419227450344600217059328
    const COFACTOR: &'static [u64] = &[0x0, 0x170b5d4430000000];

    /// COFACTOR_INV = COFACTOR^{-1} mod r
    /// = 5285428838741532253824584287042945485047145357130994810877
    const COFACTOR_INV: Fr = MontFp!("5285428838741532253824584287042945485047145357130994810877");
}

// sage: iso_G1
// Isogeny of degree 2 from Elliptic Curve defined by y^2 = x^3 +
// 258664426012969092796408009721202742408018065645352501567204841856062976176281513834280849065051431927238430294002*
// x + 22 over Finite Field of size
// 258664426012969094010652733694893533536393512754914660539884262666720468348340822774968888139573360124440321458177
// to Elliptic Curve defined by y^2 = x^3 + 1 over Finite Field of size
// 258664426012969094010652733694893533536393512754914660539884262666720468348340822774968888139573360124440321458177

impl SWCurveConfig for SwuIsoConfig {
    /// COEFF_A
    const COEFF_A: Fq = MontFp!("258664426012969092796408009721202742408018065645352501567204841856062976176281513834280849065051431927238430294002");

    /// COEFF_B
    const COEFF_B: Fq = MontFp!("22");

    /// AFFINE_GENERATOR_COEFFS = (G1_GENERATOR_X, G1_GENERATOR_Y)
    const GENERATOR: G1Affine = G1Affine::new_unchecked(G1_GENERATOR_X, G1_GENERATOR_Y);
}

// sage: G1_gen  = iso_G1.domain().random_point()
// sage: G1_gen = 30631250834960419227450344600217059328* G1_gen
// sage: G1_gen.order() ==
// 8444461749428370424248824938781546531375899335154063827935233455917409239041
// True
// sage: G1_gen
// (183898640136580512316530045470998831691790391453237259434516336279447756609241220664846162561503820562316877867830 : 69018534046895515891776145953191511526693172354818719412306559690461416836925400134233128432719372819569406562974 : 1)
/// G1_GENERATOR_X =
/// 183898640136580512316530045470998831691790391453237259434516336279447756609241220664846162561503820562316877867830
pub const G1_GENERATOR_X: Fq = MontFp!("183898640136580512316530045470998831691790391453237259434516336279447756609241220664846162561503820562316877867830");

/// G1_GENERATOR_Y =
/// 69018534046895515891776145953191511526693172354818719412306559690461416836925400134233128432719372819569406562974
pub const G1_GENERATOR_Y: Fq = MontFp!("69018534046895515891776145953191511526693172354818719412306559690461416836925400134233128432719372819569406562974");

impl SWUConfig for SwuIsoConfig {
    const ZETA: Fq = MontFp!("-11"); // arbitatry primitive root of unity (element)
}

pub const ISOGENY_MAP_TO_G1 : IsogenyMap<'_, SwuIsoConfig, g1::Config, > = IsogenyMap {
    x_map_numerator : &[
        MontFp!("193998319509726820447277314072485610595876362210707887456279225959507476652652651634192264150953923683470146535424"), 
        MontFp!("40474824132456359704279181570318738632422647360355249739068643631356267969150730939906729705473"), 
        MontFp!("193998319509726820507989550271170150152295134566185995404913197000040351261255617081226666104680020093330241093633"),
    ],

    x_map_denominator : &[
        MontFp!("161899296529825438817116726281274954529690589441420998956274574525425071876602923759626918821892"), 
        MontFp!("1"),
    ],

    y_map_numerator : &[
        MontFp!("193998319509726820507989550271170150152295134566185995404913197000040351261255617081226666104680020093330241093631"), 
        MontFp!("32333053251621136903112182208573040583096119983059602439070460434672245065050016464457115901761911040205276577794"), 
        MontFp!("129332213006484547066038603046131306324615528732935438218576102373893108782773376834518846023512776472080255287298"), 
        MontFp!("226331372761347957259321141983031841844344323660550327972398729833380409804798219928097777122126690108885281275905"),
    ],

    y_map_denominator : &[
        MontFp!("258664426012969094010652733694893533536393512754914660539884262666720468348340822774968888139573360124440321458169"), 
        MontFp!("971395779178952632902700357687649727178143536648525993737647447152550431259617542557761512931340"), 
        MontFp!("485697889589476316451350178843824863589071768324262996868823723576275215629808771278880756465676"), 
        MontFp!("1"),
    ],
};

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_gen() {
        let gen: G1Affine = SwuIsoConfig::GENERATOR;
        assert!(gen.is_on_curve());
        assert!(gen.is_in_correct_subgroup_assuming_on_curve());
    }
}
