use ark_ec::{
    models::CurveConfig,
    twisted_edwards::{Affine, MontCurveConfig, Projective, TECurveConfig},
};
use ark_ff::MontFp;

use crate::{fq::Fq, fr::Fr};

#[cfg(test)]
mod tests;

pub type EdwardsAffine = Affine<EdwardsConfig>;
pub type EdwardsProjective = Projective<EdwardsConfig>;

#[derive(Clone, Default, PartialEq, Eq)]
pub struct EdwardsConfig;

impl CurveConfig for EdwardsConfig {
    type BaseField = Fq;
    type ScalarField = Fr;

    /// COFACTOR = 4
    const COFACTOR: &'static [u64] = &[4];

    /// COFACTOR_INV (mod r) =
    /// 29745142885578832859584328103315528221570304936126890280067991221921526670592508030983158
    const COFACTOR_INV: Fr = MontFp!(
        "29745142885578832859584328103315528221570304936126890280067991221921526670592508030983158"
    );
}

// Many parameters need to be written down in the Montgomery residue form,
// discussed below. Some useful numbers:
// R for Fq: 223364648326281414938801705359223029554923725549792420683051274872200260503540791531766876
// R for Fr: 104384076783966083500464392945960916666734135485183910065100558776489954102951241798239545

impl TECurveConfig for EdwardsConfig {
    /// COEFF_A = -1
    /// Needs to be in the Montgomery residue form in Fq
    /// I.e., -1 * R for Fq
    ///     = 252557637842979910814547544293825421990201153003031094870216460866964386803867699028196261
    const COEFF_A: Fq = MontFp!("-1");

    /// COEFF_D = 4212
    /// Needs to be in the Montgomery residue form in Fq
    /// I.e., 4212 * R for Fq
    ///     = 389461279836940033614665658623660232171971995346409183754923941118154161474636585314923000
    const COEFF_D: Fq = MontFp!("4212");

    /// Generated randomly
    const GENERATOR: EdwardsAffine = EdwardsAffine::new_unchecked(GENERATOR_X, GENERATOR_Y);

    type MontCurveConfig = EdwardsConfig;

    /// Multiplication by `a` is just negation.
    #[inline(always)]
    fn mul_by_a(elem: Self::BaseField) -> Self::BaseField {
        -elem
    }
}

impl MontCurveConfig for EdwardsConfig {
    /// COEFF_A = 203563247015667910991582090642011229452721346107806307863040223071914240315202967004285204
    const COEFF_A: Fq = MontFp!("203563247015667910991582090642011229452721346107806307863040223071914240315202967004285204");

    /// COEFF_B = 272359039153593414761767159011037222092403532445017207690227512667250406992205523555677931
    const COEFF_B: Fq = MontFp!("272359039153593414761767159011037222092403532445017207690227512667250406992205523555677931");

    type TECurveConfig = EdwardsConfig;
}

/// GENERATOR_X =
/// 282406820114868156776872298252698015906762052916420164316497572033519876761239463633892227
pub const GENERATOR_X: Fq = MontFp!(
    "282406820114868156776872298252698015906762052916420164316497572033519876761239463633892227"
);

/// GENERATOR_Y =
/// 452667754940241021433619311795265643711152068500301853535337412655162600774122192283142703
pub const GENERATOR_Y: Fq = MontFp!(
    "452667754940241021433619311795265643711152068500301853535337412655162600774122192283142703"
);
