/**
 * 组件名称：Textarea 多行文本输入框
 * 单测内容：
 * 1. 修改组件内容 value
 * 2. 去除收尾空白 trimContents 与 是否显示计数器 showCounter
 * 3. 最大输入长度 maxLength、是否可清除 clearable 与 重置值 resetValue
 * 4. 是否只读 readOnly
 */

import React from 'react';
import {render, fireEvent, waitFor} from '@testing-library/react';
import '../../../src';
import {render as amisRender} from '../../../src';
import {makeEnv, wait} from '../../helper';

const setup = async (items: any[] = []) => {
  const util = render(
    amisRender(
      {
        type: 'form',
        api: '/api/xxx',
        controls: items,
        title: 'The form'
      },
      {},
      makeEnv({})
    )
  );

  const textarea = util.container.querySelector(
    '.cxd-TextareaControl > textarea.cxd-TextareaControl-input'
  )!;

  expect(textarea).toBeInTheDocument();

  return {
    ...util,
    textarea
  };
};

test('Renderer:textarea', async () => {
  const {container} = render(
    amisRender(
      {
        type: 'form',
        api: '/api/xxx',
        controls: [
          {
            type: 'textarea',
            name: 'a',
            label: 'textarea',
            minRows: 3,
            maxRows: 10,
            trimContents: true,
            value: '123'
          }
        ],
        title: 'The form',
        actions: []
      },
      {},
      makeEnv({})
    )
  );

  await waitFor(() => {
    expect(container.querySelector('[name="a"]')).toBeInTheDocument();
  });
  const textarea = container.querySelector('textarea');
  expect(textarea?.innerHTML).toEqual('123');

  fireEvent.change(textarea!, {
    target: {
      value: '456'
    }
  });
  await waitFor(() => {
    const textareaChanged = container.querySelector('textarea');
    expect(textareaChanged?.innerHTML).toEqual('456');
  });

  expect(container).toMatchSnapshot();
});

test('Renderer:textarea with trimContents & showCounter', async () => {
  const {container, textarea} = await setup([
    {
      type: 'textarea',
      name: 'text',
      label: 'textarea',
      trimContents: true,
      showCounter: true,
      value: '123'
    }
  ]);

  expect(
    container.querySelector('.cxd-TextareaControl-counter')
  ).toBeInTheDocument();
  expect(
    (container.querySelector('.cxd-TextareaControl-counter') as Element)
      .innerHTML
  ).toBe('<span>3</span>');

  fireEvent.change(textarea, {target: {value: '   12345   '}});
  fireEvent.blur(textarea);
  await wait(500);

  expect(textarea.innerHTML).toBe('12345');
  expect(
    (container.querySelector('.cxd-TextareaControl-counter') as Element)
      .innerHTML
  ).toBe('<span>5</span>');

  expect(container).toMatchSnapshot();
});

test('Renderer:textarea with maxLength & clearable & resetValue', async () => {
  const {container, textarea} = await setup([
    {
      type: 'textarea',
      name: 'text',
      label: 'textarea',
      clearable: true,
      resetValue: 'i am reset value',
      showCounter: true,
      maxLength: 9,
      value: '123'
    }
  ]);

  expect(
    container.querySelector('.cxd-TextareaControl-clear')
  ).toBeInTheDocument();
  expect(
    (container.querySelector('.cxd-TextareaControl-counter') as Element)
      .innerHTML
  ).toBe('<span>3</span><i>/</i><span>9</span>');

  fireEvent.click(container.querySelector('.cxd-TextareaControl-clear')!);
  await waitFor(() => {
    expect(textarea.innerHTML).toBe('i am reset value');
    expect(
      (container.querySelector('.cxd-TextareaControl-counter') as Element)
        .innerHTML
    ).toBe('<span>16</span><i>/</i><span>9</span>');
  });

  const submitBtn = container.querySelector(
    'button[type="submit"]'
  ) as HTMLElement;

  fireEvent.click(submitBtn);

  await waitFor(() => {
    expect(
      container.querySelector(
        '.cxd-TextareaControl.is-error.has-error--maxLength'
      )
    ).toBeInTheDocument();
  });

  expect(container).toMatchSnapshot();
});

test('Renderer:textarea with readOnly', async () => {
  const {container, textarea} = await setup([
    {
      type: 'textarea',
      name: 'text',
      label: 'textarea',
      readOnly: true,
      value: '123'
    }
  ]);

  expect(container.querySelector('textarea[readonly]')).toBeInTheDocument();

  expect(container).toMatchSnapshot();
});
