import {render, waitFor, cleanup} from '@testing-library/react';
import '../../src';
import {render as amisRender} from '../../src';
import {makeEnv} from '../helper';
import 'jest-canvas-mock';

afterEach(() => {
  cleanup();
});

const setupQRCode = async (qrcodeProps: any = {}) => {
  const result = render(
    amisRender(
      {
        type: 'page',
        body: {
          type: 'qr-code',
          value: 'amis',
          codeSize: 128,
          ...qrcodeProps
        }
      },
      {},
      makeEnv({})
    )
  );

  await waitFor(() => {
    expect(result.container.querySelector('.cxd-QrCode')).toBeInTheDocument();
  });

  return {
    ...result,
    qrcode: result.container.querySelector('.cxd-QrCode'),
    svgEl: result.container.querySelector('.cxd-QrCode')?.firstElementChild,
    canvasEl: result.container.querySelector('.cxd-QrCode')?.firstElementChild
  };
};

describe('Renderer:qr-code', () => {
  test('QRCode render with svg', async () => {
    const {container, svgEl} = await setupQRCode({
      mode: 'svg'
    });

    expect(svgEl).not.toBeNull();
    expect(container).toMatchSnapshot();
  });

  test('QRCode with background/foreground color in svg', async () => {
    const {container, svgEl} = await setupQRCode({
      backgroundColor: '#108cee',
      foregroundColor: 'yellow',
      mode: 'svg'
    });

    expect(svgEl?.firstElementChild?.getAttribute('fill')).toBe('#108cee');
    expect(svgEl?.lastElementChild?.getAttribute('fill')).toBe('yellow');
    expect(container).toMatchSnapshot();
  });

  test('QRCode with image in svg', async () => {
    const {container, svgEl} = await setupQRCode({
      mode: 'svg',
      imageSettings: {
        src: 'https://internal-amis-res.cdn.bcebos.com/images/2020-1/1578395692722/4f3cb4202335.jpeg@s_0,w_216,l_1,f_jpg',
        width: 50,
        height: 30,
        x: 20,
        y: 30
      }
    });
    const imageEl = svgEl?.lastElementChild;

    expect(imageEl).not.toBeNull();
    expect(imageEl?.getAttribute('xlink:href')).not.toBeUndefined();
    expect(Number(imageEl?.getAttribute('x'))).toBeGreaterThan(0);
    expect(Number(imageEl?.getAttribute('y'))).toBeGreaterThan(0);
    expect(Number(imageEl?.getAttribute('width'))).toBeGreaterThan(0);
    expect(Number(imageEl?.getAttribute('height'))).toBeGreaterThan(0);
    expect(container).toMatchSnapshot();
  });

  test('QRCode render with canvas', async () => {
    const {container, canvasEl} = await setupQRCode({});

    expect(canvasEl).not.toBeNull();
    ((canvasEl as HTMLCanvasElement).toDataURL as any).mockReturnValueOnce(
      'data:image/png;base64, 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'
    );
  });

  test('QRCode with background/foreground color in canvas', async () => {
    const {container, canvasEl} = await setupQRCode({
      backgroundColor: '#108cee',
      foregroundColor: 'yellow'
    });
    expect(canvasEl).not.toBeNull();
    ((canvasEl as HTMLCanvasElement).toDataURL as any).mockReturnValueOnce(
      'data:image/png;base64,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'
    );
  });

  test('QRCode with image in canvas', async () => {
    const {container, canvasEl} = await setupQRCode({
      imageSettings: {
        src: 'https://internal-amis-res.cdn.bcebos.com/images/2020-1/1578395692722/4f3cb4202335.jpeg@s_0,w_216,l_1,f_jpg',
        width: 50,
        height: 30,
        x: 20,
        y: 30
      }
    });
    expect(canvasEl).not.toBeNull();
    ((canvasEl as HTMLCanvasElement).toDataURL as any).mockReturnValueOnce(
      'data:image/png;base64,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'
    );
  });
});
