import {Button} from 'amis';
import type {SchemaCollection} from 'amis';
import React from 'react';
import {
  BaseEventContext,
  BasePlugin,
  BasicToolbarItem,
  ContextMenuEventContext,
  ContextMenuItem,
  getSchemaTpl,
  registerEditorPlugin,
  diff,
  JSONPipeOut,
  EditorNodeType,
  jsonToJsonSchema
} from 'amis-editor-core';
import {generateId} from '../../util';

export class SubFormControlPlugin extends BasePlugin {
  static id = 'SubFormControlPlugin';
  // 关联渲染器名字
  rendererName = 'input-sub-form';
  $schema = '/schemas/SubFormControlSchema.json';

  // 组件名称
  name = '子表单项';
  isBaseComponent = true;
  icon = 'fa fa-window-restore';
  pluginIcon = 'sub-form-plugin';
  description = 'SubForm, 配置一个子 form 作为当前的表单项';
  docLink = '/amis/zh-CN/components/form/input-sub-form';
  tags = ['表单项'];
  scaffold = {
    type: 'input-sub-form',
    name: 'subform',
    label: '子表单',
    form: {
      title: '标题',
      body: [
        {
          type: 'input-text',
          label: '文本',
          id: generateId(),
          name: 'text'
        }
      ]
    }
  };
  previewSchema: any = {
    type: 'form',
    className: 'text-left',
    mode: 'horizontal',
    wrapWithPanel: false,
    body: [
      {
        ...this.scaffold
      }
    ]
  };

  panelTitle = '子表单项';
  panelBodyCreator = (context: BaseEventContext) => {
    return [
      getSchemaTpl('layout:originPosition', {value: 'left-top'}),
      {
        children: ({value, onChange}: any) => {
          return (
            <Button
              size="sm"
              level="primary"
              className="m-b"
              block
              onClick={this.editDetail.bind(this, context.id)}
            >
              配置成员渲染器
            </Button>
          );
        }
      },
      {
        name: 'labelField',
        type: 'input-text',
        value: 'label',
        label: '名称字段名',
        description: '当值中存在这个字段，则按钮名称将使用此字段的值来展示。'
      },
      getSchemaTpl('btnLabel', {
        label: '按钮标签名',
        value: '设置'
      }),
      {
        name: 'minLength',
        visibleOn: 'this.multiple',
        label: '允许最少个数',
        type: 'input-number'
      },

      {
        name: 'maxLength',
        visibleOn: 'this.multiple',
        label: '允许最多个数',
        type: 'input-number'
      }
    ] as SchemaCollection;
  };

  filterProps(props: any, node: EditorNodeType) {
    props = JSONPipeOut(props);

    // 至少显示一个成员，否则啥都不显示。
    // 至少显示一个成员，否则啥都不显示。
    if (!node.state.value && !props.value) {
      node.updateState({
        value: ['']
      });
    }

    return props;
  }

  buildEditorToolbar(
    {id, info}: BaseEventContext,
    toolbars: Array<BasicToolbarItem>
  ) {
    if (info.renderer.name === 'input-sub-form') {
      toolbars.push({
        icon: 'fa fa-expand',
        order: 100,
        tooltip: '配置成员渲染器',
        onClick: this.editDetail.bind(this, id)
      });
    }
  }

  buildEditorContextMenu(
    {id, schema, region, info}: ContextMenuEventContext,
    menus: Array<ContextMenuItem>
  ) {
    if (info.renderer.name === 'input-sub-form') {
      menus.push('|', {
        label: '配置成员渲染器',
        onSelect: this.editDetail.bind(this, id)
      });
    }
  }

  editDetail(id: string) {
    const manager = this.manager;
    const store = manager.store;
    const node = store.getNodeById(id);
    const value = store.getValueOf(id);
    if (!node || !value) {
      return;
    }

    const {
      title,
      actions,
      name,
      size,
      closeOnEsc,
      showCloseButton,
      bodyClassName,
      type,
      ...rest
    } = value.form;
    const schema = {
      title,
      actions,
      name,
      size,
      closeOnEsc,
      showCloseButton,
      bodyClassName,
      type: 'dialog',
      body: {
        type: 'form',
        className: 'h-full pl-4 pr-4',
        ...rest
      }
    };

    this.manager.openSubEditor({
      title: '配置子表单项',
      value: schema,
      memberImmutable: ['body'],
      onChange: newValue => {
        const {
          title,
          actions,
          name,
          size,
          closeOnEsc,
          showCloseButton,
          bodyClassName,
          body
        } = newValue;

        newValue = {
          ...value,
          form: {
            title,
            actions,
            name,
            size,
            closeOnEsc,
            showCloseButton,
            bodyClassName,
            ...body[0]
          }
        };
        // delete newValue.form.body;
        delete newValue.form.type;
        manager.panelChangeValue(newValue, diff(value, newValue));
      }
    });
  }

  async buildDataSchemas(
    node: EditorNodeType,
    region: EditorNodeType,
    trigger?: EditorNodeType
  ) {
    // 渲染出来才能取到孩子，所以现在subform现在是拿不到的，so这里只提供基本类型，不展开
    let dataSchema: any = {
      type: 'object',
      title: node.schema?.label || node.schema?.name,
      originalValue: node.schema?.value // 记录原始值，循环引用检测需要
    };

    if (node.schema?.multiple) {
      dataSchema = {
        type: 'array',
        title: node.schema?.label || node.schema?.name,
        originalValue: dataSchema.originalValue
      };
    }

    return dataSchema;
  }
}

registerEditorPlugin(SubFormControlPlugin);
