import { Diablo2Level, Diablo2LevelNpcSuper, Diablo2MpqData } from '@diablo2/data';
import { Diablo2MpqLoader } from '@diablo2/bintools';
import { toHex } from 'binparse';
import { ChildProcess, spawn } from 'child_process';
import { promises as fs } from 'fs';
import { EventEmitter } from 'events';
import PLimit from 'p-limit';
import { createInterface } from 'readline';
import { Log, LogType } from '../logger.js';
import { run } from './child.process.js';
import { LruCache } from './lru.js';
import { Diablo2MapGenMessage, MapGenMessageInfo, MapGenMessageMap } from './map.js';
import { F_OK } from 'constants';

export const MapCommand = ['./bin/d2-map.exe', '../bin/d2-map.exe'];
export const Diablo2Path = '/app/game';
export const RegistryPath = '/app/d2.install.reg';
export const WineCommand = 'wine';

/** Wait at most 10 seconds for things to work */
const ProcessTimeout = 30_000;
/** Attempt to generate the map 3 times */
const MaxAttempts = 3;
const MaxMapsToGenerate = 5000;

interface LogMessage {
  time: number;
  level: number;
  msg: string;
  id?: string;
}

export function isLogMessage(x: unknown): x is LogMessage {
  if (typeof x !== 'object') return false;
  if (x == null) return false;
  if ('time' in x) return true;
  return false;
}

async function timeOut(message: string, timeout: number): Promise<void> {
  return new Promise((resolve, reject) => {
    const timer = setTimeout(() => reject(`${message} Timeout after ${timeout}ms`), timeout);
    timer.unref();
  });
}

function getJson<T>(s: string): T | null {
  try {
    return JSON.parse(s);
  } catch (e) {
    return null;
  }
}

function fileExists(f: string): Promise<boolean> {
  return fs
    .access(f, F_OK)
    .then(() => true)
    .catch(() => false);
}

let ProcId = 0;
const MapCache: LruCache<Diablo2Level[]> = new LruCache(100);

// Disable wine debug logging
const cwd = process.cwd();
export class Diablo2MapProcess {
  process: ChildProcess | null;
  /** Number of maps generated by this process */
  generatedCount = 0;
  events: EventEmitter = new EventEmitter();
  mpq: Diablo2MpqData;

  mapCommand: string;
  id: number;

  constructor(mpq: Diablo2MpqData, mapCommand: string) {
    this.mpq = mpq;
    this.mapCommand = mapCommand;
    this.id = ProcId++;
  }
  /**
   * Limit the map generation to a single thread
   * TODO having a pool of these map processes would be quite nice
   */
  q = PLimit(1);

  /** Get the version of WINE that is being used */
  async version(log: LogType): Promise<string> {
    const versionResponse = await run(WineCommand, ['--version']);
    const version = versionResponse.stdout;
    log.info({ version, command: WineCommand }, 'MapProcess:WineVersion');
    return version;
  }

  get isRunning(): boolean {
    return this.process != null;
  }

  /** Start the map process waiting for the `init` event before allowing anything to continue */
  async start(log: LogType): Promise<void> {
    if (this.isRunning) return;
    this.generatedCount = 0;

    const args = [this.mapCommand, Diablo2Path];
    log.info({ proc: this.id, wineArgs: args }, 'MapProcess:Starting');
    return new Promise(async (resolve) => {
      const proc = spawn(WineCommand, args, { cwd, env: { WINEPREFIX: process.env['WINEPREFIX'], WINEDEBUG: '-all' } });
      if (proc == null || proc.stdout == null) throw new Error('Failed to start command');
      this.process = proc;
      proc.stderr.on('data', (data) => {
        const line = data.toString().trim();
        if (line.includes('FS volume label and serial are not available')) return;
        Log.debug({ proc: this.id, data: line }, 'MapProcess:stderr');
        if (line.includes('We got a big Error here')) this.stop(log);
      });
      proc.on('error', (error) => {
        log.fatal({ proc: this.id, error }, 'MapProcess:Died');
        inter.close();
        this.process = null;
      });
      proc.on('close', (exitCode) => {
        inter.close();
        this.process = null;
        this.events.emit('close');
        if (exitCode == null) return;
        if (exitCode > 0) log.fatal({ proc: this.id, exitCode }, 'MapProcess:Closed');
      });

      log.info({ proc: this.id, processId: this.process.pid }, 'MapProcess:Started');
      const inter = createInterface(proc.stdout).on('line', (line): unknown => {
        const json = getJson<Diablo2MapGenMessage | LogMessage>(line);
        if (json == null) return;
        if (isLogMessage(json)) return this.events.emit('log', json);
        if (json.type) return this.events.emit(json.type, json);
        return;
      });
      await this.once('init');
      resolve();
    });
  }

  async once<T extends Diablo2MapGenMessage>(e: T['type'], cb?: () => void): Promise<T> {
    return Promise.race([
      new Promise((resolve) => {
        this.events.once(e, (data) => resolve(data));
        cb?.();
      }),
      timeOut(`Event: ${e}`, ProcessTimeout),
    ]) as Promise<T>;
  }

  async stop(log: LogType): Promise<void> {
    if (this.process == null) return;
    log.info({ proc: this.id, processId: this.process.pid }, 'MapProcess:Stop');
    this.process.kill('SIGKILL');
    this.process = null;
  }

  async command(cmd: 'seed' | 'difficulty' | 'act', value: number, log: LogType): Promise<void> {
    const startTime = Date.now();
    if (this.process == null) await this.start(log);
    const command = `$${cmd} ${value}\n`;
    const res = await this.once<MapGenMessageInfo>('info', () => this.process?.stdin?.write(command));
    if (res[cmd] !== value) {
      throw new Error(`Failed to set ${cmd}=${value} (output: ${JSON.stringify(res)}: ${command})`);
    }

    log.trace({ cmd, value, duration: Date.now() - startTime }, 'MapProcess:Command');
  }

  map(seed: number, difficulty: number, actId: number, log: LogType): Promise<Diablo2Level[]> {
    const mapKey = `${seed}_${difficulty}_${actId}`;
    const cacheData = MapCache.get(mapKey);
    if (cacheData != null) return Promise.resolve(cacheData);
    return this.q(async () => {
      const mapResult = await this.getMaps(seed, difficulty, actId, log);
      MapCache.set(mapKey, mapResult);
      return mapResult;
    });
  }

  private async getMaps(
    seed: number,
    difficulty: number,
    actId: number,
    log: LogType,
    attempt = 0,
  ): Promise<Diablo2Level[]> {
    if (this.generatedCount > MaxMapsToGenerate) {
      log.warn('GenerateMap:Restart');
      this.generatedCount = 0;
      await this.stop(log);
      await this.start(log);
    }

    if (!this.isRunning) await this.start(log);

    await this.command('seed', seed, log);
    await this.command('difficulty', difficulty, log);
    if (actId > -1) await this.command('act', actId, log);

    this.generatedCount++;
    log.info({ proc: this.id, seed: toHex(seed, 8), difficulty, generated: this.generatedCount }, 'GenerateMap:Start');
    const maps: Map<number, Diablo2Level> = new Map();

    const newMap = (msg: MapGenMessageMap): void => {
      log?.trace({ proc: this.id, mapId: msg.id }, 'GenerateMap:GotMap');
      maps.set(msg.id, this.fixMap(msg));
    };

    return await new Promise((resolve, reject) => {
      const errorHandle = (err: unknown): void => {
        this.events.off('map', newMap);
        log.info({ proc: this.id, seed: toHex(seed, 8), difficulty, attempt, err }, 'GenerateMap:Error');
        if (attempt < MaxAttempts) {
          this.getMaps(seed, difficulty, actId, log, attempt + 1)
            .then(resolve)
            .catch(reject);
        } else {
          console.log('Reject', { attempt, MaxAttempts });
          reject(err);
        }
      };
      const logLine = (json: LogMessage): void => {
        if (json.level < 30) return;
        log.info({ proc: this.id, ...json, log: json.msg }, 'MapProcess:Log');
      };
      this.events.once('close', errorHandle);
      this.events.on('map', newMap);
      this.events.on('log', logLine);
      this.events.once('done', () => {
        this.events.off('map', newMap);
        this.events.off('close', errorHandle);
        this.events.off('log', logLine);
        log?.trace({ proc: this.id, count: Object.keys(maps).length }, 'GenerateMap:Generated');
        resolve([...maps.values()]);
      });
      this.process?.stdin?.write(`$map\n`);
    });
  }

  /** Correct the names of npcs and objects */
  fixMap(map: MapGenMessageMap): MapGenMessageMap {
    map.name = this.mpq.levels.get(map.id)?.name.trim() ?? map.name;
    for (const obj of map.objects) {
      if (obj.type === 'npc') {
        if (obj.id >= this.mpq.monsters.size) {
          const superId = obj.id - this.mpq.monsters.size;

          if (superId < this.mpq.monsters.superUniques.length) {
            obj.name = this.mpq.monsters.superUniqueName(superId);
            (obj as Diablo2LevelNpcSuper).isSuperUnique = true;
            (obj as Diablo2LevelNpcSuper).superId = superId;
          }
        } else {
          obj.name = this.mpq.monsters.name(obj.id)?.trim();
        }
      }

      // Force lowercase all the sub types
      if (obj.type === 'object') {
        obj.name = this.mpq.objects.get(obj.id)?.name.trim();
      }

      if (obj.type === 'exit') {
        obj.name = this.mpq.levels.get(obj.id)?.name.trim();
      }
    }

    return map;
  }
}

async function findD2MapExe(): Promise<string | null> {
  for (const mc of MapCommand) {
    if (await fileExists(mc)) return mc;
  }

  return null;
}

export class MapCluster {
  static ProcessCount = 1;
  static instance: Promise<MapCluster>;

  static async get(log: LogType): Promise<MapCluster> {
    if (this.instance != null) return this.instance;
    this.instance = this.create(log);
    return this.instance;
  }

  static async create(log: LogType): Promise<MapCluster> {
    const mpq = await Diablo2MpqLoader.load(Diablo2Path);

    const mapCommand = await findD2MapExe();
    if (mapCommand == null) throw new Error('MapProcess:MissingMapExe');
    log.info({ exe: mapCommand }, 'MapProcess:ExeFound');

    if (await fileExists(RegistryPath)) {
      const res = await run(WineCommand, ['regedit', RegistryPath]);
      log.info({ data: res.stdout }, 'MapProcess:Registry:Update');
    }

    return new MapCluster(mpq, mapCommand);
  }

  static async map(seed: number, difficulty: number, actId: number, log: LogType): Promise<Diablo2Level[]> {
    const cluster = await MapCluster.get(log);
    const process = cluster.process;
    if (!process.isRunning) await process.start(log);
    return process.map(seed, difficulty, actId, log);
  }

  processes: Diablo2MapProcess[] = [];
  currentIndex = 0;
  constructor(mpq: Diablo2MpqData, mapCommand: string) {
    for (let i = 0; i < MapCluster.ProcessCount; i++) {
      this.processes.push(new Diablo2MapProcess(mpq, mapCommand));
    }
  }

  get process(): Diablo2MapProcess {
    const currentIndex = this.currentIndex;
    this.currentIndex = (currentIndex + 1) % this.processes.length;
    return this.processes[currentIndex];
  }
}
