// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

package network_test

import (
	"context"
	"net/netip"
	"net/url"
	"sync"
	"testing"
	"time"

	"github.com/cosi-project/runtime/pkg/controller/runtime"
	"github.com/cosi-project/runtime/pkg/resource"
	"github.com/cosi-project/runtime/pkg/resource/rtestutils"
	"github.com/cosi-project/runtime/pkg/state"
	"github.com/cosi-project/runtime/pkg/state/impl/inmem"
	"github.com/cosi-project/runtime/pkg/state/impl/namespaced"
	"github.com/siderolabs/go-pointer"
	"github.com/siderolabs/go-procfs/procfs"
	"github.com/siderolabs/go-retry/retry"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
	"go.uber.org/zap/zaptest"

	"github.com/cozystack/talm/internal/app/machined/pkg/controllers/ctest"
	netctrl "github.com/cozystack/talm/internal/app/machined/pkg/controllers/network"
	"github.com/siderolabs/talos/pkg/machinery/config/container"
	"github.com/siderolabs/talos/pkg/machinery/config/types/v1alpha1"
	"github.com/siderolabs/talos/pkg/machinery/constants"
	"github.com/siderolabs/talos/pkg/machinery/resources/config"
	"github.com/siderolabs/talos/pkg/machinery/resources/network"
)

type ResolverConfigSuite struct {
	suite.Suite

	state state.State

	runtime *runtime.Runtime
	wg      sync.WaitGroup

	ctx       context.Context //nolint:containedctx
	ctxCancel context.CancelFunc
}

func (suite *ResolverConfigSuite) State() state.State { return suite.state }

func (suite *ResolverConfigSuite) Ctx() context.Context { return suite.ctx }

func (suite *ResolverConfigSuite) SetupTest() {
	suite.ctx, suite.ctxCancel = context.WithTimeout(context.Background(), 3*time.Minute)

	suite.state = state.WrapCore(namespaced.NewState(inmem.Build))

	var err error

	suite.runtime, err = runtime.NewRuntime(suite.state, zaptest.NewLogger(suite.T()))
	suite.Require().NoError(err)
}

func (suite *ResolverConfigSuite) startRuntime() {
	suite.wg.Add(1)

	go func() {
		defer suite.wg.Done()

		suite.Assert().NoError(suite.runtime.Run(suite.ctx))
	}()
}

func (suite *ResolverConfigSuite) assertResolvers(requiredIDs []string, check func(*network.ResolverSpec, *assert.Assertions)) {
	assertResources(suite.ctx, suite.T(), suite.state, requiredIDs, check, rtestutils.WithNamespace(network.ConfigNamespaceName))
}

func (suite *ResolverConfigSuite) assertNoResolver(id string) error {
	resources, err := suite.state.List(
		suite.ctx,
		resource.NewMetadata(network.ConfigNamespaceName, network.ResolverSpecType, "", resource.VersionUndefined),
	)
	if err != nil {
		return err
	}

	for _, res := range resources.Items {
		if res.Metadata().ID() == id {
			return retry.ExpectedErrorf("spec %q is still there", id)
		}
	}

	return nil
}

func (suite *ResolverConfigSuite) TestDefaults() {
	suite.Require().NoError(suite.runtime.RegisterController(&netctrl.ResolverConfigController{}))

	suite.startRuntime()

	suite.assertResolvers(
		[]string{
			"default/resolvers",
		}, func(r *network.ResolverSpec, asrt *assert.Assertions) {
			asrt.Equal(
				[]netip.Addr{
					netip.MustParseAddr(constants.DefaultPrimaryResolver),
					netip.MustParseAddr(constants.DefaultSecondaryResolver),
				}, r.TypedSpec().DNSServers,
			)
			asrt.Empty(r.TypedSpec().SearchDomains)
			asrt.Equal(network.ConfigDefault, r.TypedSpec().ConfigLayer)
		},
	)
}

func (suite *ResolverConfigSuite) TestWithHostnameStatus() {
	suite.Require().NoError(suite.runtime.RegisterController(&netctrl.ResolverConfigController{}))

	hostnameStatus := network.NewHostnameStatus(network.NamespaceName, network.HostnameID)
	hostnameStatus.TypedSpec().Hostname = "irrelevant"
	hostnameStatus.TypedSpec().Domainname = "example.org"
	suite.Require().NoError(suite.state.Create(suite.ctx, hostnameStatus))

	u, err := url.Parse("https://foo:6443")
	suite.Require().NoError(err)

	cfg := config.NewMachineConfig(
		container.NewV1Alpha1(
			&v1alpha1.Config{
				ConfigVersion: "v1alpha1",
				MachineConfig: &v1alpha1.MachineConfig{
					MachineNetwork: &v1alpha1.NetworkConfig{},
				},
				ClusterConfig: &v1alpha1.ClusterConfig{
					ControlPlane: &v1alpha1.ControlPlaneConfig{
						Endpoint: &v1alpha1.Endpoint{
							URL: u,
						},
					},
				},
			},
		),
	)

	suite.Require().NoError(suite.state.Create(suite.ctx, cfg))

	suite.startRuntime()

	suite.assertResolvers(
		[]string{
			"default/resolvers",
		}, func(r *network.ResolverSpec, asrt *assert.Assertions) {
			asrt.Equal(
				[]netip.Addr{
					netip.MustParseAddr(constants.DefaultPrimaryResolver),
					netip.MustParseAddr(constants.DefaultSecondaryResolver),
				}, r.TypedSpec().DNSServers,
			)
			asrt.Equal([]string{"example.org"}, r.TypedSpec().SearchDomains)
			asrt.Equal(network.ConfigDefault, r.TypedSpec().ConfigLayer)
		},
	)

	// make domain name empty
	hostnameStatus.TypedSpec().Domainname = ""
	suite.Require().NoError(suite.state.Update(suite.ctx, hostnameStatus))

	suite.assertResolvers(
		[]string{
			"default/resolvers",
		}, func(r *network.ResolverSpec, asrt *assert.Assertions) {
			asrt.Empty(r.TypedSpec().SearchDomains)
		},
	)

	// bring back domain name, but disable via machine config
	hostnameStatus.TypedSpec().Domainname = "example.org"
	suite.Require().NoError(suite.state.Update(suite.ctx, hostnameStatus))

	cfg.Container().RawV1Alpha1().MachineConfig.MachineNetwork.NetworkDisableSearchDomain = pointer.To(true)
	suite.Require().NoError(suite.state.Update(suite.ctx, cfg))

	suite.assertResolvers(
		[]string{
			"default/resolvers",
		}, func(r *network.ResolverSpec, asrt *assert.Assertions) {
			asrt.Empty(r.TypedSpec().SearchDomains)
		},
	)
}

func (suite *ResolverConfigSuite) TestCmdline() {
	suite.Require().NoError(
		suite.runtime.RegisterController(
			&netctrl.ResolverConfigController{
				Cmdline: procfs.NewCmdline("ip=172.20.0.2:172.21.0.1:172.20.0.1:255.255.255.0:master1:eth1::10.0.0.1:10.0.0.2:10.0.0.1"),
			},
		),
	)

	suite.startRuntime()

	suite.assertResolvers(
		[]string{
			"cmdline/resolvers",
		}, func(r *network.ResolverSpec, asrt *assert.Assertions) {
			asrt.Equal(
				[]netip.Addr{
					netip.MustParseAddr("10.0.0.1"),
					netip.MustParseAddr("10.0.0.2"),
				}, r.TypedSpec().DNSServers,
			)
			asrt.Empty(r.TypedSpec().SearchDomains)
		},
	)
}

func (suite *ResolverConfigSuite) TestMachineConfiguration() {
	suite.Require().NoError(suite.runtime.RegisterController(&netctrl.ResolverConfigController{}))

	suite.startRuntime()

	u, err := url.Parse("https://foo:6443")
	suite.Require().NoError(err)

	cfg := config.NewMachineConfig(
		container.NewV1Alpha1(
			&v1alpha1.Config{
				ConfigVersion: "v1alpha1",
				MachineConfig: &v1alpha1.MachineConfig{
					MachineNetwork: &v1alpha1.NetworkConfig{
						NameServers: []string{"2.2.2.2", "3.3.3.3"},
						Searches:    []string{"example.com", "example.org"},
					},
				},
				ClusterConfig: &v1alpha1.ClusterConfig{
					ControlPlane: &v1alpha1.ControlPlaneConfig{
						Endpoint: &v1alpha1.Endpoint{
							URL: u,
						},
					},
				},
			},
		),
	)

	suite.Require().NoError(suite.state.Create(suite.ctx, cfg))

	suite.assertResolvers(
		[]string{
			"configuration/resolvers",
		}, func(r *network.ResolverSpec, asrt *assert.Assertions) {
			asrt.Equal(
				[]netip.Addr{
					netip.MustParseAddr("2.2.2.2"),
					netip.MustParseAddr("3.3.3.3"),
				}, r.TypedSpec().DNSServers,
			)

			asrt.Equal(
				[]string{"example.com", "example.org"},
				r.TypedSpec().SearchDomains,
			)
		},
	)

	ctest.UpdateWithConflicts(suite, cfg, func(r *config.MachineConfig) error {
		r.Container().RawV1Alpha1().MachineConfig.MachineNetwork.NameServers = nil
		r.Container().RawV1Alpha1().MachineConfig.MachineNetwork.Searches = nil

		return nil
	})

	suite.Assert().NoError(
		retry.Constant(3*time.Second, retry.WithUnits(100*time.Millisecond)).Retry(
			func() error {
				return suite.assertNoResolver("configuration/resolvers")
			},
		),
	)
}

func (suite *ResolverConfigSuite) TearDownTest() {
	suite.T().Log("tear down")

	suite.ctxCancel()

	suite.wg.Wait()
}

func TestResolverConfigSuite(t *testing.T) {
	suite.Run(t, new(ResolverConfigSuite))
}
