<?php
if(!function_exists('parse_padding')){
        function parse_padding($source)
        {
            $length  = strlen(strval(count($source['source']) + $source['first']));
            return 40 + ($length - 1) * 8;
        }
}

if(!function_exists('parse_class')){
    function parse_class($name)
    {
        $names = explode('\\', $name);
        return '<abbr title="'.$name.'">'.end($names).'</abbr>';
    }
}

if(!function_exists('parse_file')){
    function parse_file($file, $line)
    {
        return '<a class="toggle" title="'."{$file} line {$line}".'">'.basename($file)." line {$line}".'</a>';
    }
}

if(!function_exists('parse_args')){
    function parse_args($args)
    {
        $result = [];

        foreach ($args as $key => $item) {
            switch (true) {
                case is_object($item):
                    $value = sprintf('<em>object</em>(%s)', parse_class(get_class($item)));
                break;
                case is_array($item):
                    if(count($item) > 3){
                    $value = sprintf('[%s, ...]', parse_args(array_slice($item, 0, 3)));
                    } else {
                    $value = sprintf('[%s]', parse_args($item));
                    }
                break;
                case is_string($item):
                    if(strlen($item) > 20){
                    $value = sprintf(
                    '\'<a class="toggle" title="%s">%s...</a>\'',
                    htmlentities($item),
                    htmlentities(substr($item, 0, 20))
                    );
                    } else {
                    $value = sprintf("'%s'", htmlentities($item));
                    }
                break;
                case is_int($item):
                case is_float($item):
                    $value = $item;
                break;
                case is_null($item):
                    $value = '<em>null</em>';
                break;
                case is_bool($item):
                    $value = '<em>' . ($item ? 'true' : 'false') . '</em>';
                break;
                case is_resource($item):
                    $value = '<em>resource</em>';
                break;
                default:
                    $value = htmlentities(str_replace("\n", '', var_export(strval($item), true)));
                break;
            }

            $result[] = is_int($key) ? $value : "'{$key}' => {$value}";
        }

        return implode(', ', $result);
    }
}


?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>系统发生错误</title>
    <style>
        body,div,dl,dt,dd,ul,ol,li,h1,h2,h3,h4,h5,h6,pre,code,form,fieldset,legend,input,button,textarea,p,blockquote,th,td { margin:0; padding:0; box-sizing: border-box; -webkit-box-sizing: border-box; -moz-box-sizing: border-box; -ms-box-sizing: border-box;}
        body { background:#fff; color:#555; font-size:14px; font-family: "Microsoft Yahei"; }
        a { color:#555; text-decoration:none; }
        a:hover { text-decoration:none; }
        img { border:none; }
        .clearfix:after, .clearfix:before {content: ""; display: block; height:0; clear:both; visibility: hidden;}
        .clearfix { *zoom:1; }
        .errPage{width:800px;margin:0 auto;}
        .errPage .header{margin-top:160px;}
        .errPage .header .left{width:290px;height:290px;float:left;}
        .errPage .header .left img{width:100%;height:100%;}
        .errPage .header .right{width:395px;float:right;margin-top:15px;margin-right: 50px;}
        .errPage .header .right .picture{width:228px;height:80px;}
        .errPage .header .right .picture img{width:100%;height:100%;}
        .errPage .header .right .title{font-size:20px;color:#979DAB;margin-top:26px;height:70px;}
        .errPage .header .right .question{margin-top:40px;}
        .errPage .header .right .question .btn{-moz-user-select:none;-webkit-user-select:none;-ms-user-select:none;-khtml-user-select:none;user-select:none;display:inline-block;*display:inline;zoom:0;width:116px;height:28px;border-radius:18px;background-color:#999;font-size:12px;color:#fff;text-align:center;line-height:28px;cursor:pointer;}
        .errPage .header .right .question .btn.on{background-color:#1951FC;}
        .errPage .header .right .question .btn img{width:10px;height:6px;margin-left:10px;}
        .errPage .conter{width:800px;height:424px;padding:25px 29px;background-color:#F7F7F7;border-radius:18px;overflow:auto;font-size:14px;color:#666666;margin-top:55px;line-height:2.4;display:none;}
        .source-code pre {margin: 0;}
        .source-code pre li {height: 18px;line-height: 18px;}
        .source-code pre ol{margin: 0;color: #4288ce;display: inline-block;min-width: 100%;box-sizing: border-box;font-size:14px;font-family: "Century Gothic",Consolas,"Liberation Mono",Courier,Verdana;padding-left: <?php echo (isset($source) && !empty($source)) ? parse_padding($source) : 40;  ?>px;}
        .line-error{background: #f8cbcb;}
        /* SPAN elements with the classes below are added by prettyprint. */
        pre.prettyprint .pln { color: #000 }  /* plain text */
        pre.prettyprint .str { color: #080 }  /* string content */
        pre.prettyprint .kwd { color: #008 }  /* a keyword */
        pre.prettyprint .com { color: #800 }  /* a comment */
        pre.prettyprint .typ { color: #606 }  /* a type name */
        pre.prettyprint .lit { color: #066 }  /* a literal value */
        /* punctuation, lisp open bracket, lisp close bracket */
        pre.prettyprint .pun, pre.prettyprint .opn, pre.prettyprint .clo { color: #660 }
        pre.prettyprint .tag { color: #008 }  /* a markup tag name */
        pre.prettyprint .atn { color: #606 }  /* a markup attribute name */
        pre.prettyprint .atv { color: #080 }  /* a markup attribute value */
        pre.prettyprint .dec, pre.prettyprint .var { color: #606 }  /* a declaration; a variable name */
        pre.prettyprint .fun { color: red }  /* a function name */
        h2 {color: #4288ce;font-weight: 400;padding: 6px 0;margin: 6px 0 0;font-size: 14px;border-bottom: 1px solid #eee;}
        abbr {cursor: help;text-decoration: underline;text-decoration-style: dotted;}
        h1 {margin: 10px 0 0;font-size: 28px;font-weight: 500;line-height: 32px;}
        .source-code::-webkit-scrollbar{width:10px;height:10px;}
        .source-code::-webkit-scrollbar-track{border-radius:2px;}
        .source-code::-webkit-scrollbar-thumb{background: #bfbfbf;border-radius:10px;}
        .source-code::-webkit-scrollbar-thumb:hover{background: #333;}
    </style>
</head>
<body>
<div class="errPage">
    <div class="header">
        <div class="left"><img src="data:image/png;base64,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"></div>
        <div class="right">
            <div class="picture"><img src="data:image/png;base64,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"></div>
            <div class="title">
                <div class="info">
                    <div>
                        <h2><?php echo nl2br(htmlentities($message)); ?></h2>
                    </div>
                    <div>
                        <h1 style="font-size: 14px;">[<?php echo $code; ?>]&nbsp;<?php echo isset($name) ? sprintf('%s in %s', parse_class($name), parse_file($file, $line)) : $exception->getMessage(); ?></h1>
                    </div>
                </div>
            </div>
            <div class="question">
                <div class="btn">查看错误代码<img src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAwCAYAAABXAvmHAAABtElEQVRoQ+2Y3SoFURiGv+dmHHABTt2AM1FSkkRKUlIi2SmpnRJKJElJbWduwCGX4U6Wds3sttkz3/x8a+2xa83hNGvN87zvrFnTIBN+MOH8EgXabjA2EBswJhAfIWOA5uGxAXOExgliA0UBOuemReRYRL6AG2PQhcODNJDA90RkJrnzO7AYQsK7QA58yh1EwquAAh9MwptABfggEl4EFPiOiCwMrQXvEmYBBf4MOAm9JkwCCtw5cJTGHVKisYACdQEcZl+ZoSQaCSgwXeCgZHMb3h/Ma6K2gAJ/CeyXbVa+m6gloNz8Ctgrgw+xJioLKPDXwG5VeN8SlQQU+Ftgpy68T4lSAQX+DthuCu9LQhVQ4O+BLSu8D4lCAQX+EdjwBW+VyBVQ4J+Add/wFokRAQX+GVgLBd9U4o+AAv8CrIaGbyKRFfgWkdkM6CuwMi74ChIPwGZ63UDAOTcnIp8Z0DdgedzwZRLAgDvbwIeIzCcT9ICltuAViQ5wOtJA/4RzbkpE+p8FP0C3bfiMRO4vmtKd+L9IFHFEgbYbig3EBowJxEfIGKB5eGzAHKFxgtiAMUDz8F/6UtIxXjMNGgAAAABJRU5ErkJggg=="></div>
                <div class="btn on"><a style="color: #ffffff" href="javascript:history.back(-1)">返回上一页</a></div>
            </div>
        </div>
    </div>
    <div class="clearfix"></div>
    <div class="conter source-code">
        <?php if(!empty($source)){?>
        <div class="source-code">
            <pre class="prettyprint lang-php"><ol start="<?php echo $source['first']; ?>"><?php foreach ((array) $source['source'] as $key => $value) { ?><li class="line-<?php echo $key + $source['first']; ?>"><code><?php echo htmlentities($value); ?></code></li><?php } ?></ol></pre>
        </div>
        <?php }?>
    </div>
</div>
<script src="https://cdn.bootcss.com/jquery/3.4.1/jquery.min.js"></script>
<script>

    var LINE =<?php echo isset($line) ? $line : 0; ?>;
    var down = "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAwCAYAAABXAvmHAAABuElEQVRoQ+2Y4SoEURiGn+8mXIkbcAuKkpKSlFBSkhIpSUmhLUlJbYqUW3ADfvLPVbiAV1uztYY1Z+Z8x1id+bez55x9nvc9u81ZY8QvG3F+skDbDeYGcgORCeQtFBlg9PTcQHSEkQv8vwYk7QJjQMfMXiMDcpsu6Rh4B7pm9tZf+FMDBfxe8eYLMP0XJCTd9VgKrkczmxwmoFJkrUtIugVmSlwTZvbUu1du4AxY/SsSkrrAbInn2czGv22gd1PSPTDVtoSkG2CuiuPbX6G2JSRdA/NV8F+20OCEtiQkXQELIfA/CrSxnSRdAouh8JUCvykh6QJYqgMfJPAbEpI6wHJd+GCBlBKSzoGVJvC1BFJISDoF1prC1xbwlJB0AqzHwDcS8JAoHsw2YuEbC8RISDoCNj3gowSaSEg6BLa84KMF6khIOgC2PeFdBEIkJO0DO97wbgIVEg9A75Q3eLmdM1zPxEMeAEvsuMG7NtCnrJBwhU8iEPKdKFcS89p1Cw2ClJpwT77/WckEiiaS/0WTVCBma4TOzQKhSaUalxtIlWzourmB0KRSjcsNpEo2dN3cQGhSqcaNfAMfxuz5MbcQ4poAAAAASUVORK5CYII=";
    var up = "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADAAAAAwCAYAAABXAvmHAAABtElEQVRoQ+2Y3SoFURiGv+dmHHABTt2AM1FSkkRKUlIi2SmpnRJKJElJbWduwCGX4U6Wds3sttkz3/x8a+2xa83hNGvN87zvrFnTIBN+MOH8EgXabjA2EBswJhAfIWOA5uGxAXOExgliA0UBOuemReRYRL6AG2PQhcODNJDA90RkJrnzO7AYQsK7QA58yh1EwquAAh9MwptABfggEl4EFPiOiCwMrQXvEmYBBf4MOAm9JkwCCtw5cJTGHVKisYACdQEcZl+ZoSQaCSgwXeCgZHMb3h/Ma6K2gAJ/CeyXbVa+m6gloNz8Ctgrgw+xJioLKPDXwG5VeN8SlQQU+Ftgpy68T4lSAQX+DthuCu9LQhVQ4O+BLSu8D4lCAQX+EdjwBW+VyBVQ4J+Add/wFokRAQX+GVgLBd9U4o+AAv8CrIaGbyKRFfgWkdkM6CuwMi74ChIPwGZ63UDAOTcnIp8Z0DdgedzwZRLAgDvbwIeIzCcT9ICltuAViQ5wOtJA/4RzbkpE+p8FP0C3bfiMRO4vmtKd+L9IFHFEgbYbig3EBowJxEfIGKB5eGzAHKFxgtiAMUDz8F/6UtIxXjMNGgAAAABJRU5ErkJggg==";
    $(document).ready(function(){

        $.getScript = function(src, func){
            var script = document.createElement('script');
            
            script.async  = 'async';
            script.src    = src;
            script.onload = func || function(){};
            
            $('head')[0].appendChild(script);
        }

        var k = true;
        $(".question .btn").on('click',function(){
            if(k){
                $('.conter').show();
                $(this).addClass('on');
                $(this).find('img').attr('src',down);
                k = false;
            }else {
                $('.conter').hide();
                $(this).removeClass('on');
                $(this).find('img').attr('src',up);
                k = true;
            }
       });
        var ol    = $('ol', $('.prettyprint')[0]);
        // 设置出错行
        var err_line = $('.line-' + LINE, ol[0])[0];
        if(err_line) err_line.className = err_line.className + ' line-error';
        $.getScript('//cdn.bootcss.com/prettify/r298/prettify.min.js', function(){
            prettyPrint();

            // 解决Firefox浏览器一个很诡异的问题
            // 当代码高亮后，ol的行号莫名其妙的错位
            // 但是只要刷新li里面的html重新渲染就没有问题了
            if(window.navigator.userAgent.indexOf('Firefox') >= 0){
                ol[0].innerHTML = ol[0].innerHTML;
            }
        });
    });
</script>
</body>
</html>