# Coding Style

General Go coding style, this should be followed by people developing gommon and [other Go projects
under dyweb](https://github.com/dyweb?utf8=%E2%9C%93&q=&type=&language=go)

## TODO

- package
  - [ ] `types` package, to export API and reduce import cycle, a package the defines struct is really needed

## Reference

- https://github.com/golang/go/wiki/CodeReviewComments
- https://golang.org/doc/effective_go.html

## Package

Based on [Go Blog: Package names][Go: Package names]

Use small packages instead of a single flat package

- a large package can have many import and is easier to cause import cycle if it is a library
- it's easy to merge smaller packages into large one but hard in the other direction (TODO: refer hashicorp blog)

````text
// bad
agent.go
raft.go
http.go
grpc.go

// good
agent
  server.go
raft
  leader.go
  replication.go
  store.go
transport
  http.go
  grpc.go
````

Package name should be same as folder name

- avoid `go-mylib`, `go.mylib` as end of import path for `mylib` (we know it's go code ...)

````go
// bad
// file src/github.com/dyweb/gommon/go-errors/wrapper.go
package errors

// good
// file src/github.com/dyweb/gommon/errors/wrapper.go
package errors
````

Use `alllowercase` no `camelCase` or `snake_case`

- should try to use a single word in first place []
- `camelCase` cause trouble for some filesystems, i.e. windows users (even worse when they mount it into a linux vm)
- `snake_case` normally have special meaning in go, i.e. `foo_test.go` (it's test file), `fs_windows.go` (only build for windows platform)
- this `hardtoreadname` also distinguish it from other identifiers like `varName` `funcName`

````text
// good
termutil

// bad
termUtil
term_util
````

Rename your package if most users have to rename it when import

- when user rename the import and copy paste the code to another file, editors can't fill in the right import
- people can have many different renames even inside a single project `httpUtil`, `httpUtility`, it's better just give people a name that don't conflict and can use it everywhere

````go
import (
	"net/http"
	// good
	"github.com/dyweb/gommon/util/httputil"
	// bad
	httputil "github.com/dyweb/gommon/util/http"
	// even worse
	httputil "github.com/dyweb/gommon/util/go-http.v2"
)
````

Use `util.go` instead of `util` package

- a lot of times, those `util` are only used once in current package
- you will have a hard time rename all the `util` packages
- if there are too many things that can't fit into `util.go`, better make it a dedicated package with another name

````text
// good
pkg
   agent
     util.go

// bad, a single util.go in the util package
pkg
   agent
     util
       util.go
````

## File

At start of the file, briefly mention what the file does, it should also include some tricky internal implementation 
and warning for new developers.

````go
// good example for good code
// controller.go is the glue for reconciliation logic, it is generated by operator-sdk, 
// the core bossiness logic are implemented in placement.go and operates on object directly so they can be unit tested
````

````go
// good example for bad code
// fsm.go implements a state machine using nested for loop and if else with unhandled errors, 
// it is called fsm because it imports a fsm library
//
// If you tried to refactor the code and got stopped in the middle +1 for the following error
// 25
````

Put definitions on the top

- constant, package level vars (sentiel errors)
- exported interface
- factory functions
- exported struct

````go
// bad: sort the following content randomly while still keeps it compile

// good
const DefaltSuccessRatio = 0.8

var ErrOutOfResource = errors.New("out of resource")

type Executor interface {
	Run(ctx contex.Context, task Task) error
}

func NewShellExecutor(cmd string) (Executor, error) {
	return &ShellExecutor{cmd: cmd}, nil
} 

type Task struct {
	spec Spec
	retry int
}

type ShellExecutor struct {
	cmd string
}
````

Group implementations together
 
- if a interface have multiple implementations, put them close to each other, but **do NOT intersect the methods from
  different structs**, this only makes copy and paste easier

````go
// good

func (s *ShellExecutor) Name() string {
	return "shell"
}

func (s *ShellExecutor) Run(ctx context.Context, task Task) error {
	cmd := exec.Command(s.cmd)
	return cmd.Run()
}

func (d *DockerExecutor) Name() string {
	return "docker"
}

func (d *DockerExecutor) Run(ctx context.Context, task Task) error  {
    // create docker client, exec inside a container etc.
}

// bad
func (s *ShellExecutor) Name() string {
	return "shell"
}

func (d *DockerExecutor) Name() string {
	return "docker"
}

func (s *ShellExecutor) Run(ctx context.Context, task Task) error {
	cmd := exec.Command(s.cmd)
	return cmd.Run()
}

func (d *DockerExecutor) Run(ctx context.Context, task Task) error  {
    // create docker client, exec inside a container etc.
}
````
 
Keep file small
  
- if a file have many struct and many methods, you should group them by functionality and split into smaller files
- if a struct has too many/lengthy methods in one file, you should consider decouple the struct

## Struct

Use struct literal initialization
  
- you type less and rename variable is easier when there is no editor's help

````go
// good
task := Task{
	Id: RandomID(),
	Parent: Ppid,
	Name: "clean up"
}
executeTask(task)

// bad
task := Task{}
task.Id = RandomID()
task.Parent = Ppid
task.Name = "clean up"

// even worse
task := &Task{Id: RandomID()} // don't use pointer when initialize struct
task.Name = "clean up" // for fields that can be assigned directly, put them in struct literal
executeTask(*task) // don't create a pointer and de reference it
````

- do NOT use embedding for struct that will be serialized (in JSON etc.)
- prefer function over struct methods
  - TODO: explain this ...

## Func

- when pass/return a pointer, use `&` at call site, NOT initialization
  - pointer means ownership
  - pointer does not means no copy, sometimes it causes allocation on heap which can be avoided
  - [ ] TODO: refer to Bill's blog and course

## Variable

Use short names for common operations

````go
// good
b, err := json.Marshal(Task)
w.Write(b)

// bad
encodedTask, err := json.Marhsal(Task)
responseWriter.Write(encodedTask)
````

## Error handling

Return early to avoid nesting 

````go
// good
if execErr != nil {
	return execErr
}
if cErr := container.Err(); cErr != nil {
	return cErr
}
postExecHook(task)


// bad
if execErr != nil {
	return execErr
} else {
	if cErr := container.Err(); cErr != nil {
		return cErr
    } else {
    	postExecHook(task)
    }
}
````

- [ ] if there are many checks at the start of a function, might put validation in a new function

## Test

- use `packagename_test` instead of `packagename` for test files, unless you need to test specific internal implementation
- use helper function that accepts `testing.T` and don't return error
  - your code should handle error gracefully, your test should stop when there is error/unexpected result
  - this may cause error source line harder to find since go's test fail function does allow pass skip
  - [ ] TODO: might open a issue in go's repo, or there are already issues like that
- use [subtest][Go: Subtest and Sub-benchmarks]
  - this allows setup and teardown using vanilla go code without any framework
- use `TestMain` for package level setup and tear down
- [ ] TODO: add golden file, add refer to hashicorp test video
- when assert error thant will cause panic for following code use `require` instead of `assert`
- define anonymous struct for table driven test 

## Ref

[Go: Package names]: https://blog.golang.org/package-names
[Go: Subtest and Sub-benchmarks]: https://blog.golang.org/subtests