# 2018-12-16 Design

This doc is the design based on the [motivation for error categorization](2018-12-10-error-categorization.md)
and [reference](2018-12-14-reference.md). It defines what will be covered in gommon v0.0.9 for gommon/errors.

Follow the go 2 design, we have following three problems

- error handling, too many `if err != nil`
- [error formatting](https://go.googlesource.com/proposal/+/master/design/go2draft-error-printing.md)
- [error inspection](https://go.googlesource.com/proposal/+/master/design/go2draft-error-inspection.md)

The solutions are

- can't solve the error handling, only we have is multi error, which is useful for validation
- since we already have stack trace, we should print it properly, use format properly
  - we can also store the wrapping call site, though this should be optional, NOTE: the wrapping call site is not same as stack trace
- `Is` with unwrap, `As` need more thought, include `Walk` and other util function from [hashicorp/errorswrap](https://github.com/hashicorp/errwrap/blob/master/errwrap.go)
  - contains wrapper/inspector for common error from standard library, see `ConvertSystemError` in https://github.com/gravitational/trace/blob/master/errors.go
  - give high level category for errors

Use cases

- a docker client that retry when the error is network error
- a api server that returns validation error after calling some other API
- a cli that tells user the config file looking is not found and there is a similar file

What should be in `errors`, what should be in `errorx`

- `errors` is for generating, wrapping, unwrapping, inspecting error, basic formatting
- `errorx` is for advanced formatting, provide solution for common error (i.e. if it's error not found, what's the dir, what is being looking for)

## Error inspection

- `Is` in go 2 does follow the chain and do exact match
- `As` do type match and return the error

For unwrapping we need to handle two things, both wrapping and multi error

- support both `UnWrap` and `Causer` but deprecate `Causer` interface and make it not exported
- support `MutliError` when unwrap, flatten multi error unless the target error type is MultiError

I think for gommon, consider upspin, hashicorp/errorswrap we can have the following new methods

````go
// Is is sames as go 2 proposal, it is manly used for checking sentinel error 
// It does unwrapping under the hood
func Is(err, target error) bool {
	for {
		if err == target {
			return true
		}
		// TODO: check both Causer and Wrapper interface
		// TODO: also consider multi error, might use a walk function
		wrapper, ok := err.(Wrapper)
		if !ok {
			return false
		}
		err = wrapper.Unwrap()
		if err == nil {
			return false
		}
	}
}

// IsType checks if err's type is same as targets using reflect
func IsType(err, target error) bool {
	ttype := reflect.TypeOf(target).String()
	for {
		if reflect.TypeOf(err).String() == ttype {
			return true
		}
	}
}


// TODO: GetType or GetTypeOf ... maybe ~~MatchType~~ (IsType already does this) 
func GetTypeOf(err, target error) (error, bool) {
	
}

// TODO: this is really hard to use without ploymorphism when you want to use switch case ...
// I think IsType is a better choice for now
func As(ptrToError interface{}, err error) (ok bool) {
	
}

type WalkFunc func(error) bool // TODO: a stop sign or continue sign, this can reduce some looping

func Walk(err error, cb WalkFunc) {
	switch e := err.(type) {
	case *WrappedError: // TODO: make our implementation not exported, user shouldn't use that
	    if cb(e) {
    	    Walk(e.cause, cb)
	    }
	 case Wrapper:
	    if cb(e) {
    	    cb(e.Unwrap())
	    }
	 case MultiError:
	 	errs := e.Errors()
	 	for i := 0; i < len(errs); i++ {
	 		// TODO: have return value from WalkFunc makes Walk also need to return a bool
	 	}
    default:
    	cb(e)
	}
}
````

