# Log v2

Design Goals

- easy to maintain, use standard library as much as possible
- acceptable and easy to use by default, increase performance if really needed
- testable, should not rely on human eyeball to see if the log is correct
- configurable, if a library use gommon/log, then the application using the library can config the logging of that library like the library itself
- out of box, default config struct, http handler (unix sock?), print log entry relations
- not using package level variable, especially pools

Non Goals

- sampling

## Survey

## log

https://golang.org/src/log/log.go

````go
func (l *Logger) Printf(format string, v ...interface{}) {
  	l.Output(2, fmt.Sprintf(format, v...))
}

// Output writes the output for a logging event. The string s contains
// the text to print after the prefix specified by the flags of the
// Logger. A newline is appended if the last character of s is not
// already a newline. Calldepth is used to recover the PC and is
// provided for generality, although at the moment on all pre-defined
// paths it will be 2.
func (l *Logger) Output(calldepth int, s string) error {
  	// Get time early if we need it.
  	var now time.Time
  	if l.flag&(Ldate|Ltime|Lmicroseconds) != 0 {
  		now = time.Now()
  	}
  	var file string
  	var line int
  	l.mu.Lock()
  	defer l.mu.Unlock()
  	if l.flag&(Lshortfile|Llongfile) != 0 {
  		// Release lock while getting caller info - it's expensive.
  		l.mu.Unlock()
  		var ok bool
  		_, file, line, ok = runtime.Caller(calldepth)
  		if !ok {
  			file = "???"
  			line = 0
  		}
  		l.mu.Lock()
  	}
  	l.buf = l.buf[:0]
  	l.formatHeader(&l.buf, now, file, line)
  	l.buf = append(l.buf, s...)
  	if len(s) == 0 || s[len(s)-1] != '\n' {
  		l.buf = append(l.buf, '\n')
  	}
  	_, err := l.out.Write(l.buf)
  	return err
}
````

### Logrus

https://github.com/sirupsen/logrus

The first version is entirely modeled after logrus
 
common

- not use pointer receiver on Entry struct to avoid race condition
- using formatter + writer 
  - introduced extra copy from `fmt.Sprintf`, it is possible to put data into writer directly
  - [ ] TODO: will there be race condition causing different log mixed with each other, like `[info][warn]msg1msg2` instead of `[info]msg1 [warn]msg2`
  
differences

- no pool for log entry
- no `*ln` function, just `*` and `*f` `*` means Debug, Info ...

### apex/log

- use handler instead of formatter + writer

````go
// The HandlerFunc type is an adapter to allow the use of ordinary functions as
// log handlers. If f is a function with the appropriate signature,
// HandlerFunc(f) is a Handler object that calls f.
type HandlerFunc func(*Entry) error

// HandleLog calls f(e).
func (f HandlerFunc) HandleLog(e *Entry) error {
	return f(e)
}
````

I guess it's modeled after `net/http`

````go
// The HandlerFunc type is an adapter to allow the use of
// ordinary functions as HTTP handlers. If f is a function
// with the appropriate signature, HandlerFunc(f) is a
// Handler that calls f.
type HandlerFunc func(ResponseWriter, *Request)

// ServeHTTP calls f(w, r).
func (f HandlerFunc) ServeHTTP(w ResponseWriter, r *Request) {
f(w, r)
}
````