/*
Copyright 2025 The Dapr Authors
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package scheduler

import (
	"os"

	"github.com/spf13/cobra"

	"github.com/dapr/cli/pkg/print"
	"github.com/dapr/cli/pkg/scheduler"
	"github.com/dapr/kit/signals"
)

var (
	schedulerImportFile string
)

var SchedulerImportCmd = &cobra.Command{
	Use:   "import",
	Short: "Import all jobs and actor reminders from a binary file generated by 'dapr scheduler export'.",
	Long: `Import jobs and actor reminders to Scheduler from a binary file generated by 'dapr scheduler export'.
dapr scheduler import -f export.bin`,
	Args: cobra.NoArgs,
	RunE: func(cmd *cobra.Command, args []string) error {
		ctx := signals.Context()

		err := scheduler.Import(ctx, scheduler.ExportImportOptions{
			SchedulerNamespace: schedulerNamespace,
			KubernetesMode:     kubernetesMode,
			TargetFile:         schedulerImportFile,
		})
		if err != nil {
			return err
		}

		print.InfoStatusEvent(os.Stdout, "Import from '%s' complete.", schedulerImportFile)

		return nil
	},
}

func init() {
	SchedulerImportCmd.Flags().MarkHidden("namespace")
	SchedulerImportCmd.Flags().StringVarP(&schedulerImportFile, "input-file", "f", "", "Input file to import jobs and actor reminders from.")
	SchedulerImportCmd.MarkFlagRequired("input-file")
	SchedulerImportCmd.MarkFlagFilename("input-file")
	SchedulerCmd.AddCommand(SchedulerImportCmd)
}
