/*
Copyright 2025 The Dapr Authors
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package scheduler

import (
	"os"

	"github.com/gocarina/gocsv"
	"github.com/spf13/cobra"

	"github.com/dapr/cli/pkg/print"
	"github.com/dapr/cli/pkg/scheduler"
	"github.com/dapr/cli/utils"
	"github.com/dapr/kit/ptr"
	"github.com/dapr/kit/signals"
)

var (
	listFilterType   *string
	listOutputFormat *string
)

var ListCmd = &cobra.Command{
	Use:   "list",
	Short: "List scheduled jobs in Scheduler.",
	Args:  cobra.NoArgs,
	RunE: func(cmd *cobra.Command, args []string) error {
		ctx := signals.Context()

		opts := scheduler.ListOptions{
			SchedulerNamespace: schedulerNamespace,
			KubernetesMode:     kubernetesMode,
			Filter: scheduler.Filter{
				Type: *listFilterType,
			},
		}
		opts.Filter.Namespace = ptr.Of(daprNamespace)

		var list any
		var empty bool
		switch *listOutputFormat {
		case outputFormatShort:
			ll, err := scheduler.List(ctx, opts)
			if err != nil {
				return err
			}
			empty = len(ll) == 0
			list = ll
		default:
			ll, err := scheduler.ListWide(ctx, opts)
			if err != nil {
				return err
			}
			empty = len(ll) == 0
			list = ll
		}

		if empty {
			print.FailureStatusEvent(os.Stderr, "No jobs found in namespace %q", daprNamespace)
			return nil
		}

		var err error
		switch *listOutputFormat {
		case outputFormatYAML:
			err = utils.PrintDetail(os.Stdout, "yaml", list)
		case outputFormatJSON:
			err = utils.PrintDetail(os.Stdout, "json", list)
		default:
			var table string
			table, err = gocsv.MarshalString(list)
			if err != nil {
				break
			}

			utils.PrintTable(table)
		}

		if err != nil {
			return err
		}

		return nil
	},
}

func init() {
	listOutputFormat = outputFunc(ListCmd)
	listFilterType = filterFunc(ListCmd)
	SchedulerCmd.AddCommand(ListCmd)
}
