/*
 *   Copyright (c) 2021 Dario Lucia (https://www.dariolucia.eu)
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and 
 *   limitations under the License.
 */

package eu.dariolucia.ccsds.cfdp.entity.indication;

import eu.dariolucia.ccsds.cfdp.entity.CfdpTransactionStatus;
import eu.dariolucia.ccsds.cfdp.protocol.pdu.ConditionCode;
import eu.dariolucia.ccsds.cfdp.protocol.pdu.FinishedPdu;
import eu.dariolucia.ccsds.cfdp.protocol.pdu.tlvs.FilestoreResponseTLV;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * The Transaction-Finished.indication primitive shall be used to indicate to the source
 * or destination CFDP user that the transaction is complete and that the source CFDP user is
 * authorized to modify or delete any retransmission buffer (file) that it was sharing with the
 * protocol entity to conserve persistent storage space.
 *
 * Transaction-Finished.indication is generated on Notice of Completion of a file
 * transmission procedure either at the source CFDP entity or, optionally, at the destination
 * CFDP entity.
 *
 * The effect on receipt of Transaction-Finished.indication by an application is
 * undefined.
 *
 * Transaction-Finished.indication is always generated by the source CFDP entity and
 * can also be generated by the destination CFDP entity. The condition code indicates the
 * condition under which the transaction was finished.
 *
 * Ref. CCSDS 727.0-B-5, 3.4.8
 */
public class TransactionFinishedIndication implements ICfdpTransactionIndication {

    private final long transactionId;

    private final ConditionCode conditionCode;

    private final FinishedPdu.FileStatus fileStatus;

    private final boolean dataComplete;

    private final List<FilestoreResponseTLV> filestoreResponses = new LinkedList<>();

    private final CfdpTransactionStatus statusReport;

    /**
     * TransactionFinishedIndication full constructor.
     *
     * @param transactionId The Transaction ID parameter shall uniquely identify a single instance of FDU delivery
     * @param conditionCode The Condition code parameter shall provide additional information on some change in the transaction status
     * @param fileStatus The file status
     * @param dataComplete True if the data is complete, otherwise false
     * @param filestoreResponses Can be null. The filestore responses if present
     * @param statusReport Can be null. Implementation specific
     */
    public TransactionFinishedIndication(long transactionId, ConditionCode conditionCode, FinishedPdu.FileStatus fileStatus, boolean dataComplete, List<FilestoreResponseTLV> filestoreResponses, CfdpTransactionStatus statusReport) {
        this.transactionId = transactionId;
        this.conditionCode = conditionCode;
        this.fileStatus = fileStatus;
        this.dataComplete = dataComplete;
        this.statusReport = statusReport;
        if(filestoreResponses != null) {
            this.filestoreResponses.addAll(filestoreResponses);
        }
    }

    /**
     * The Transaction ID parameter shall uniquely identify a single instance of FDU
     * delivery and shall contain the ID of the source CFDP entity together with a sequence number
     * that is specific to that entity.
     *
     * At any moment, any given transaction ID is unique within the CFDP addressing
     * domain that encompasses the source CFDP entity.
     *
     * @return the transaction ID
     */
    @Override
    public long getTransactionId() {
        return transactionId;
    }

    /**
     * The Condition code parameter shall provide additional information on some change
     * in transaction status.
     *
     * @return the condition code
     */
    public ConditionCode getConditionCode() {
        return conditionCode;
    }

    /**
     * The file status.
     *
     * @return the file status
     */
    public FinishedPdu.FileStatus getFileStatus() {
        return fileStatus;
    }

    /**
     * In the CCSDS Blue Book, this parameter is defined as "delivery code" in the {@link FinishedPdu}.
     *
     * If true, means that metadata, all file data, and EOF have been received, and the
     * checksum has been verified.
     *
     * @return true if the data is complete (everything was received correctly)
     */
    public boolean isDataComplete() {
        return dataComplete;
    }

    /**
     * The filestore responses. Optionally present.
     *
     * @return the filestore responses (can be empty, never null)
     */
    public List<FilestoreResponseTLV> getFilestoreResponses() {
        return Collections.unmodifiableList(filestoreResponses);
    }

    /**
     * The Status report parameter shall indicate the status of the indicated file delivery
     * transaction. The format and scope of the status report parameter are specific to the
     * implementation. It could contain information such as:
     * <ol>
     *     <li>whether the transaction is finished, canceled, suspended, or active;</li>
     *     <li>what extents of the FDU are known to have been successfully received by the
     * receiving CFDP entity;</li>
     *     <li>what extents of the FDU are known to have been transmitted by the sending CFDP
     * entity.</li>
     * </ol>
     *
     * @return the status report
     */
    @Override
    public CfdpTransactionStatus getStatusReport() {
        return statusReport;
    }

    @Override
    public String toString() {
        return "TransactionFinishedIndication{" +
                "transactionId=" + getTransactionId() +
                ", conditionCode=" + getConditionCode() +
                ", fileStatus=" + getFileStatus() +
                ", dataComplete=" + isDataComplete() +
                ", filestoreResponses=" + filestoreResponses +
                ", statusReport=" + getStatusReport() +
                '}';
    }
}
