/*
 *   Copyright (c) 2021 Dario Lucia (https://www.dariolucia.eu)
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and 
 *   limitations under the License.
 */

package eu.dariolucia.ccsds.cfdp.entity.request;

/**
 * The Resume.request primitive shall be used to request that a suspended transaction be
 * resumed.
 *
 * Resume.request is generated by any CFDP user at any time during the lifetime of any
 * transaction, except in the case of a transaction sent in unacknowledged mode, for which it
 * can only be generated by the transaction’s source user.
 *
 * Receipt of Resume.request shall have no effect on any transaction that is not currently
 * suspended. For a transaction that is currently suspended, it shall cause the local CFDP entity
 * to initiate Resume procedures.
 *
 * User application developers may find it advisable to provide automated support for one
 * special case of transaction resumption, which applies to any transaction the user application
 * can determine is currently suspended solely due to transaction inactivity (i.e., as a result of
 * an Inactivity fault). On receipt of any service indication produced because of arrival of a
 * PDU for such a transaction, the user application might infer that the transaction’s inactivity is
 * ended and thereupon automatically resume the transaction.
 *
 * Ref. CCSDS 727.0-B-5, 3.4.4
 */
public class ResumeRequest implements ICfdpRequest {

    private final long transactionId;

    /**
     * ResumeRequest full constructor.
     *
     * @param transactionId the transaction ID that uniquely identify a single instance of FDU delivery
     */
    public ResumeRequest(long transactionId) {
        this.transactionId = transactionId;
    }

    /**
     * The Transaction ID parameter shall uniquely identify a single instance of FDU
     * delivery and shall contain the ID of the source CFDP entity together with a sequence number
     * that is specific to that entity.
     *
     * At any moment, any given transaction ID is unique within the CFDP addressing
     * domain that encompasses the source CFDP entity.
     *
     * @return the transaction ID
     */
    public long getTransactionId() {
        return transactionId;
    }

    @Override
    public String toString() {
        return "ResumeRequest{" +
                "transactionId=" + transactionId +
                '}';
    }
}
