/*
 *  Copyright 2018-2019 Dario Lucia (https://www.dariolucia.eu)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package eu.dariolucia.ccsds.sle.utl.util;

import java.util.Date;

/**
 * This class is used to generate statistical samples ({@link DataRateSample}. Acquisition of the samples is driven
 * by an external clock. A sample is generated by calling the sample() method. Transmitted and received units must be
 * continuously added using the related methods addOut(long) and addIn(long) respectively.
 */
public class DataRateCalculator {

	private Date lastSamplingTime;
	private long totalInUnitsSinceReset;
	private long totalOutUnitsSinceReset;
	
	private long tempInUnits;
	private long tempOutUnits;

	/**
	 * This method resets the internal state of the calculator.
	 */
	public synchronized void reset() {
		this.lastSamplingTime = new Date();
		this.totalInUnitsSinceReset = 0;
		this.totalOutUnitsSinceReset = 0;
		this.tempInUnits = 0;
		this.tempOutUnits = 0;
	}

	/**
	 * This method adds the provided amount of incoming units to the internal accumulator variable.
	 * @param units the incoming units to add
	 */
	public synchronized void addIn(long units) {
		this.tempInUnits += units;
		this.totalInUnitsSinceReset += units;
	}

	/**
	 * This method adds the provided amount of outgoing units to the internal accumulator variable.
	 * @param units the outgoing units to add
	 */
	public synchronized void addOut(long units) {
		this.tempOutUnits += units;
		this.totalOutUnitsSinceReset += units;
	}

	/**
	 * This method computes the data rates, taking into account the amount of in/out units since last acquisition, and
	 * generates a new sample.
	 *
	 * @return the new sample
	 */
	public synchronized DataRateSample sample() {
		Date now = new Date();
		if(this.lastSamplingTime == null) {
			this.lastSamplingTime = now;
			return new DataRateSample(now, 0, 0, this.totalInUnitsSinceReset, this.totalOutUnitsSinceReset);
		} else {
			double inRate = ((double) this.tempInUnits / (double) (now.getTime() - this.lastSamplingTime.getTime())) * 1000;
			double outRate = ((double) this.tempOutUnits / (double) (now.getTime() - this.lastSamplingTime.getTime())) * 1000;
			this.tempInUnits = 0;
			this.tempOutUnits = 0;
			this.lastSamplingTime = now;
			return new DataRateSample(now, inRate, outRate, this.totalInUnitsSinceReset, this.totalOutUnitsSinceReset);
		}
	}
}
