/*
 * Copyright 2018-2019 Dario Lucia (https://www.dariolucia.eu)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package eu.dariolucia.ccsds.tmtc.datalink.channel.sender;

import eu.dariolucia.ccsds.tmtc.datalink.builder.ITransferFrameBuilder;
import eu.dariolucia.ccsds.tmtc.datalink.channel.VirtualChannelAccessMode;
import eu.dariolucia.ccsds.tmtc.datalink.pdu.AbstractTransferFrame;
import eu.dariolucia.ccsds.tmtc.transport.pdu.BitstreamData;
import eu.dariolucia.ccsds.tmtc.transport.pdu.IPacket;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * This class represents an abstraction of a virtual channel using for sending purposes. Typical use cases for this class
 * (and derivatives) are in the implementation of a telemetry stream generated by a spacecraft simulator, or for the
 * construction and sending of telecommands. When frames are emitted, they are sent to all the {@link IVirtualChannelSenderOutput} registered to the sender.
 *
 * This class supports two modes:
 * <ul>
 *     <li>Pull mode: in pull mode, an external entity requires the class to generate a frame. The class will then ask a {@link IVirtualChannelDataProvider} to
 *     return an amount of data (in the form configured when instantiating the class) to fill at most one frame.</li>
 *     <li>Push mode: in push mode, an external entity requires the class to encapsulate one or more space packets, or bit streams or user data. Transfer
 *     frames are generated and emitted as needed.</li>
 * </ul>
 *
 * This class keeps track of the virtual channel frame counter and of the amount of free space in each frame under construction.
 *
 * This class is not thread safe.
 *
 * @param <T> the type of transfer frame
 */
public abstract class AbstractSenderVirtualChannel<T extends AbstractTransferFrame> {

    private final List<IVirtualChannelSenderOutput> listeners = new CopyOnWriteArrayList<>();

    private final int spacecraftId;

    private final int virtualChannelId;

    private final VirtualChannelAccessMode mode;

    private final boolean fecfPresent;

    private final AtomicInteger virtualChannelFrameCounter = new AtomicInteger(0);

    private final IVirtualChannelDataProvider dataProvider;

    private final AtomicLong emittedFrames = new AtomicLong(0);

    protected ITransferFrameBuilder<T> currentFrame;

    /**
     * This constructor initialises the virtual channel sender in push mode.
     *
     * @param spacecraftId the spacecraft id
     * @param virtualChannelId the virtual channel id
     * @param mode the mode
     * @param fecfPresent true if the FECF is present, false otherwise
     */
    protected AbstractSenderVirtualChannel(int spacecraftId, int virtualChannelId, VirtualChannelAccessMode mode, boolean fecfPresent) {
        this(spacecraftId, virtualChannelId, mode, fecfPresent, null);
    }

    /**
     * This constructor initialises the virtual channel sender in pull mode (if the data provider is specified) or push
     * mode (if the data provider is null).
     *
     * @param spacecraftId the spacecraft id
     * @param virtualChannelId the virtual channel id
     * @param mode the mode
     * @param fecfPresent true if the FECF is present, false otherwise
     * @param dataProvider the data provider to use: if specified then pull mode is enabled.
     */
    protected AbstractSenderVirtualChannel(int spacecraftId, int virtualChannelId, VirtualChannelAccessMode mode, boolean fecfPresent, IVirtualChannelDataProvider dataProvider) {
        this.spacecraftId = spacecraftId;
        this.virtualChannelId = virtualChannelId;
        this.mode = mode;
        this.fecfPresent = fecfPresent;
        this.dataProvider = dataProvider;
    }

    /**
     * Depending on the frame size and access mode, request either space packets, bit stream or user data to
     * fill up a frame. It is responsibility of the data provider to deliver the data if available. The
     * post-condition of this method is to generate and emit at most one frame:
     * <ul>
     * <li>if the data provider delivers not enough data, the method returns false: the caller can decide what to do
     *   (i.e. generate an idle frame from another virtual channel, force the dispatch of the frame, generate an idle
     *   packet to fill up the frame);</li>
     * <li>if the data provider delivers too much data, the method returns true and a frame is emitted: the remaining
     *   data is used to fill up the next frame. If the data completes or exceeds also the next frame, an exception
     *   is thrown.</li>
     *</ul>
     * @return true if the frame is emitted, false otherwise (not enough data)
     */
    public boolean pullNextFrame() {
        if(dataProvider == null) {
            throw new IllegalStateException("Virtual channel not instantiated in pull mode");
        }
        // Calculate how much free space we have to fill up the frame, and the maximum amount of data that this
        // channel can handle to avoid violating the constraint on the generation of at most one frame
        int availableSpaceInCurrentFrame = getRemainingFreeSpace();
        if(mode == VirtualChannelAccessMode.PACKET || mode == VirtualChannelAccessMode.ENCAPSULATION) {
            List<IPacket> packets = this.dataProvider.generateSpacePackets(getVirtualChannelId(), availableSpaceInCurrentFrame, availableSpaceInCurrentFrame + getMaxUserDataLength() - 1);
            // Compute if a frame will be emitted or not
            int newDataSize = packets == null ? 0 : packets.stream().map(IPacket::getLength).reduce(0, Integer::sum);
            if(newDataSize >= availableSpaceInCurrentFrame + getMaxUserDataLength()) {
                // Two frames or more would be generated: error by the data provider
                throw new IllegalStateException("Virtual channel " + getVirtualChannelId() + " requested max " + (availableSpaceInCurrentFrame + getMaxUserDataLength() - 1) + " bytes to data provider " +
                        "but " + newDataSize + " bytes were received (as space packets), cannot process");
            } else {
                if(packets != null && !packets.isEmpty()) {
                    dispatch(packets);
                }
                // If the amount of received data is less than availableSpaceInCurrentFrame, then no frame was emitted
                // but the data has been stored for the next pull request: return false.
                // Received data is equal or more than availableSpaceInCurrentFrame, then one frame was emitted
                // and the remaining data has been stored for the next pull request: return true.
                return newDataSize >= availableSpaceInCurrentFrame;
            }
        } else if(mode == VirtualChannelAccessMode.BITSTREAM) {
            BitstreamData data = this.dataProvider.generateBitstreamData(getVirtualChannelId(), availableSpaceInCurrentFrame);
            // Compute if a frame will be emitted or not
            int newDataSize = data == null ? 0 : data.getNumBits()/8;
            if(data != null) {
                int reportByte = (data.getNumBits() % 8 == 0 ? 0 : 1);
                newDataSize += reportByte;
            }
            // For bitstream data, no segmentation is possible, so data must be less than or equal the requested amount
            if(newDataSize > availableSpaceInCurrentFrame) {
                // Constraint violation
                throw new IllegalStateException("Virtual channel " + getVirtualChannelId() + " requested max " + (availableSpaceInCurrentFrame) + " bytes to data provider " +
                        "but " + newDataSize + " bytes were received (as bitstream), cannot process");
            } else {
                int remainingData = 0;
                if(data != null) {
                    remainingData = dispatch(data);
                }
                // At this stage, a frame is emitted if the remainingData is equal to the user data of a full frame.
                // The above is true only if there was meaningful data.
                return newDataSize > 0 && remainingData == getMaxUserDataLength();
            }
        } else if(mode == VirtualChannelAccessMode.DATA) {
            byte[] data = this.dataProvider.generateData(getVirtualChannelId(), availableSpaceInCurrentFrame);
            // Compute if a frame will be emitted or not
            int newDataSize = data == null ? 0 : data.length;
            // For user data, no segmentation is possible, so data must be less than or equal the requested amount
            if(newDataSize > availableSpaceInCurrentFrame) {
                // Constraint violation
                throw new IllegalStateException("Virtual channel " + getVirtualChannelId() + " requested max " + (availableSpaceInCurrentFrame) + " bytes to data provider " +
                        "but " + newDataSize + " bytes were received (as user data), cannot process");
            } else {
                int remainingData = 0;
                if(data != null) {
                    remainingData = dispatch(data);
                }
                // At this stage, a frame is emitted if the remainingData is equal to the user data of a full frame.
                // The above is true only if there was meaningful data.
                return newDataSize > 0 && remainingData == getMaxUserDataLength();
            }
        } else {
            throw new IllegalStateException("Virtual channel access mode " + mode + " not supported, cannot generate frame for virtual channel " + getVirtualChannelId());
        }
    }

    /**
     * This method returns the spacecraft ID generated by this virtual channel.
     *
     * @return the spacecraft ID
     */
    public int getSpacecraftId() {
        return spacecraftId;
    }

    /**
     * This method returns the virtual channel ID linked to this object.
     *
     * @return the virtual channel ID
     */
    public int getVirtualChannelId() {
        return virtualChannelId;
    }

    /**
     * This method returns the mode (packet mode, bitstream mode, channel access mode, encapsulation) configured for this virtual channel.
     *
     * @return the virtual channel access mode
     */
    public VirtualChannelAccessMode getMode() {
        return mode;
    }

    /**
     * This method returns whether generated frames have the Frame Error Control Field.
     *
     * @return true if the FECF is added to the generated frames, false otherwise
     */
    public boolean isFecfPresent() {
        return fecfPresent;
    }

    /**
     * This method returns the next virtual channel frame counter value that will be set into the next generated
     * transfer frame.
     *
     * @return the next virtual channel frame counter
     */
    public int getNextVirtualChannelFrameCounter() {
        return this.virtualChannelFrameCounter.get();
    }

    /**
     * This method allows to set the next virtual channel frame counter value.
     *
     * @param number the value of the virtual channel frame counter to be set into the next generated transfer frame
     */
    public void setVirtualChannelFrameCounter(int number) {
        this.virtualChannelFrameCounter.set(number);
    }

    protected int incrementVirtualChannelFrameCounter(int modulus) {
        int toReturn = this.virtualChannelFrameCounter.get() % modulus;
        if (toReturn == 0) {
            this.virtualChannelFrameCounter.set(1);
        } else {
            this.virtualChannelFrameCounter.set(toReturn + 1);
        }
        return toReturn;
    }

    /**
     * This method registers a listener to this virtual channel. If the listener is already registered, then it is
     * registered twice.
     *
     * @param listener the listener to be registered
     */
    public final void register(IVirtualChannelSenderOutput listener) {
        this.listeners.add(listener);
    }

    /**
     * This method deregisters a listener to this virtual channel. If the listener was not registered, nothing happens.
     *
     * @param listener the listener to be deregistered
     */
    public final void deregister(IVirtualChannelSenderOutput listener) {
        this.listeners.remove(listener);
    }

    protected final void notifyTransferFrameGenerated(T frame, int currentBufferedData) {
        // Increase emission
        this.emittedFrames.incrementAndGet();
        // Notify listeners
        this.listeners.forEach(o -> o.transferFrameGenerated(this, frame, currentBufferedData));
    }

    protected int calculateRemainingData(List<IPacket> packets, int i) {
        int bytes = 0;
        for (; i < packets.size(); ++i) {
            bytes += packets.get(i).getLength();
        }
        return bytes;
    }

    /**
     * This method requests the generation of one or more transfer frames, which contain the provided
     * packets. The method returns the amount of free bytes that are still available in the last generated but not emitted
     * frame.
     *
     * For TM-based VCs, the VC will wait to receive a sufficient amount of packets to emit its last frame, i.e. frames
     * are emitted once they are full. Depending on the amount and size of packets, this method can result in zero, one or
     * multiple frames being emitted.
     *
     * For TC-based VCs, the VC will always emit at least one frame. The VC will try to pack TC packets inside a single
     * frame and will avoid segmentation if possible. For packets larger than the maximum frame size, the VC ensures
     * that the large space packets is segmented across frames. Subsequent packets will use different frames.
     *
     * @param packets the packets to be encapsulated inside transfer frames
     * @return the amount of free bytes that are still available in the last generated but not emitted frame, or a value equal to getMaxUserDataLength if there is no pending frame
     */
    public abstract int dispatch(Collection<IPacket> packets);

    /**
     * This method calls dispatch(Collections.singletonList(isp)).
     *
     * @param isp the packet to dispatch
     *
     * @return the amount of free bytes that are still available in the last generated but not emitted frame, or a value equal to getMaxUserDataLength if there is no pending frame
     */
    public int dispatch(IPacket isp) {
        return dispatch(Collections.singletonList(isp));
    }

    /**
     * This method calls dispatch(Arrays.asList(isp)).
     *
     * @param isp the packets to dispatch
     * @return the amount of free bytes that are still available in the last generated but not emitted frame, or a value equal to getMaxUserDataLength if there is no pending frame
     */
    public int dispatch(IPacket... isp) {
        return dispatch(Arrays.asList(isp));
    }

    /**
     * This method, when supported (AOS frames) dispatches a frame per invocation, containing the provided data as bitstream.
     *
     * @param bitstreamData the data to be put inside the frame
     * @return the data available in the frame (typically, as returned by getMaxUserDataLength)
     */
    public abstract int dispatch(BitstreamData bitstreamData);

    /**
     * This method is used to write the user data field of a frame directly, i.e. using the virtual channel in Virtual
     * Channel Access (VCA) mode.
     *
     * For TM-based VCs, the VC will wait to receive a sufficient amount of packets to emit its last frame, i.e. frames
     * are emitted once they are full. Depending on the amount and size of space packets, this method can result in zero, one or
     * multiple frames being emitted.
     *
     * For TC-based VCs, the VC will always emit at least one frame. The VC will try to pack the provided data inside a single
     * frame and will avoid segmentation if possible. For user data larger than the maximum frame size, the VC ensures
     * that the user data is segmented across frames. At the end of the process, all frames are anyway emitted, i.e. there
     * will be no remaining pending frame.
     *
     * @param userData the data to be put in the frame user data field
     * @return the amount of free bytes that are still available in the last generated but not emitted frame, or a value equal to getMaxUserDataLength if there is no pending frame
     */
    public abstract int dispatch(byte[] userData);

    /**
     * This method dispatches an idle frame filled with the provided idle pattern. The idle pattern is repeated until
     * the frame user data has been filled.
     *
     * @param idlePattern the idle pattern to be used
     */
    public abstract void dispatchIdle(byte[] idlePattern);

    /**
     * This method returns the maximum amount of user data that a transfer frame can have, when generated by this
     * virtual channel object.
     *
     * @return the maximum amount of user data per transfer frame (in bytes)
     */
    public abstract int getMaxUserDataLength();

    /**
     * This method returns true if there is a transfer frame pending filling.
     *
     * @return true if there is a partial transfer frame that was not emitted, false otherwise
     */
    public boolean isPendingFramePresent() {
        return this.currentFrame != null;
    }

    /**
     * This method returns the amount of user data space still available in the pending frame under construction. If
     * there is no pending frame, then the result is equal to the one returned by getMaxUserDataLength().
     *
     * @return free amount of user data space still available for the current frame (in bytes)
     */
    public int getRemainingFreeSpace() {
        if (this.currentFrame != null) {
            return this.currentFrame.getFreeUserDataLength();
        } else {
            return getMaxUserDataLength();
        }
    }

    /**
     * This method forces the reset of the frame currently under construction, i.e. it is deleted.
     */
    public void reset() {
        this.currentFrame = null;
    }

    /**
     * This method returns the number of frames emitted by this virtual channel after its creation. It can be used to
     * check how many frames have been emitted by a dispatch() invocation.
     *
     * @return the total number of emitted frames since the creation of this object
     */
    public long getNbOfEmittedFrames() {
        return this.emittedFrames.get();
    }
}
