/*
 * Copyright 2018-2019 Dario Lucia (https://www.dariolucia.eu)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package eu.dariolucia.ccsds.tmtc.algorithm;

import eu.dariolucia.ccsds.tmtc.util.StringUtil;
import org.junit.jupiter.api.Test;

import java.util.Arrays;

import static org.junit.jupiter.api.Assertions.*;

class ReedSolomonAlgorithmTest {

    private static final String testFrame = "07B11A15185F00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03A01250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03B01250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03C01250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03D0125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000";
    private static final String encodedTestFrame = "07B11A15185F00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03A01250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03B01250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03C01250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000C6AC03D0125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000FAD47CE809B8EBA844984DFD7D37F544A7F62C75D0FF97937FFE5A5CD424B92D2ACCFF0AEFAB0977918FCBA0DBDD54E657DC0098CE8B95E0B6398D1AAA177FA134E866B4645EC4C9AD9365954549C85D6BD363F000DA9CC26E48F067EEDD1A859967FA1120F27484FF3CA64012391C76AA7C38AA19F11C4CF54C9EA3AD5B83AB09895E6F9BBB416B329C15E3CB9F4019D637081154E182A8ACA395ED45A267CE";

    @Test
    public void testFrameEncodingDecoding() {
        {
            byte[] cadu = StringUtil.toByteArray(encodedTestFrame);
            byte[] frame = StringUtil.toByteArray(testFrame);
            byte[] encoded = ReedSolomonAlgorithm.TM_255_223.encodeFrame(frame, 5);
            assertArrayEquals(cadu, encoded);

            byte[] decoded = ReedSolomonAlgorithm.TM_255_223.decodeFrame(encoded, 5, true);
            assertArrayEquals(frame, decoded);
        }
        {
            byte[] cadu = StringUtil.toByteArray(encodedTestFrame);
            byte[] frame = StringUtil.toByteArray(testFrame);
            byte[] encoded = ReedSolomonAlgorithm.TM_255_223.encodeFrame(frame, 5);
            assertArrayEquals(cadu, encoded);

            encoded[1200] = 0x00;

            byte[] decoded = ReedSolomonAlgorithm.TM_255_223.decodeFrame(encoded, 5, true);
            assertNull(decoded);
        }
    }

    @Test
    public void testEncodingSpeed() {
        {
            byte[] frame = StringUtil.toByteArray(testFrame);
            long time = System.currentTimeMillis();
            int encodedTimes = 0;
            for(int i = 0; i < 10000; ++i) {
                byte[] encoded = ReedSolomonAlgorithm.TM_255_223.encodeFrame(frame, 5);
                // Artificial counting...
                if(encoded != null) {
                    ++encodedTimes;
                }
            }
            long timeEnd = System.currentTimeMillis();
            double rate = ((double)encodedTimes/(double)(timeEnd - time))*1000;
            System.out.println("RS encoded frames: " + encodedTimes + ", encoding time: " + (timeEnd - time) + " ms, encoding speed: " + rate + " frames/sec");
            // Guarantee at least 1 mbps: at 1279 bytes per CADU, this means ca 100 frames per second
            assertTrue(rate > 100);
        }
    }

}