/*
 *   Copyright (c) 2019 Dario Lucia (https://www.dariolucia.eu)
 *
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */

package eu.dariolucia.ccsds.tmtc.coding.encoder;

import eu.dariolucia.ccsds.tmtc.algorithm.ReedSolomonAlgorithm;
import eu.dariolucia.ccsds.tmtc.datalink.builder.TmTransferFrameBuilder;
import eu.dariolucia.ccsds.tmtc.datalink.pdu.TmTransferFrame;
import eu.dariolucia.ccsds.tmtc.util.StringUtil;
import org.junit.jupiter.api.Test;

import java.util.List;
import java.util.Optional;

import static org.junit.jupiter.api.Assertions.*;

public class TmEncoderTest {

    public static final String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

    public static final String EXPECTED_TM_2 = "034776C7272895B0F8FB18CEDDF270BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAC7FA407604D06B85E471649D6D3DBA3672D4BBEE619515F9F050878C44A66F558FF480EC09A0D70BC8E2C93ADA7B746CE5A977DCC32A2BF3E0A10F18894CDEAB1FE901D81341AE1791C59275B4F6E8D9CB52EFB9865457E7C1421E311299BD563FD203B026835C2F238B24EB69EDD1B396A5DF730CA8AFCF82843C6225337AAE249567843CA1921B540FCCF81769D6A4628A02A22B3949C3E58B92ADF0B5013D7BE755698B916EDDD8E83EA508ADC278C5767F7ECFC39F6D8CEEB6DE1A77E6E16D635240885C3A7FD520F714ADF606FCE18240CC2B0A9CE49F4D6817E0E203A0B2E1874A668FCEA2AB71C848D98D398B973CD59A4E3C63186E876B2AF30EBB5BE424F26154C4D3879B0D27B0CFD2202E3418A1A0295F8CDE9435F2638C2A469";


    @Test
    public void testTmEncoding() {
        // Create TM
        TmTransferFrameBuilder builder = TmTransferFrameBuilder.create(1115, 0, true, false)
                .setSpacecraftId(123)
                .setVirtualChannelId(1)
                .setMasterChannelFrameCount(22)
                .setVirtualChannelFrameCount(14)
                .setPacketOrderFlag(false)
                .setSynchronisationFlag(true)
                .setOcf(new byte[] { 0, 0, 0, 0 });
        builder.addData(new byte[TmTransferFrameBuilder.computeUserDataLength(1115, 0, true, false)]);

        TmTransferFrame tmtf = builder.build();

        // RS, randomize and ASM
        ReedSolomonEncoder<TmTransferFrame> rsEncoder = new ReedSolomonEncoder<>(ReedSolomonAlgorithm.TM_255_223, 5);
        TmRandomizerEncoder<TmTransferFrame> randomizerEncoder = new TmRandomizerEncoder<>();
        TmAsmEncoder<TmTransferFrame> asmEncoder = new TmAsmEncoder<>();
        Optional<byte[]> encoded = List.of(tmtf).stream().map(TmTransferFrame::getFrame).map(rsEncoder.asFunction()).map(randomizerEncoder.asFunction()).map(asmEncoder.asFunction()).findFirst();
        assertTrue(encoded.isPresent());
        assertEquals(1115 + 4 + 160, encoded.get().length);
        assertArrayEquals(StringUtil.toByteArray(EXPECTED_TM), encoded.get());
    }

    @Test
    public void testTmEncodingLongerSyncMarker() {
        // Create TM
        TmTransferFrameBuilder builder = TmTransferFrameBuilder.create(1115, 0, true, false)
                .setSpacecraftId(123)
                .setVirtualChannelId(1)
                .setMasterChannelFrameCount(22)
                .setVirtualChannelFrameCount(14)
                .setPacketOrderFlag(false)
                .setSynchronisationFlag(true)
                .setOcf(new byte[] { 0, 0, 0, 0 });
        builder.addData(new byte[TmTransferFrameBuilder.computeUserDataLength(1115, 0, true, false)]);

        TmTransferFrame tmtf = builder.build();

        // RS, randomize and ASM
        ReedSolomonEncoder<TmTransferFrame> rsEncoder = new ReedSolomonEncoder<>(ReedSolomonAlgorithm.TM_255_223, 5);
        TmRandomizerEncoder<TmTransferFrame> randomizerEncoder = new TmRandomizerEncoder<>();
        TmAsmEncoder<TmTransferFrame> asmEncoder = new TmAsmEncoder<>(new byte[] {0x03, 0x47, 0x76, (byte) 0xC7, 0x27, 0x28, (byte) 0x95, (byte) 0xB0});
        Optional<byte[]> encoded = List.of(tmtf).stream().map(TmTransferFrame::getFrame).map(rsEncoder.asFunction()).map(randomizerEncoder.asFunction()).map(asmEncoder.asFunction()).findFirst();
        assertTrue(encoded.isPresent());
        assertEquals(1115 + 8 + 160, encoded.get().length);
        assertArrayEquals(StringUtil.toByteArray(EXPECTED_TM_2), encoded.get());
    }

    @Test
    public void testNullInput() {
        try {
            ReedSolomonEncoder<TmTransferFrame> enc = new ReedSolomonEncoder<>(null, 4);
            fail("NullPointerException expected");
        } catch(NullPointerException e) {
            // Good
        }

        try {
            ReedSolomonEncoder<TmTransferFrame> enc = new ReedSolomonEncoder<>(ReedSolomonAlgorithm.TM_255_223, 4);
            enc.apply(null, null);
            fail("NullPointerException expected");
        } catch(NullPointerException e) {
            // Good
        }

        try {
            TmRandomizerEncoder<TmTransferFrame> enc = new TmRandomizerEncoder<>();
            enc.apply(null, null);
            fail("NullPointerException expected");
        } catch(NullPointerException e) {
            // Good
        }

        try {
            TmAsmEncoder<TmTransferFrame> enc = new TmAsmEncoder<>();
            enc.apply(null, null);
            fail("NullPointerException expected");
        } catch(NullPointerException e) {
            // Good
        }
    }
}
