/*
 * Copyright 2018-2019 Dario Lucia (https://www.dariolucia.eu)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package eu.dariolucia.ccsds.tmtc.datalink.pdu;

import eu.dariolucia.ccsds.tmtc.util.StringUtil;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.*;

class AosTransferFrameTest {

    // AOS Frame with FHEC, CLCW and FECF
    static final byte[] AOS_FRAME_01 = StringUtil.toByteArray( "5EEAFED1234AF2DA0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000AAAAAAAA1000");

    // M_PDU with FHEC, CLCW and FECF
    static final byte[] AOS_FRAME_02 = StringUtil.toByteArray( "5EEAFED1235AB22B00000814C000002D00031900000000000000010000A8000000000000000000000000000000000000000000000000000000000000807307FFC0000410000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004E9DAAAAAAAAD7FB");

    // B_PDU with FHEC, CLCW and FECF
    static final byte[] AOS_FRAME_03 = StringUtil.toByteArray( "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");

    @Test
    public void testAosFrameHeaderControlField() {
        byte[] message = new byte[] { 5, 14, 14, 10, 4, 10 };
        byte[] codeword = AosTransferFrame.AOS_FRAME_HEADER_ERROR_CONTROL_RS_UTIL.encodeCodeword(message);
        byte[] decodedMessage = AosTransferFrame.AOS_FRAME_HEADER_ERROR_CONTROL_RS_UTIL.decodeCodeword(codeword, false);
        assertArrayEquals(message, decodedMessage);
        decodedMessage = AosTransferFrame.AOS_FRAME_HEADER_ERROR_CONTROL_RS_UTIL.decodeCodeword(codeword, true);
        assertNotNull(decodedMessage);
        assertArrayEquals(message, decodedMessage);
    }

    @Test
    public void testAosTransferFrameDecoding() {
        AosTransferFrame ttf = new AosTransferFrame(AOS_FRAME_01, true, 0, AosTransferFrame.UserDataType.IDLE, true, true);
        assertEquals(123, ttf.getSpacecraftId());
        assertEquals(42, ttf.getVirtualChannelId());
        assertEquals(0xFED123, ttf.getVirtualChannelFrameCount());
        assertTrue(ttf.isValidHeader());
        assertTrue(ttf.isValid());


        ttf = new AosTransferFrame(AOS_FRAME_02, true, 0, AosTransferFrame.UserDataType.M_PDU, true, true);
        assertEquals(123, ttf.getSpacecraftId());
        assertEquals(42, ttf.getVirtualChannelId());
        assertEquals(0xFED123, ttf.getVirtualChannelFrameCount());
        assertFalse(ttf.isValidHeader());
        assertFalse(ttf.isValid());
        assertFalse(ttf.isIdleFrame());
        assertEquals(0, ttf.getFirstHeaderPointer());
        assertNotNull(ttf.getPacketZoneCopy());

        ttf = new AosTransferFrame(AOS_FRAME_03, true, 0, AosTransferFrame.UserDataType.B_PDU, true, true);
        assertEquals(123, ttf.getSpacecraftId());
        assertEquals(42, ttf.getVirtualChannelId());
        assertEquals(0xFED123, ttf.getVirtualChannelFrameCount());
        assertFalse(ttf.isValidHeader());
        assertFalse(ttf.isValid());
        assertFalse(ttf.isIdleFrame());
        assertNotEquals(0, ttf.getFhec());
        assertEquals(2048, ttf.getBitstreamDataPointer());
    }
}