using System.Reflection;
using Hippo.Application.Common.Interfaces;
using Hippo.Core.Entities;
using Hippo.Infrastructure.Data.Interceptors;
using Hippo.Infrastructure.Identity;
using MediatR;
using Microsoft.AspNetCore.Identity.EntityFrameworkCore;
using Microsoft.EntityFrameworkCore;

namespace Hippo.Infrastructure.Data;

public class ApplicationDbContext : IdentityDbContext<Account>, IApplicationDbContext
{
    private readonly IMediator _mediator;
    private readonly AuditableEntitySaveChangesInterceptor _auditableEntitySaveChangesInterceptor;

    public ApplicationDbContext(
            DbContextOptions<ApplicationDbContext> options,
            IMediator mediator,
            AuditableEntitySaveChangesInterceptor auditableEntitySaveChangesInterceptor) : base(options)
    {
        _mediator = mediator;
        _auditableEntitySaveChangesInterceptor = auditableEntitySaveChangesInterceptor;
    }

    public ApplicationDbContext(
            IMediator mediator,
            AuditableEntitySaveChangesInterceptor auditableEntitySaveChangesInterceptor)
    {
        _mediator = mediator;
        _auditableEntitySaveChangesInterceptor = auditableEntitySaveChangesInterceptor;
    }

    public DbSet<App> Apps => Set<App>();

    public DbSet<Certificate> Certificates => Set<Certificate>();

    public DbSet<Channel> Channels => Set<Channel>();

    public DbSet<EnvironmentVariable> EnvironmentVariables => Set<EnvironmentVariable>();

    public DbSet<Revision> Revisions => Set<Revision>();

    public DbSet<RevisionComponent> RevisionComponents => Set<RevisionComponent>();

    protected override void OnModelCreating(ModelBuilder builder)
    {
        builder.ApplyConfigurationsFromAssembly(Assembly.GetExecutingAssembly());

        base.OnModelCreating(builder);
    }

    protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
    {
        optionsBuilder.AddInterceptors(_auditableEntitySaveChangesInterceptor);
    }

    public override async Task<int> SaveChangesAsync(CancellationToken cancellationToken = new CancellationToken())
    {
        await _mediator.DispatchDeletedDomainEvents(this);

        var result = await base.SaveChangesAsync(cancellationToken);

        await _mediator.DispatchDomainEvents(this);

        return result;
    }
}
