import {text2Uint8Array, Uint32ToUint8Array, Uint16ToUint8Array} from '../util/util.es6';

/**
 *
 * @param {AudioContext} context
 * @param {Float32Array} audiobuffer
 *
 * @return {Promise}
 */
const Play = (context, audiobuffer) => {
  let abort;
  let promise = new Promise((resolve, reject) => {
    let source = context.createBufferSource();
    let soundBuffer = context.createBuffer(1, audiobuffer.length, 22050);
    let buffer = soundBuffer.getChannelData(0);
    for(let i=0; i<audiobuffer.length; i++) {
      buffer[i] = audiobuffer[i];
    }
    source.buffer = soundBuffer;
    source.connect(context.destination);
    source.onended = () => {
      resolve(true);
    };
    abort = reason => {
      source.disconnect();
      reject(reason);
    };
    source.start(0);
  });
  promise.abort = abort;
  return promise;
}

let context = null;

/**
 * Play an audio buffer.
 *
 * @param {Float32Array} audiobuffer
 *
 * @return {Promise}
 */
export const PlayBuffer = (audiobuffer) => {
  if (null === context) {
    context = new AudioContext();
  }

  if (!context) {
    if (process.env.NODE_ENV === 'development') {
      throw new Error('No player available!');
    }
    throw new Error();
  }

  return Play(context, audiobuffer);
}

/**
 * Convert a Uint8Array wave buffer to a Float32Array WaveBuffer
 *
 * @param {Uint8Array} buffer
 *
 * @return {Float32Array}
 */
export const Uint8ArrayToFloat32Array = (buffer) => {
  const audio = new Float32Array(buffer.length);
  for(let i=0; i < buffer.length; i++) {
    audio[i] = (buffer[i] - 128) / 256;
  }

  return audio
}

/**
 * Converts a Uint8Array buffer to a Uint8Array wave buffer
 *
 * @param {Uint8Array} audiobuffer
 *
 * @return {Uint8Array}
 */
export const ToWavBuffer = (audiobuffer) => {
  // Calculate buffer size.
  const realbuffer = new Uint8Array(
    4 + // "RIFF"
    4 + // uint32 filesize
    4 + // "WAVE"
    4 + // "fmt "
    4 + // uint32 fmt length
    2 + // uint16 fmt
    2 + // uint16 channels
    4 + // uint32 sample rate
    4 + // uint32 bytes per second
    2 + // uint16 block align
    2 + // uint16 bits per sample
    4 + // "data"
    4 + // uint32 chunk length
    audiobuffer.length
  );

  let pos=0;
  const write = (buffer) => {
    realbuffer.set(buffer, pos);
    pos+=buffer.length;
  };

  //RIFF header
  write(text2Uint8Array('RIFF')); // chunkID
  write(Uint32ToUint8Array(audiobuffer.length + 12 + 16 + 8 - 8)); // ChunkSize
  write(text2Uint8Array('WAVE')); // riffType
  //format chunk
  write(text2Uint8Array('fmt '));
  write(Uint32ToUint8Array(16)); // ChunkSize
  write(Uint16ToUint8Array(1)); // wFormatTag - 1 = PCM
  write(Uint16ToUint8Array(1)); // channels
  write(Uint32ToUint8Array(22050)); // samplerate
  write(Uint32ToUint8Array(22050)); // bytes/second
  write(Uint16ToUint8Array(1)); // blockalign
  write(Uint16ToUint8Array(8)); // bits per sample
  //data chunk
  write(text2Uint8Array('data'));
  write(Uint32ToUint8Array(audiobuffer.length)); // buffer length
  write(audiobuffer);

  return realbuffer;
};

/**
 *
 * @param {Uint8Array} audiobuffer
 *
 * @return void
 */
export const RenderBuffer = (audiobuffer) => {
  const filename = "sam.wav";

  const blob = new Blob([ToWavBuffer(audiobuffer)], { type: "audio/vnd.wave" });

  const url     = (window.URL || window.webkitURL);
  const fileURL = url.createObjectURL(blob);
  const a       = document.createElement('a');
  a.href        = fileURL;
  a.target      = '_blank';
  a.download    = filename;
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
  url.revokeObjectURL(fileURL);
}
