using System.Diagnostics.Tracing;

namespace DotNext.Net.Cluster.Consensus.Raft;

internal interface ILeaderStateMetrics
{
    void ReportBroadcastTime(TimeSpan value);
}

internal interface IFollowerStateMetrics
{
    void ReportHeartbeat();
}

/// <summary>
/// Contains a set of callbacks that can be used to report
/// runtime metrics generated by Raft cluster node.
/// </summary>
[Obsolete("Use System.Diagnostics.Metrics infrastructure instead.", UrlFormat = "https://learn.microsoft.com/en-us/dotnet/core/diagnostics/metrics")]
public class MetricsCollector : ILeaderStateMetrics, IFollowerStateMetrics
{
    /// <summary>
    /// Reports about broadcast time.
    /// </summary>
    /// <remarks>
    /// Broadcast time is the time spent accessing the cluster nodes caused by Leader states.
    /// </remarks>
    /// <param name="value">The broadcast time.</param>
    public virtual void ReportBroadcastTime(TimeSpan value)
    {
    }

    /// <summary>
    /// Sets counter that allows to track the broadcast time.
    /// </summary>
    public EventCounter? BroadcastTimeCounter
    {
        private get;
        init;
    }

    /// <inheritdoc />
    void ILeaderStateMetrics.ReportBroadcastTime(TimeSpan value)
    {
        BroadcastTimeCounter?.WriteMetric(value.TotalMilliseconds);
        ReportBroadcastTime(value);
    }

    /// <summary>
    /// Reports that node becomes a candidate.
    /// </summary>
    public virtual void MovedToCandidateState()
        => CandidateStateCounter?.Increment();

    /// <summary>
    /// Sets counter that allows to track the number of transitions to candidate state.
    /// </summary>
    public IncrementingEventCounter? CandidateStateCounter
    {
        init;
        private get;
    }

    /// <summary>
    /// Reports that node becomes a follower.
    /// </summary>
    public virtual void MovedToFollowerState()
        => FollowerStateCounter?.Increment();

    /// <summary>
    /// Sets counter that allows to track the number of transitions to follower state.
    /// </summary>
    public IncrementingEventCounter? FollowerStateCounter
    {
        private get;
        init;
    }

    /// <summary>
    /// Reports that node becomes a leader.
    /// </summary>
    public virtual void MovedToLeaderState()
        => LeaderStateCounter?.Increment();

    /// <summary>
    /// Sets counter that allows to track the number of transitions to leader state.
    /// </summary>
    public IncrementingEventCounter? LeaderStateCounter
    {
        private get;
        init;
    }

    /// <summary>
    /// Reports that node receives a heartbeat from leader node.
    /// </summary>
    public virtual void ReportHeartbeat()
    {
    }

    /// <summary>
    /// Sets counter that allows to track the number of received hearbeats from leader state.
    /// </summary>
    public IncrementingEventCounter? HeartbeatCounter
    {
        private get;
        init;
    }

    /// <inheritdoc />
    void IFollowerStateMetrics.ReportHeartbeat()
    {
        HeartbeatCounter?.Increment();
        ReportHeartbeat();
    }
}