package fmtc

import (
	"fmt"
	"os"
	"strconv"
	"strings"

	"github.com/faradey/madock/src/helper/cli/color"
	"golang.org/x/term"
)

// MaxVisibleItems is the maximum number of items to show at once
const MaxVisibleItems = 15

// InteractiveSelector displays an interactive selector with arrow key navigation
type InteractiveSelector struct {
	Title        string
	Options      []string
	Selected     int
	Recommended  int
	errorMsg     string
	numberBuffer string // Buffer for multi-digit number input
	scrollOffset int    // For scrolling in long lists
}

// NewInteractiveSelector creates a new interactive selector
func NewInteractiveSelector(title string, options []string, recommended int) *InteractiveSelector {
	// Filter out empty options and track the recommended index
	var filteredOptions []string
	newRecommended := -1
	for i, opt := range options {
		if opt != "" {
			if i == recommended {
				newRecommended = len(filteredOptions)
			}
			filteredOptions = append(filteredOptions, opt)
		}
	}

	selected := 0
	if newRecommended >= 0 {
		selected = newRecommended
	}

	return &InteractiveSelector{
		Title:       title,
		Options:     filteredOptions,
		Selected:    selected,
		Recommended: newRecommended,
	}
}

// Run displays the selector and returns the selected index and value
func (s *InteractiveSelector) Run() (int, string) {
	// Check if we're in a terminal
	if !term.IsTerminal(int(os.Stdin.Fd())) {
		// Fallback to non-interactive mode
		return s.Selected, s.Options[s.Selected]
	}

	// Save terminal state and set raw mode
	oldState, err := term.MakeRaw(int(os.Stdin.Fd()))
	if err != nil {
		// Fallback to non-interactive mode
		return s.Selected, s.Options[s.Selected]
	}
	defer term.Restore(int(os.Stdin.Fd()), oldState)

	s.render()

	// Read input
	buf := make([]byte, 3)
	for {
		n, err := os.Stdin.Read(buf)
		if err != nil {
			break
		}

		if n == 1 {
			switch buf[0] {
			case 13, 10: // Enter
				// If there's a number in buffer, try to select it
				if s.numberBuffer != "" {
					idx, err := strconv.Atoi(s.numberBuffer)
					if err == nil && idx >= 0 && idx < len(s.Options) {
						s.Selected = idx
						s.errorMsg = ""
						s.numberBuffer = ""
						s.clearAndPrintResult()
						return s.Selected, s.Options[s.Selected]
					}
					s.errorMsg = fmt.Sprintf("Invalid option '%s'. Enter 0-%d", s.numberBuffer, len(s.Options)-1)
					s.numberBuffer = ""
					s.render()
					continue
				}
				s.clearAndPrintResult()
				return s.Selected, s.Options[s.Selected]
			case 3: // Ctrl+C
				s.clearLines(s.calculateRenderHeight())
				term.Restore(int(os.Stdin.Fd()), oldState)
				os.Exit(0)
			case 127, 8: // Backspace (127 for Mac, 8 for others)
				if len(s.numberBuffer) > 0 {
					s.numberBuffer = s.numberBuffer[:len(s.numberBuffer)-1]
					s.errorMsg = ""
					s.render()
				}
			case 27: // Escape - clear number buffer
				if s.numberBuffer != "" {
					s.numberBuffer = ""
					s.errorMsg = ""
					s.render()
				}
			case 'j', 'J': // vim down
				s.numberBuffer = ""
				s.moveDown()
			case 'k', 'K': // vim up
				s.numberBuffer = ""
				s.moveUp()
			case '0', '1', '2', '3', '4', '5', '6', '7', '8', '9':
				s.numberBuffer += string(buf[0])
				s.errorMsg = ""

				// Preview: update selection if valid
				idx, err := strconv.Atoi(s.numberBuffer)
				if err == nil && idx >= 0 && idx < len(s.Options) {
					s.Selected = idx
					s.updateScrollOffset()
				}
				s.render()
			}
		} else if n == 3 && buf[0] == 27 && buf[1] == 91 {
			s.numberBuffer = "" // Clear buffer on arrow keys
			switch buf[2] {
			case 65: // Up arrow
				s.moveUp()
			case 66: // Down arrow
				s.moveDown()
			}
		}
	}

	return s.Selected, s.Options[s.Selected]
}

// calculateRenderHeight returns the number of lines used by render
func (s *InteractiveSelector) calculateRenderHeight() int {
	visibleCount := len(s.Options)
	if visibleCount > MaxVisibleItems {
		visibleCount = MaxVisibleItems
	}
	height := visibleCount + 4 // visible options + borders + help

	// Add scroll indicators if needed
	if len(s.Options) > MaxVisibleItems {
		if s.scrollOffset > 0 {
			height++ // "↑ X more above"
		}
		if s.scrollOffset+MaxVisibleItems < len(s.Options) {
			height++ // "↓ X more below"
		}
	}

	if s.errorMsg != "" {
		height++
	}
	if s.numberBuffer != "" {
		height++
	}
	return height
}

// updateScrollOffset adjusts scroll to keep selected item visible
func (s *InteractiveSelector) updateScrollOffset() {
	if len(s.Options) <= MaxVisibleItems {
		s.scrollOffset = 0
		return
	}

	// If selected is above visible area, scroll up
	if s.Selected < s.scrollOffset {
		s.scrollOffset = s.Selected
	}

	// If selected is below visible area, scroll down
	if s.Selected >= s.scrollOffset+MaxVisibleItems {
		s.scrollOffset = s.Selected - MaxVisibleItems + 1
	}
}

func (s *InteractiveSelector) moveUp() {
	if s.Selected > 0 {
		s.Selected--
		s.errorMsg = ""
		s.updateScrollOffset()
		s.render()
	}
}

func (s *InteractiveSelector) moveDown() {
	if s.Selected < len(s.Options)-1 {
		s.Selected++
		s.errorMsg = ""
		s.updateScrollOffset()
		s.render()
	}
}

func (s *InteractiveSelector) render() {
	// Move cursor up and clear previous render
	s.clearLines(s.calculateRenderHeight())

	// Calculate visible range
	visibleStart := s.scrollOffset
	visibleEnd := s.scrollOffset + MaxVisibleItems
	if visibleEnd > len(s.Options) {
		visibleEnd = len(s.Options)
	}

	// Calculate width based on visible options
	maxWidth := len(s.Title) + 4
	for i := visibleStart; i < visibleEnd; i++ {
		opt := s.Options[i]
		optLen := len(fmt.Sprintf("%d) %s", i, opt))
		if i == s.Recommended {
			optLen += 14
		}
		if optLen+8 > maxWidth {
			maxWidth = optLen + 8
		}
	}
	if maxWidth < 40 {
		maxWidth = 40
	}

	// Title with scroll info
	titleSuffix := ""
	if len(s.Options) > MaxVisibleItems {
		titleSuffix = fmt.Sprintf(" (%d-%d of %d)", visibleStart, visibleEnd-1, len(s.Options))
	}

	// Top border
	fmt.Printf("%s┌─ %s%s%s%s ", color.Cyan, color.Green, s.Title, color.Gray+titleSuffix, color.Cyan)
	titleLen := len(s.Title) + len(titleSuffix)
	borderLen := maxWidth - titleLen - 4
	if borderLen > 0 {
		fmt.Print(strings.Repeat("─", borderLen))
	}
	fmt.Printf("┐%s\r\n", color.Reset)

	// Scroll up indicator
	if visibleStart > 0 {
		fmt.Printf("%s│%s  %s↑ %d more above%s", color.Cyan, color.Reset, color.Gray, visibleStart, color.Reset)
		padding := maxWidth - 17
		if padding > 0 {
			fmt.Print(strings.Repeat(" ", padding))
		}
		fmt.Printf(" %s│%s\r\n", color.Cyan, color.Reset)
	}

	// Visible options only
	for i := visibleStart; i < visibleEnd; i++ {
		opt := s.Options[i]
		isSelected := i == s.Selected
		isRecommended := i == s.Recommended

		fmt.Printf("%s│%s ", color.Cyan, color.Reset)

		if isSelected {
			fmt.Printf("%s▸ %s", color.Green, color.Reset)
		} else {
			fmt.Print("  ")
		}

		if isSelected {
			fmt.Printf("%s%d) %s", color.Green, i, opt)
		} else {
			fmt.Printf("%s%d)%s %s", color.Cyan, i, color.Reset, opt)
		}

		if isRecommended {
			fmt.Printf(" %s(recommended)%s", color.Gray, color.Reset)
		}

		// Padding
		visibleLen := len(fmt.Sprintf("%d) %s", i, opt)) + 4
		if isRecommended {
			visibleLen += 14
		}
		padding := maxWidth - visibleLen
		if padding > 0 {
			fmt.Print(strings.Repeat(" ", padding))
		}

		fmt.Printf(" %s│%s\r\n", color.Cyan, color.Reset)
	}

	// Scroll down indicator
	if visibleEnd < len(s.Options) {
		remaining := len(s.Options) - visibleEnd
		fmt.Printf("%s│%s  %s↓ %d more below%s", color.Cyan, color.Reset, color.Gray, remaining, color.Reset)
		padding := maxWidth - 17
		if padding > 0 {
			fmt.Print(strings.Repeat(" ", padding))
		}
		fmt.Printf(" %s│%s\r\n", color.Cyan, color.Reset)
	}

	// Bottom border
	fmt.Printf("%s└%s┘%s\r\n", color.Cyan, strings.Repeat("─", maxWidth), color.Reset)

	// Help text
	fmt.Printf("%s↑/↓%s Navigate  %s•%s  %sEnter%s Select  %s•%s  %sNumber+Enter%s Jump to  %s•%s  %sCtrl+C%s Cancel\r\n",
		color.Cyan, color.Gray,
		color.Gray, color.Reset,
		color.Cyan, color.Gray,
		color.Gray, color.Reset,
		color.Cyan, color.Gray,
		color.Gray, color.Reset,
		color.Cyan, color.Reset,
	)

	// Show number buffer if typing
	if s.numberBuffer != "" {
		fmt.Printf("  %s→ Jump to: %s%s%s\r\n", color.Cyan, color.Green, s.numberBuffer, color.Reset)
	}

	// Error message if any
	if s.errorMsg != "" {
		fmt.Printf("  %s↳ %s%s\r\n", color.Red, s.errorMsg, color.Reset)
	}
}

func (s *InteractiveSelector) clearLines(n int) {
	for i := 0; i < n; i++ {
		fmt.Print("\033[2K") // Clear line
		if i < n-1 {
			fmt.Print("\033[A") // Move up
		}
	}
	fmt.Print("\r") // Return to start of line
}

func (s *InteractiveSelector) clearAndPrintResult() {
	s.clearLines(s.calculateRenderHeight())
	fmt.Printf("%s✓ %s: %s%s%s\r\n",
		color.Green,
		s.Title,
		color.Cyan,
		s.Options[s.Selected],
		color.Reset,
	)
}
