import { Slot } from '@radix-ui/react-slot';
import { type VariantProps, cva } from 'class-variance-authority';
import { LucideIcon, PlusIcon } from 'lucide-react';
import * as React from 'react';

import { cn } from '~/components/utils';

const buttonVariants = cva(
  "cursor-pointer disabled:cursor-not-allowed inline-flex items-center justify-center whitespace-nowrap rounded-md text-sm font-medium transition-all disabled:opacity-75 [&_svg]:pointer-events-none [&_svg:not([class*='size-'])]:size-4 shrink-0 [&_svg]:shrink-0 outline-none focus-visible:border-ring focus-visible:ring-ring/50 focus-visible:ring-[3px] aria-invalid:ring-destructive/20 dark:aria-invalid:ring-destructive/40 aria-invalid:border-destructive",
  {
    variants: {
      variant: {
        default:
          'bg-primary text-primary-foreground shadow-xs hover:bg-primary/90',
        secondary:
          'bg-secondary text-secondary-foreground shadow-xs hover:bg-secondary/80',
        destructive:
          'bg-destructive text-white shadow-xs hover:bg-destructive/80 focus-visible:ring-destructive/20 dark:focus-visible:ring-destructive/40',
        outline:
          'border bg-background shadow-xs hover:bg-accent hover:text-accent-foreground dark:bg-input/30 dark:border-input dark:hover:bg-input/50',
        ghost:
          'hover:bg-accent hover:text-accent-foreground dark:hover:bg-accent',
        link: 'text-primary underline-offset-4 hover:underline',
        //  customization
        secondaryline:
          'bg-secondary text-secondary-foreground shadow-xs hover:bg-secondary/80 border',
        subaction:
          'border border-violet-300 dark:border-violet-700 bg-transparent text-secondary-foreground shadow-xs hover:bg-secondary/80',
        primary:
          'border border-transparent bg-brand text-white shadow-sm enabled:hover:bg-brand/90',
        soft: 'border-violet-300 dark:border-violet-700 text-violet-600 dark:text-violet-400 enabled:hover:bg-violet-100 dark:enabled:hover:bg-violet-900/40'
      },
      size: {
        default: 'h-9 px-4 py-2 has-[>svg]:px-3',
        sm: 'h-8 rounded-md gap-1.5 px-3 has-[>svg]:px-2.5',
        lg: 'h-10 rounded-md px-6 has-[>svg]:px-4',
        icon: 'size-8'
      }
    },
    defaultVariants: {
      variant: 'outline',
      size: 'default'
    }
  }
);

function Button({
  className,
  variant,
  size,
  asChild = false,
  ...props
}: React.ComponentProps<'button'> &
  VariantProps<typeof buttonVariants> & {
    asChild?: boolean;
  }) {
  const Comp = asChild ? Slot : 'button';
  const kind = props.type || 'button';

  return (
    <Comp
      data-slot="button"
      type={kind}
      className={cn(buttonVariants({ variant, size, className }))}
      {...props}
    />
  );
}

type ButtonProps = React.ComponentProps<'button'> &
  VariantProps<typeof buttonVariants> & {
    asChild?: boolean;
  };

export { Button, buttonVariants };

export const ButtonWithPlus = (props: ButtonProps) => {
  return (
    <Button {...props}>
      <PlusIcon className="-ml-1.5 mr-1.5 h-4 w-4" aria-hidden="true" />
      {props.children}
    </Button>
  );
};

export const TextButton = (props: ButtonProps) => {
  return <Button {...props} variant="link" />;
};

export interface IconButtonProps extends ButtonProps {
  icon: LucideIcon;
}

export const IconButton = ({ icon, ...props }: IconButtonProps) => {
  const Icon = icon;
  return (
    <Button {...props} variant="ghost">
      <Icon className="h-4 w-4" aria-hidden="true" />
    </Button>
  );
};
