import { app, BrowserWindow } from 'electron';

import {
  registerDialogSubscribers,
  registerElectronStoreSubscribers,
  registerGrpcClientSubscribers,
  registerGrpcWebClientSubscribers,
  registerOSSubscribers,
  registerProtobufSubscribers,
  unregisterDialogSubscribers,
  unregisterGrpcClientSubscribers,
  unregisterGrpcWebClientSubscribers,
} from './ipc';

// This allows TypeScript to pick up the magic constant that's auto-generated by Forge's Webpack
// plugin that tells the Electron app where to look for the Webpack-bundled app code (depending on
// whether you're running in development or production).
declare const MAIN_WINDOW_WEBPACK_ENTRY: string;
declare const MAIN_WINDOW_PRELOAD_WEBPACK_ENTRY: string;

// Handle creating/removing shortcuts on Windows when installing/uninstalling.
// eslint-disable-next-line global-require
if (require('electron-squirrel-startup')) {
  // eslint-disable-line global-require
  app.quit();
}

registerOSSubscribers();
registerElectronStoreSubscribers();
registerProtobufSubscribers();

const createWindow = (): void => {
  // Create the browser window.
  const mainWindow = new BrowserWindow({
    height: 600,
    width: 1000,
    minHeight: 600,
    minWidth: 800,
    center: true,
    webPreferences: {
      preload: MAIN_WINDOW_PRELOAD_WEBPACK_ENTRY,
      sandbox: false,
    },
  });

  // and load the index.html of the app.
  mainWindow.loadURL(MAIN_WINDOW_WEBPACK_ENTRY);

  // Open the DevTools.
  // mainWindow.webContents.openDevTools();

  registerDialogSubscribers(mainWindow);
  registerGrpcClientSubscribers(mainWindow);
  registerGrpcWebClientSubscribers(mainWindow);

  mainWindow.on('close', () => {
    unregisterDialogSubscribers();
    unregisterGrpcClientSubscribers();
    unregisterGrpcWebClientSubscribers();
  });
};

// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.on('ready', createWindow);

// Quit when all windows are closed, except on macOS. There, it's common
// for applications and their menu bar to stay active until the user quits
// explicitly with Cmd + Q.
app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('activate', () => {
  // On OS X it's common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (BrowserWindow.getAllWindows().length === 0) {
    createWindow();
  }
});

// In this file you can include the rest of your app's specific main process
// code. You can also put them in separate files and import them here.
