"""j2cl_rta build rule.

Run the rta algorithm against a call graph generated by j2cl.
"""

load(
    "//build_defs/internal_do_not_use:rta_utils.bzl",
    "RTA_ASPECT_ATTRS",
    "get_aspect_providers",
    "get_j2cl_info_from_aspect_providers",
)
load(":provider.bzl", "J2clInfo")

_TransitiveLibraryInfo = provider(fields = ["files"])
_J2clRtaInfo = provider()

def _library_info_aspect_impl(target, ctx):
    j2cl_info = target[J2clInfo] if J2clInfo in target else get_j2cl_info_from_aspect_providers(target)

    library_info_file = j2cl_info._private_.library_info if j2cl_info else []

    transitive_library_infos = []
    for attr in RTA_ASPECT_ATTRS:
        if hasattr(ctx.rule.attr, attr):
            for target in getattr(ctx.rule.attr, attr):
                # The aspect is not applied on source files and they don't have any provider.
                if _TransitiveLibraryInfo in target:
                    transitive_library_infos.append(target[_TransitiveLibraryInfo].files)

    return [_TransitiveLibraryInfo(
        files = depset(library_info_file, transitive = transitive_library_infos),
    )]

_library_info_aspect = aspect(
    attr_aspects = RTA_ASPECT_ATTRS,
    required_aspect_providers = get_aspect_providers(),
    provides = [_TransitiveLibraryInfo],
    implementation = _library_info_aspect_impl,
)

def _j2cl_rta_impl(ctx):
    # our rule can assume that "_TransitiveLibraryInfo" provider is present in all of its
    # dependencies because the aspect have been applied on them.
    all_library_info_files = depset(
        transitive = [t[_TransitiveLibraryInfo].files for t in ctx.attr.targets],
    ).to_list()

    unused_types_list = ctx.outputs.unused_types_list
    removal_code_info_file = ctx.outputs.removal_code_info_file

    rta_args = ctx.actions.args()
    rta_args.use_param_file("@%s", use_always = True)
    rta_args.add("--unusedTypesOutput", unused_types_list)
    rta_args.add("--removalCodeInfoOutput", removal_code_info_file)
    if ctx.attr.legacy_keep_jstype_interfaces_do_not_use:
        rta_args.add("--legacy_keep_jstype_interfaces_do_not_use")
    rta_args.add_all(all_library_info_files)

    jvm_args = []
    if ctx.attr.generate_unused_methods_for_testing_do_not_use:
        jvm_args.append("--jvm_flag=-Dj2clrta.generate_unused_methods_for_testing=true")

    # Run rta algorithm
    ctx.actions.run(
        inputs = all_library_info_files,
        outputs = [unused_types_list, removal_code_info_file],
        arguments = jvm_args + [rta_args],
        progress_message = "Running J2CL rapid type analysis",
        executable = ctx.executable._rta_runner,
        execution_requirements = {"supports-workers": "1"},
        mnemonic = "J2clRta",
    )

    return [
        _J2clRtaInfo(
            unused_types_list = unused_types_list,
            removal_code_info_file = removal_code_info_file,
        ),
    ]

j2cl_rta = rule(
    attrs = {
        # TODO(b/114732596): Add a check on targets provided in "targets" field.
        "targets": attr.label_list(aspects = [_library_info_aspect]),
        "generate_unused_methods_for_testing_do_not_use": attr.bool(default = False),
        "legacy_keep_jstype_interfaces_do_not_use": attr.bool(default = False),
        "_rta_runner": attr.label(
            default = Label("//tools/java/com/google/j2cl/tools/rta:J2clRta_worker"),
            cfg = "exec",
            executable = True,
        ),
    },
    outputs = {
        "unused_types_list": "%{name}_unused_types.list",
        "removal_code_info_file": "%{name}_removal_code_info",
    },
    implementation = _j2cl_rta_impl,
)
