---
icon: pixel:play
---

# Getting Started

> Get started with H3.

<!-- automd:file src="../.partials/beta.md" -->

> [!IMPORTANT]
> You are currently reading H3 v2 docs. See [v1.h3.dev](https://v1.h3.dev/) for legacy docs.

<!-- /automd -->

## Overview

⚡ H3 (short for H(TTP), pronounced as /eɪtʃθriː/, like h-3) is a lightweight, fast, and composable server framework for modern JavaScript runtimes. It is based on web standard primitives such as [Request](https://developer.mozilla.org/en-US/docs/Web/API/Request), [Response](https://developer.mozilla.org/en-US/docs/Web/API/Response), [URL](https://developer.mozilla.org/en-US/docs/Web/API/URL), and [Headers](https://developer.mozilla.org/en-US/docs/Web/API/Headers). You can integrate H3 with any compatible runtime or [mount](/guide/api/h3#h3mount) other web-compatible handlers to H3 with almost no added latency.

H3 is designed to be extendable and composable. Instead of providing one big core, you start with a lightweight [H3 instance](/guide/api/h3) and then import built-in, tree-shakable [utilities](/utils) or bring your own for more functionality.
Composable utilities has several advantages:

- The server only includes used code and runs them exactly where is needed.
- Application size can scale better. Usage of utilities is explicit and clean, with less global impact.
- H3 is minimally opinionated and won't limit your choices.

All utilities, share an [H3Event](/guide/api/h3event) context.

:read-more{to="/utils" title="built-in H3 utilities"}

## Quick Start

> [!TIP]
> You try H3 online [on ⚡️ Stackblitz ](https://stackblitz.com/github/h3js/h3/tree/main/playground?file=server.mjs).

Install `h3` as a dependency:

:pm-install{name="h3"}

Create a new file for server entry:

```ts [server.mjs]
import { H3, serve } from "h3";

const app = new H3().get("/", (event) => "⚡️ Tadaa!");

serve(app, { port: 3000 });
```

Then, run the server using your favorite runtime:

::code-group

```bash [node]
node --watch ./server.mjs
```

```bash [deno]
deno run -A --watch ./server.mjs
```

```bash [bun]
bun run --watch server.mjs
```

::

And tadaa! We have a web server running locally.

### What Happened?

Okay, let's now break down our hello world example.

We first created an [H3](/guide/api/h3) app instance using `new H3()`:

```ts
const app = new H3();
```

[H3](/guide/api/h3) is a tiny class capable of [matching routes](/guide/basics/routing), [generating responses](/guide/basics/response) and calling [middleware](/guide/basics/middleware) and [global hooks](/guide/api/h3#global-hooks).

Then we add a route for handling HTTP GET requests to `/` path.

```ts
app.get("/", (event) => {
  return { message: "⚡️ Tadaa!" };
});
```

:read-more{title="Routing" to="/guide/basics/routing"}

We simply returned an object. H3 automatically [converts](/guide/basics/response#response-types) values into web responses.

:read-more{title="Sending Response" to="/guide/basics/response"}

Finally, we use `serve` method to start the server listener. Using `serve` method you can easily start an H3 server in various runtimes.

```js
serve(app, { port: 3000 });
```

> [!TIP]
> The `serve` method is powered by [💥 srvx](https://srvx.h3.dev/), a runtime-agnostic universal server listener based on web standards that works seamlessly with [Deno](https://deno.com/), [Node.js](https://nodejs.org/) and [Bun](https://bun.sh/).

We also have [`app.fetch`](/guide/api/h3#h3fetch) which can be directly used to run H3 apps in any web-compatible runtime or even directly called for testing purposes.

:read-more{to="/guide/api/h3#h3fetch" title="H3.fetch"}

```js
import { H3, serve } from "h3";

const app = new H3().get("/", () => "⚡️ Tadaa!");

// Test without listening
const response = await app.request("/");
console.log(await response.text());
```

You can directly import `h3` library from CDN alternatively. This method can be used for Bun, Deno and other runtimes such as Cloudflare Workers.

```js
import { H3 } from "https://esm.sh/h3";

const app = new H3().get("/", () => "⚡️ Tadaa!");

export const fetch = app.fetch;
```

<!-- ::read-more{to="https://nitro.build"}
For zero config setup deployment to different providers, we recommend using [Nitro](https://nitro.build).
:: -->
